/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "pbs_ifl.h"

static char ident[] = "@(#) $RCSfile: parse_at.c,v $ $Revision: 2.1 $";

#define ISNAMECHAR(x) ( (isgraph(x)) && ((x) != '#') && ( (x) != '@') )
#ifndef NULL
#define NULL 0
#endif

/*
 *
 * parse_at_item
 *
 * syntax:
 *
 *	at_name[@host]
 *
 */

int
parse_at_item(at_item, at_name_out, host_name_out)
char *at_item;
char **at_name_out;
char **host_name_out;
{
    int i;
    char *c;
    static char at_name[MAXPATHLEN+1];
    static int a_pos;
    static char host_name[PBS_MAXSERVERNAME];
    static int h_pos;

    /* initialize static data for this parsing call */
    for ( i=0; i<MAXPATHLEN+1; i++ ) at_name[i]='\0';
    a_pos= 0;
    for ( i=0; i<PBS_MAXSERVERNAME; i++ ) host_name[i]='\0';
    h_pos = 0;

    /* Begin the parse */
    c = at_item;
    while ( isspace(*c) ) c++;

    /* Looking for something before the @ sign */
    while ( *c != '\0' ) {
        if ( ISNAMECHAR(*c) ) {
            if ( a_pos >= MAXPATHLEN ) return 1;
            at_name[a_pos++]=*c;
        } else
            break;
        c++;
    }
    if ( a_pos == 0 ) return 1;

    /* Looking for a server */
    if ( *c == '@' ) {
        c++;
        while ( *c != '\0' ) {
            if ( ISNAMECHAR(*c) ) {
                if ( h_pos >= PBS_MAXSERVERNAME ) return 1;
                host_name[h_pos++]=*c;
            } else
                break;
            c++;
        }
        if ( h_pos == 0 ) return 1;
    }

    if ( *c != '\0' ) return 1;

    /* set char * pointers to static data, to arguments */
    if ( at_name_out != NULL ) *at_name_out = at_name;
    if ( host_name_out != NULL ) *host_name_out = host_name;

    return 0;
}



/*
 *
 * parse_at_list
 *
 * syntax:
 *
 *	at_name[@host] [,at_name[@host],...]
 *
 * Arguments:
 *
 *	list		List of at_name[@host] items.
 *	use_count	If true, make sure no host is repeated in the list,
 *			and host is defaulted only once.
 *	abs_path	If true, make sure host starts with a '/'.
 *
 */

struct hostlist {
    char host[PBS_MAXHOSTNAME];
    struct hostlist *next;
};

int
parse_at_list(list, use_count, abs_path)
char *list;
int use_count;
int abs_path;
{
    char *b, *c, *s, *l;
    int comma;
    char *user, *host;
    struct hostlist *ph, *nh, *hostlist = (struct hostlist *)0;

    if ( strlen(list) == 0 ) return (1);

    if ( (l = (char *)malloc(strlen(list)+1)) == (char *)0 ) {
	fprintf(stderr, "Out of memory.\n");
	exit(1);
    }
    strcpy(l, list);
    c = l;
    while ( *c != '\0' ) {
	/* Drop leading white space */
	while ( isspace((int)*c) ) c++;

	/* Find the next comma */
        s = c;
        while ( *c != ',' && *c ) c++;

	/* If requested, is this an absolute path */
        if ( abs_path && *s != '/' ) return 1;
	
	/* Drop any trailing blanks */
        comma = (*c == ',');
        *c = '\0';
	b = c - 1;
	while ( isspace((int)*b) ) *b-- = '\0';

	/* Parse the individual list item */
        if ( parse_at_item(s, &user, &host) ) {
            return 1;
        }

	/* The user part must be given */
        if ( strcmp(user, "") == 0 ) return 1;

	/* If requested, make sure the host name is not repeated */
        if ( use_count ) {
	    ph = hostlist;
	    while ( ph ) {
		if ( strcmp(ph->host, host) == 0 ) return 1;
		ph = ph->next;
	    }
	    nh = (struct hostlist *) malloc(sizeof(struct hostlist));
	    if ( !nh ) {
		fprintf(stderr, "Out of memory\n");
		exit(1);
	    }
	    nh->next = hostlist;
	    strcpy(nh->host, host);
	    hostlist = nh;
        }
        if ( comma ) {
            c++;
        }
    }

    /* Release memory for hostlist and argument list */
    ph = hostlist;
    while ( ph ) {
	nh = ph->next;
	free(ph);
	ph = nh;
    }
    free(l);

    /* Make sure the list does not end with a comma */
    if ( comma ) return 1;

    return 0;
}
