/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include "pbs_ifl.h"
#ifndef NULL
#define NULL 0
#endif

static char ident[] = "@(#) $RCSfile: parse_stage.c,v $ $Revision: 2.1 $";

#define ISNAMECHAR(x) ( (isgraph(x)) && ((x) != '@') && ((x) != ':') )

/*
 *
 * parse_stage_name
 *
 * syntax:
 *
 *	locat_file@hostname:remote_file
 *
 * Arguments:
 *
 *	pair	A staged file name pair.
 *	local	Local file name.
 *	host	Remote host.
 *	remote	Remote file name.
 *
 */

int
parse_stage_name(pair, local, host, remote)
char *pair;
char **local;
char **host;
char **remote;
{
    int i;
    char *c;
    static char local_name[MAXPATHLEN+1];
    static int l_pos;
    static char host_name[PBS_MAXSERVERNAME];
    static int h_pos;
    static char remote_name[MAXPATHLEN+1];
    static int r_pos;

    /* initialize static data for this parsing call */
    for ( i=0; i<MAXPATHLEN+1; i++ ) local_name[i]='\0';
    l_pos= 0;
    for ( i=0; i<PBS_MAXSERVERNAME; i++ ) host_name[i]='\0';
    h_pos = 0;
    for ( i=0; i<MAXPATHLEN+1; i++ ) remote_name[i]='\0';
    r_pos= 0;

    /* Begin the parse */
    c = pair;
    while ( isspace(*c) ) c++;

    /* Looking for something before the @ sign */
    while ( *c != '\0' ) {
        if ( ISNAMECHAR(*c) ) {
            if ( l_pos >= MAXPATHLEN ) return 1;
            local_name[l_pos++]=*c;
        } else
            break;
        c++;
    }
    if ( l_pos == 0 ) return 1;

    /* Looking for something between the @ and the : */
    if ( *c == '@' ) {
        c++;
        while ( *c != '\0' ) {
            if ( ISNAMECHAR(*c) ) {
                if ( h_pos >= PBS_MAXSERVERNAME ) return 1;
                host_name[h_pos++]=*c;
            } else
                break;
            c++;
        }
    }
    if ( h_pos == 0 ) return 1;

    /* Looking for something after the : */
    if ( *c == ':' ) {
        c++;
        while ( *c != '\0' ) {
            if ( ISNAMECHAR(*c) ) {
                if ( r_pos >= MAXPATHLEN ) return 1;
                remote_name[r_pos++]=*c;
            } else
                break;
            c++;
        }
    }
    if ( r_pos == 0 ) return 1;

    if ( *c != '\0' ) return 1;

    /* set char * pointers to static data, to arguments */
    if ( local  != NULL ) *local = local_name;
    if ( host   != NULL ) *host = host_name;
    if ( remote != NULL ) *remote = remote_name;

    return (0);
}



/*
 *
 * parse_stage_list
 *
 * syntax:
 *
 *	local_file@hostname:remote_file [,...]
 *
 * Arguments:
 *
 *	list	List of staged file name pairs.
 *
 */

int
parse_stage_list(list)
char *list;
{
    char *b, *c, *s, *l;
    int comma;
    char *local, *host, *remote;

    if ( strlen(list) == 0 ) return (1);

    if ( (l = (char *)malloc(strlen(list)+1)) == (char *)0 ) {
        fprintf(stderr, "Out of memory.\n");
        exit(1);
    }
    strcpy(l, list);
    c = l;
    while ( *c != '\0' ) {
	/* Drop leading white space */
	while ( isspace((int)*c) ) c++;

	/* Find the next comma */
        s = c;
        while ( *c != '\0' && *c != ',' ) c++;

	/* Drop any trailing blanks */
        comma = (*c == ',');
        *c = '\0';
	b = c - 1;
	while ( isspace((int)*b) ) *b-- = '\0';

	/* Parse the individual list item */
        if ( parse_stage_name(s, &local, &host, &remote) ) {
	    (void)free(l);
            return 1;
        }

	/* Make sure all parts of the item are present */
        if ( strlen(local)  == 0 ) {
		(void)free(l);
		return 1;
	}
        if ( strlen(host)   == 0 ) {
		(void)free(l);
		return 1;
	}
        if ( strlen(remote) == 0 ) {
		(void)free(l);
		return 1;
	}

        if ( comma ) {
            c++;
        }
    }
    if ( comma ) {
		(void)free(l);
		return 1;
	}

    (void)free(l);

    return 0;
}
