/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * Synopsis:
 * 	long double disrl(int stream, int *retval)
 *
 *	Gets a Data-is-Strings floating point number from <stream> and converts
 *	it into a long double and returns it.  The number from <stream> consists
 *	of two consecutive signed integers.  The first is the coefficient, with
 *	its implied decimal point at the low-order end.  The second is the
 *	exponent as a power of 10.
 *
 *	*<retval> gets DIS_SUCCESS if everything works well.  It gets an error
 *	code otherwise.  In case of an error, the <stream> character pointer is
 *	reset, making it possible to retry with some other conversion strategy.
 *
 *	By fiat of the author, neither loss of significance nor underflow are
 *	errors.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <math.h>
#include <stddef.h>
#include <stdio.h>

#include "dis.h"
#include "dis_.h"

dis_long_double_t
disrl(stream, retval)
    int			stream;
    int			*retval;
    {
	int		expon;
	unsigned	uexpon;
	int		locret;
	int		negate;
	unsigned	ndigs;
	unsigned	nskips;
	dis_long_double_t 	ldval;

	assert(retval != NULL);
	assert(disr_commit != NULL);

	ldval = 0.0L;
	locret = disrl_(stream, &ldval, &ndigs, &nskips, LDBL_DIG, 1);
	if (locret == DIS_SUCCESS) {
		locret = disrsi_(stream, &negate, &uexpon, 1);
		if (locret == DIS_SUCCESS) {
			expon = negate ? nskips - uexpon : nskips + uexpon;
			if (expon + (int)ndigs > LDBL_MAX_10_EXP) {
				if (expon + (int)ndigs > LDBL_MAX_10_EXP + 1) {
					ldval = ldval < 0.0L ?
					    -HUGE_VAL : HUGE_VAL;
					locret = DIS_OVERFLOW;
				} else {
				        ldval *= disp10l_(expon - 1);
					if (ldval > LDBL_MAX / 10.0L) {
						ldval = ldval < 0.0L ?
						    -HUGE_VAL : HUGE_VAL;
						locret = DIS_OVERFLOW;
					} else
					        ldval *= 10.0L;
				}
			} else {
				if (expon < LDBL_MIN_10_EXP) {
					ldval *= disp10l_(expon + (int)ndigs);
					ldval /= disp10l_((int)ndigs);
				} else
				        ldval *= disp10l_(expon);
			}
		}
	}
	if ((*disr_commit)(stream, locret == DIS_SUCCESS) < 0)
	        locret = DIS_NOCOMMIT;
	*retval = locret;
	return (ldval);
}
