/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*	PBS_rdrpy

	Read the reply to a batch request.
	A reply structure is allocated and cleared.
	The reply is read and decoded into the structure.
	The reply structure is returned.

	The caller MUST free the reply structure by calling
	PBS_FreeReply().
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>
#include "libpbs.h"
#include "dis.h"

static char ident[] = "@(#) $RCSfile: PBSD_rdrpy.c,v $ $Revision: 2.1 $";

struct batch_reply *PBSD_rdrpy(c)
	int c;
{
	int		    rc;
	struct batch_reply *reply;
	int		    sock;

	/* clear any prior error message */

	if( connection[c].ch_errtxt != (char *)NULL ) {
		free( connection[c].ch_errtxt );
		connection[c].ch_errtxt = (char *)NULL;
	}

	if ((reply = (struct batch_reply *)malloc(sizeof (struct batch_reply))) == 0) {
		connection[c].ch_errno = PBSE_SYSTEM;
		pbs_errno = PBSE_SYSTEM;
		return ((struct batch_reply *)0);
	}
	(void)memset(reply, 0, sizeof (struct batch_reply));

	sock = connection[c].ch_socket;
	DIS_tcp_setup(sock);

	if (rc = decode_DIS_replyCmd(sock, reply))   {
	    (void)free(reply);
	    pbs_errno = PBSE_PROTOCOL;
	    connection[c].ch_errno = PBSE_PROTOCOL;
	    connection[c].ch_errtxt = strdup(dis_emsg[rc]);
	    return (struct batch_reply *)NULL;
	}
	DIS_tcp_reset(sock, 0);		/* reset DIS read buffer */

	connection[c].ch_errno = reply->brp_code;
	pbs_errno = reply->brp_code;
	if (reply->brp_choice == BATCH_REPLY_CHOICE_Text) {
		connection[c].ch_errtxt = strdup(reply->brp_un.brp_txt.brp_str);
	}
	return reply;
}


/*
 * PBS_FreeReply - Free a batch_reply structure allocated in PBS_rdrpy() 
 *
 *	Any additional allocated substructures pointed to from the
 *	reply structure are freed, then the base struture itself is gone.
 */

void PBSD_FreeReply(reply)
	struct batch_reply *reply;
{
	struct brp_select   *psel;
	struct brp_select   *pselx;
	struct brp_cmdstat  *pstc;
	struct brp_cmdstat  *pstcx;
	struct attrl        *pattrl;
	struct attrl	    *pattrx;

	if (reply == 0)
		return;
	if (reply->brp_choice == BATCH_REPLY_CHOICE_Text) {
		if (reply->brp_un.brp_txt.brp_str) {
			(void)free(reply->brp_un.brp_txt.brp_str);
			reply->brp_un.brp_txt.brp_str = (char *)0;
			reply->brp_un.brp_txt.brp_txtlen = 0;
		}

	} else if (reply->brp_choice == BATCH_REPLY_CHOICE_Select) {
		psel = reply->brp_un.brp_select;
		while (psel) {
		    pselx = psel->brp_next;
		    (void)free(psel);
		    psel = pselx;
		}
		
	} else if (reply->brp_choice == BATCH_REPLY_CHOICE_Status) {
		pstc = reply->brp_un.brp_statc;
		while (pstc) {
			pstcx = pstc->brp_stlink;
			pattrl = pstc->brp_attrl;
			while (pattrl) {
				pattrx = pattrl->next;
				if (pattrl->name)
					(void)free(pattrl->name);
				if (pattrl->resource)
					(void)free(pattrl->resource);
				if (pattrl->value)
					(void)free(pattrl->value);
				(void)free(pattrl);
				pattrl = pattrx;
			}
			(void)free(pstc);
			pstc = pstcx;
		}
	} else if (reply->brp_choice == BATCH_REPLY_CHOICE_RescQuery) {
		(void)free(reply->brp_un.brp_rescq.brq_avail);
		(void)free(reply->brp_un.brp_rescq.brq_alloc);
		(void)free(reply->brp_un.brp_rescq.brq_resvd);
		(void)free(reply->brp_un.brp_rescq.brq_down);
	}

	(void)free(reply);
}
