/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <string.h>
#include <unistd.h>
#include <stdio.h>
#include <fcntl.h>
#include "portability.h"
#include "libpbs.h"
#include "dis.h"

/*	PBS_rdytocmt.c

	This function does the Ready To Commit sub-function of
	the Queue Job request.
*/

static char ident[] = "@(#) $RCSfile: PBSD_submit.c,v $ $Revision: 2.1 $";

int PBSD_rdytocmt(connect, jobid)
	int connect;
	char * jobid;
{
	int	rc;
	struct batch_reply *reply;
	int	sock;

	sock = connection[connect].ch_socket;
	DIS_tcp_setup(sock);

	if ( (rc=encode_DIS_ReqHdr(sock, PBS_BATCH_RdytoCommit,pbs_current_user)) ||
	     (rc = encode_DIS_JobId(sock, jobid))  ||
	     (rc = encode_DIS_ReqExtend(sock, (char *)0)) ) {
		connection[connect].ch_errtxt = strdup(dis_emsg[rc]);
		return (pbs_errno = PBSE_PROTOCOL);
	}
	if (DIS_tcp_wflush(sock)) {
		return (pbs_errno = PBSE_PROTOCOL);
	}
	
	/* read reply */

	reply = PBSD_rdrpy(connect);

	PBSD_FreeReply(reply);

	return connection[connect].ch_errno;
}

/*	PBS_commit.c

	This function does the Commit sub-function of
	the Queue Job request.
*/

int PBSD_commit(connect, jobid)
	int connect;
	char * jobid;
{
	struct batch_reply *reply;
	int rc;
	int sock;
	
	sock = connection[connect].ch_socket;
	DIS_tcp_setup(sock);

	if ( (rc = encode_DIS_ReqHdr(sock, PBS_BATCH_Commit, pbs_current_user)) ||
	     (rc = encode_DIS_JobId(sock, jobid)) ||
	     (rc = encode_DIS_ReqExtend(sock, (char *)0)) ) {
		connection[connect].ch_errtxt = strdup(dis_emsg[rc]);
		return (pbs_errno = PBSE_PROTOCOL);
	}
	
	if (DIS_tcp_wflush(sock)) {
		return (pbs_errno = PBSE_PROTOCOL);
	}

	reply = PBSD_rdrpy(connect);

	PBSD_FreeReply(reply);

	return connection[connect].ch_errno;
}

/*	PBS_scbuf.c
 *
 *	Send a chunk of a of the job script to the server.
 *	Called by pbs_submit.  The buffer length could be
 *	zero; the server should handle that case...
*/

static int PBSD_scbuf(c, reqtype, seq, buf, len, jobid, which)
	int   c;		/* connection handle     */
	int   reqtype;		/* request type */
	int   seq;		/* file chunk sequence number */
	char *buf;		/* file chunk */
	int   len;		/* length of chunk */
	char *jobid;		/* job id (for types 1 and 2 only) */
	enum job_file which;	/* standard file type, see libpbs.h */
{
	struct batch_reply   *reply;
	int	rc;
	int	sock;

	sock = connection[c].ch_socket;
	DIS_tcp_setup(sock);

	if (jobid == (char *)0)
		jobid = "";	/* use null string for null pointer */

	if ( (rc = encode_DIS_ReqHdr(sock, reqtype, pbs_current_user)) ||
	     (rc = encode_DIS_JobFile(sock, seq, buf, len, jobid, which)) ||
	     (rc = encode_DIS_ReqExtend(sock, (char *)0)) ) {
		connection[c].ch_errtxt = strdup(dis_emsg[rc]);
		return (pbs_errno = PBSE_PROTOCOL);
	}
	if (DIS_tcp_wflush(sock)) {
		return (pbs_errno = PBSE_PROTOCOL);
	}
				
	/* read reply */

	reply = PBSD_rdrpy(c);
	
	PBSD_FreeReply(reply);

	return connection[c].ch_errno;
}

/*	PBS_jscript.c
 *
 *	The Job Script subfunction of the Queue Job request
 *	-- the function PBS_scbuf is called repeatedly to
 *	transfer chunks of the script to the server.
*/

int PBSD_jscript(c, script_file)
	int c;
	char *script_file;
{
	int i;
	int fd;
	int cc;
	char s_buf[SCRIPT_CHUNK_Z];
	
	if( (fd = open(script_file, O_RDONLY, 0)) < 0 ) {
		return (-1);
	}
	i = 0;
	cc = read(fd, s_buf, SCRIPT_CHUNK_Z);
	while ((cc > 0) && (PBSD_scbuf(c,PBS_BATCH_jobscript,i,s_buf,cc,(char *)0,JScript) == 0)) {
		i++;
		cc = read(fd, s_buf, SCRIPT_CHUNK_Z);
	}

	close(fd);
	if (cc < 0)	/* read failed */
		return (-1);
	return connection[c].ch_errno;
}

/*	PBS_jobfile.c
 *
 *	The Job File function used to move files related to
 *	a job between servers.
 *	-- the function PBS_scbuf is called repeatedly to
 *	transfer chunks of the script to the server.
*/

int PBSD_jobfile(c, req_type, path, jobid, which)
	int   c;
	int   req_type;
	char *path;
	char *jobid;
	enum job_file which;
{
	int   i;
	int   cc;
	int   fd;
	char  s_buf[SCRIPT_CHUNK_Z];
	
	if( (fd = open(path, O_RDONLY, 0)) < 0 ) {
		return (-1);
	}
	i = 0;
	cc = read(fd, s_buf, SCRIPT_CHUNK_Z);
	while ((cc > 0) && 
	       (PBSD_scbuf(c, req_type, i, s_buf, cc, jobid, which ) == 0) ) {
		i++;
		cc = read(fd, s_buf, SCRIPT_CHUNK_Z);
	}

	close(fd);
	if (cc < 0)	/* read failed */
		return (-1);
	return connection[c].ch_errno;
}

/*	PBS_queuejob.c

	This function sends the first part of the Queue Job request
*/

char * PBSD_queuejob(connect, jobid, destin, attrib, extend)
	int connect;
	char * jobid;
	char * destin;
	struct attropl * attrib;
	char * extend;
{
	struct batch_reply *reply;
	char  *return_jobid = (char *)NULL;
	int    rc;
	int    sock;
	
	sock = connection[connect].ch_socket;
	DIS_tcp_setup(sock);

	/* first, set up the body of the Queue Job request */
	
	if ( (rc = encode_DIS_ReqHdr(sock, PBS_BATCH_QueueJob, pbs_current_user)) ||
	     (rc = encode_DIS_QueueJob(sock, jobid, destin, attrib)) ||
	     (rc = encode_DIS_ReqExtend(sock, extend)) ) {
		connection[connect].ch_errtxt = strdup(dis_emsg[rc]);
		pbs_errno = PBSE_PROTOCOL;
		return return_jobid;
	}
	if (DIS_tcp_wflush(sock)) {
		pbs_errno = PBSE_PROTOCOL;
		return return_jobid;
	}

	/* read reply from stream into presentation element */

	reply = PBSD_rdrpy(connect);
	if ( reply == NULL ) {
                pbs_errno = PBSE_PROTOCOL;
	} else if ( reply->brp_choice &&
	            reply->brp_choice != BATCH_REPLY_CHOICE_Text &&
	            reply->brp_choice != BATCH_REPLY_CHOICE_Queue) {
		pbs_errno = PBSE_PROTOCOL;
	} else if( connection[connect].ch_errno == 0 )
		return_jobid = strdup(reply->brp_un.brp_jid);

	PBSD_FreeReply(reply);
	return return_jobid;
}
