/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
 
static char ident[] = "@(#) $RCSfile: dec_rpys.c,v $ $Revision: 2.1 $";

/*
 * decode_DIS_replySvr() - decode a Batch Protocol Reply Structure for Server
 * 
 *	This routine decodes a batch reply into the form used by server.
 *	The only difference between this and the command version is on status
 *	replies.  For the server,  the attributes are decoded into a list of
 *	server svrattrl structures rather than a commands's attrl.
 *
 * 	batch_reply structure defined in libpbs.h, it must be allocated
 *	by the caller.
 */
 
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <stdlib.h>
#include "libpbs.h"
#include "list_link.h"
#include "dis.h"

int decode_DIS_replySvr(sock, reply)
	int		    sock;
	struct batch_reply *reply;
{
	int		      ct;
	int		      i;
	struct brp_select    *psel;
	struct brp_select   **pselx;
	struct brp_status    *pstsvr;
	int		      rc = 0;

	/* first decode "header" consisting of protocol type and version */

	i = disrui(sock, &rc);
	if (rc != 0) return rc;
	if (i != PBS_BATCH_PROT_TYPE) return DIS_PROTO;
	i = disrui(sock, &rc);
	if (rc != 0) return rc;
	if (i != PBS_BATCH_PROT_VER) return DIS_PROTO;

	/* next decode code, auxcode and choice (union type identifier) */

	reply->brp_code    = disrsi(sock, &rc);
	if (rc) return rc;
	reply->brp_auxcode = disrsi(sock, &rc);
	if (rc) return rc;
	reply->brp_choice  = disrui(sock, &rc);
	if (rc) return rc;


	switch (reply->brp_choice) {

	    case BATCH_REPLY_CHOICE_NULL:
		break;	/* no more to do */

	    case BATCH_REPLY_CHOICE_Queue:
	    case BATCH_REPLY_CHOICE_RdytoCom:
	    case BATCH_REPLY_CHOICE_Commit:
		if (rc = disrfst(sock, PBS_MAXSVRJOBID+1,reply->brp_un.brp_jid))
			return (rc);
		break;

	    case BATCH_REPLY_CHOICE_Select:

		/* have to get count of number of strings first */

		reply->brp_un.brp_select = (struct brp_select *)0;
		pselx = &reply->brp_un.brp_select;
		ct = disrui(sock, &rc);
		if (rc) return rc;

		while (ct--) {
			psel = (struct brp_select *)malloc(sizeof (struct brp_select));
			if (psel == 0) return DIS_NOMALLOC;
			psel->brp_next = (struct brp_select *)0;
			psel->brp_jobid[0] = '\0';
			rc = disrfst(sock, PBS_MAXSVRJOBID+1, psel->brp_jobid);
			if (rc) {
				(void)free(psel);
				return rc;
			}
			*pselx = psel;
			pselx = &psel->brp_next;
		}
		break;

	    case BATCH_REPLY_CHOICE_Status:

		/* have to get count of number of status objects first */

		CLEAR_HEAD(reply->brp_un.brp_status);
		ct = disrui(sock, &rc);
		if (rc) return rc;

		while (ct--) {
			pstsvr = (struct brp_status *)malloc(sizeof (struct brp_status));
			if (pstsvr == 0) return DIS_NOMALLOC;

			CLEAR_LINK(pstsvr->brp_stlink);
			pstsvr->brp_objname[0] = '\0';
			CLEAR_HEAD(pstsvr->brp_attr);

			pstsvr->brp_objtype = disrui(sock, &rc);
			if (rc == 0) {
				rc = disrfst(sock, PBS_MAXSVRJOBID+1,
					     pstsvr->brp_objname);
			}
			if (rc) {
				(void)free(pstsvr);
				return rc;
			}
			append_link(&reply->brp_un.brp_status, 
				    &pstsvr->brp_stlink, pstsvr);
			rc = decode_DIS_svrattrl(sock, &pstsvr->brp_attr);
		}
		break;
		
	    case BATCH_REPLY_CHOICE_Text:
		
		/* text reply */

		reply->brp_un.brp_txt.brp_str = disrcs(sock,
				(size_t *)&reply->brp_un.brp_txt.brp_txtlen,
				&rc);
		break;

	    case BATCH_REPLY_CHOICE_Locate:

		/* Locate Job Reply */

		rc = disrfst(sock, PBS_MAXDEST+1, reply->brp_un.brp_locate);
		break;

	    default:
		return -1;
	}

	return rc;
}
