/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
 
static char ident[] = "@(#) $RCSfile: enc_reply.c,v $ $Revision: 2.1 $";

/*
 * encode_DIS_reply() - encode a Batch Protocol Reply Structure
 * 
 * 	batch_reply structure defined in libpbs.h, it must be allocated
 *	by the caller.
 */
 
#include <pbs_config.h>   /* the master config generated by configure */

#include "libpbs.h"
#include "list_link.h"
#include "attribute.h"
#include "dis.h"

int encode_DIS_reply(sock, reply)
	int		    sock;
	struct batch_reply *reply;
{
	int		    ct;
	int		    i;
	struct brp_select  *psel;
	struct brp_status  *pstat;
	svrattrl	   *psvrl;

	int rc;

	/* first encode "header" consisting of protocol type and version */

	if ( (rc = diswui(sock, PBS_BATCH_PROT_TYPE))   ||
	     (rc = diswui(sock, PBS_BATCH_PROT_VER)) )
		return rc;

	/* next encode code, auxcode and choice (union type identifier) */

	if ( (rc = diswsi(sock, reply->brp_code)) 	||
	     (rc = diswsi(sock, reply->brp_auxcode))	||
	     (rc = diswui(sock, reply->brp_choice)))
		return rc;

	switch (reply->brp_choice) {

	    case BATCH_REPLY_CHOICE_NULL:
		break;	/* no more to do */

	    case BATCH_REPLY_CHOICE_Queue:
	    case BATCH_REPLY_CHOICE_RdytoCom:
	    case BATCH_REPLY_CHOICE_Commit:
		if (rc = diswst(sock, reply->brp_un.brp_jid))
			return (rc);
		break;

	    case BATCH_REPLY_CHOICE_Select:

		/* have to send count of number of strings first */

		ct = 0;
		psel = reply->brp_un.brp_select;
		while (psel) {
			++ct;
			psel = psel->brp_next;
		}	
		if (rc = diswui(sock, ct))
			return rc;

		psel = reply->brp_un.brp_select;
		while (psel) {
			/* now encode each string */
			if (rc = diswst(sock, psel->brp_jobid))
				return rc;
			psel = psel->brp_next;
		}
		break;

	    case BATCH_REPLY_CHOICE_Status:

		/* encode "server version" of status structure.
		 *
		 * Server always uses svrattrl form.
		 * Commands decode into their form.
		 *
		 * First need to encode number of status objects and then
		 * the object itself.
		 */

		ct = 0;
		pstat = (struct brp_status *)GET_NEXT(reply->brp_un.brp_status);
		while (pstat) {
			++ct;
			pstat =(struct brp_status *)GET_NEXT(pstat->brp_stlink);
		}
		if (rc = diswui(sock, ct))
			return rc;
		pstat = (struct brp_status *)GET_NEXT(reply->brp_un.brp_status);
		while (pstat) {
			if ( (rc = diswui(sock, pstat->brp_objtype))	||
			     (rc = diswst(sock, pstat->brp_objname)) )
				return rc;

			psvrl = (svrattrl *)GET_NEXT(pstat->brp_attr);
			if (rc = encode_DIS_svrattrl(sock, psvrl))
				return rc;
			pstat =(struct brp_status *)GET_NEXT(pstat->brp_stlink);
		}
		break;

	    case BATCH_REPLY_CHOICE_Text:
		
		/* text reply */

		if (rc = diswcs(sock, reply->brp_un.brp_txt.brp_str,
				      (size_t)reply->brp_un.brp_txt.brp_txtlen))	
			return rc;
		break;

	    case BATCH_REPLY_CHOICE_Locate:

		/* Locate Job Reply */

		if (rc = diswst(sock, reply->brp_un.brp_locate))
			return rc;
		break;

	    case BATCH_REPLY_CHOICE_RescQuery:

		/* Query Resources Reply */

		ct = reply->brp_un.brp_rescq.brq_number;
		if (rc = diswui(sock, ct))
			return rc;
		for (i=0; (i<ct) && (rc == 0); ++i) {
			rc =diswui(sock,*(reply->brp_un.brp_rescq.brq_avail+i));
		}
		if (rc) return rc;
		for (i=0; (i<ct) && (rc == 0); ++i) {
			rc =diswui(sock,*(reply->brp_un.brp_rescq.brq_alloc+i));
		}
		if (rc) return rc;
		for (i=0; (i<ct) && (rc == 0); ++i) {
			rc =diswui(sock,*(reply->brp_un.brp_rescq.brq_resvd+i));
		}
		if (rc) return rc;
		for (i=0; (i<ct) && (rc == 0); ++i) {
			rc =diswui(sock,*(reply->brp_un.brp_rescq.brq_down+i));
		}
		if (rc) return rc;
		break;

	    default:
		return -1;
	}

	return 0;
}
