/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*	pbs_connect.c
 *
 *	Open a connection with the pbs server.  At this point several
 *	things are stubbed out, and other things are hard-wired.
 *
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdlib.h>
#include <errno.h>
#include <stdio.h>
#include <pwd.h>
#include <string.h>
#include <netdb.h>
#include <unistd.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include "libpbs.h"
#include "dis.h"

static char ident[] = "@(#) $RCSfile: pbsD_connect.c,v $ $Revision: 2.1 $";

#ifdef uid_t
static uid_t pbs_current_uid;
#else
static int pbs_current_uid;
#endif

extern char pbs_current_user[PBS_MAXUSER];
extern struct connect_handle connection[NCONNECTS];
extern time_t pbs_tcp_timeout;

static unsigned int dflt_port = 0;
static char dflt_server[PBS_MAXSERVERNAME+1];
static int got_dflt = FALSE;
static char server_name[PBS_MAXSERVERNAME+1];
static unsigned int server_port;
static char *pbs_destn_file = PBS_DEFAULT_FILE;

char *pbs_server = 0;

char *
pbs_default()
{
	FILE *fd;
	char *pn;
	char *server;

	if( got_dflt != TRUE) {
		server = getenv("PBS_DEFAULT");
		if( (server == (char *)NULL) || (*server == '\0') ) {
			fd = fopen(pbs_destn_file, "r");
			if( fd == NULL ) return (char *)NULL;
			fgets(dflt_server, PBS_MAXSERVERNAME, fd);
			if (pn = strchr(dflt_server, (int)'\n'))
				*pn = '\0';
			fclose(fd);

		} else {
			strncpy(dflt_server, server, PBS_MAXSERVERNAME);
		}
		got_dflt = TRUE;
	}
	strcpy(server_name, dflt_server);
	return dflt_server;
}

static char *
PBS_get_server(server,port)
char *server;
unsigned int *port;
{
	int   i;
	char *pc;
	
	for( i=0;i<PBS_MAXSERVERNAME+1;i++ ) 
		server_name[i] = '\0';
	
	if (dflt_port == 0) 
		dflt_port = get_svrport(PBS_BATCH_SERVICE_NAME, "tcp",
					PBS_BATCH_SERVICE_PORT_DIS);

	/* first, get the "net.address[:port]" into 'server_name' */
		
	if( (server == (char *)NULL) || (*server == '\0') ) {
		if (pbs_default() == NULL)
			return NULL;
	} else {
		strncpy(server_name, server, PBS_MAXSERVERNAME);
	}
	
	/* now parse out the parts from 'server_name' */

	if (pc = strchr(server_name, (int)':')) {
		/* got a port number */
		*pc++ = '\0';
		*port = atoi(pc);
	} else {
		*port = dflt_port;
	}

	return server_name;
}

/*
 * PBS_authenticate - call pbs_iff(1) to authenticate use to the PBS server.
 */

static int PBSD_authenticate(psock)
	int			      psock;
{
	char   cmd[PBS_MAXSERVERNAME + 80];
	int    cred_type;
	int    i;
	int    j;
	FILE	*piff;

	/* Use pbs_iff to authenticate me */

	(void)sprintf(cmd, "%s %s %u %d", IFF_PATH, server_name, server_port,
		      psock);
	piff = (FILE *)popen(cmd, "r");
	if (piff == (FILE *)0)
		return (-1);

	
	i=read(fileno(piff), &cred_type, sizeof (int));
	if ((i != sizeof (int)) || 
	    (cred_type != PBS_credentialtype_none))
		return (-1);
	j = pclose(piff);
	if (j != 0)
		return (-1);


	return 0;
}

int pbs_connect(server)
char * server;
{
	struct sockaddr_in server_addr;
	struct hostent *hp;
	struct hostent *gethostbyname();
	int out;
	int i;
	struct passwd *pw;
	struct type_BATCH_BatchReply *reply;
	
	/* Reserve a connection state record */
	
	out = -1;
	for( i=1;i<NCONNECTS;i++ ) {
		if( connection[i].ch_inuse ) continue;
		out = i;
		connection[out].ch_inuse = 1;
		connection[out].ch_errno = 0;
		connection[out].ch_socket= -1;
		connection[out].ch_errtxt = (char *)NULL;
		break;
	}
	if( out < 0 ) {
		pbs_errno = PBSE_NOCONNECTS;
		return -1;
	}
	
	/* get server host and port	*/
	
	server = PBS_get_server(server, &server_port);
	if( server == (char *)NULL ) {
		connection[out].ch_inuse = 0;
		pbs_errno = PBSE_NOSERVER;
		return -1;
	}
		
	/* determine who we are */

	pbs_current_uid = getuid();
	if ((pw = getpwuid(pbs_current_uid)) == NULL) {
		pbs_errno = PBSE_SYSTEM;
		return -1;
	}
	strcpy(pbs_current_user,pw->pw_name);

	/* get socket	*/

	connection[out].ch_socket = socket(AF_INET, SOCK_STREAM, 0);
	if( connection[out].ch_socket < 0 ) {
		connection[out].ch_inuse = 0;
		pbs_errno = PBSE_PROTOCOL;
		return -1;
	}

	/* and connect... */

	pbs_server = server;    /* set for error messages from commands */
	
	server_addr.sin_family = AF_INET;
	hp = NULL;
	hp = gethostbyname(server);
	if( hp == NULL ) {
		close( connection[out].ch_socket);
		connection[out].ch_inuse = 0;
		pbs_errno = PBSE_BADHOST;
		return -1;
	}
	memcpy((char *)&server_addr.sin_addr, hp->h_addr_list[0], hp->h_length);
	server_addr.sin_port = htons(server_port);
	
	if (connect(connection[out].ch_socket, (struct sockaddr *)&server_addr, sizeof(server_addr)) < 0) {
		close( connection[out].ch_socket);
		connection[out].ch_inuse = 0;
		pbs_errno = errno;
		return -1;
	}

	/* Have pbs_iff authencate connection */

	if(PBSD_authenticate(connection[out].ch_socket)) {
		close( connection[out].ch_socket);
		connection[out].ch_inuse = 0;
		pbs_errno = PBSE_PERM;
		return -1;
	}

	/* setup DIS support routines for following pbs_* calls */

	DIS_tcp_setup(connection[out].ch_socket);
	pbs_tcp_timeout = 10800;	/* set for 3 hour time out */

	return out;
		
}

int pbs_disconnect(connect)
int connect;
{
	shutdown(connection[connect].ch_socket, 2);
	close( connection[connect].ch_socket );
	if( connection[connect].ch_errtxt != (char *)NULL ) 
		free( connection[connect].ch_errtxt );
	connection[connect].ch_errno = 0;
	connection[connect].ch_inuse = 0;
	return 0;
}

int pbs_query_max_connections()
{
  return NCONNECTS - 1;
}
