/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

/*	pbs_resc.c
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <string.h>
#include <stdio.h>
#include "libpbs.h"
#include "dis.h"

static char ident[] = "@(#) $RCSfile: pbsD_resc.c,v $ $Revision: 2.3 $";

/* Variables for this file */

static int nodes_avail = 0;
static int nodes_alloc = 0;
static int nodes_resrv = 0;
static int nodes_down  = 0;
static char *resc_nodes = "nodes";

/*
 * encode_DIS_resc() - encode a resource related request,
 *	Used by pbs_rescquery(), pbs_rescreserve() and pbs_rescfree()
 *
 *	Data items are:		signed int	resource handle
 *				unsigned int	count of query strings
 *		followed by that number of:
 *				 string	resource query
 */

static int encode_DIS_Resc(sock, rlist, ct, rh)
	int	      sock;
	char	    **rlist;
	int	      ct;
	resource_t    rh;
{
	int    i;
	char **pr;
	int    rc;

	if ( (rc = diswsi(sock, rh)) == 0) {  /* resource reservation handle */

		/* next send the number of resource strings */

		if ( (rc = diswui(sock, ct)) == 0) {
		
			/* now send each string (if any) */

			for (i = 0; i < ct; ++i) {
				if ( (rc = diswst(sock, *(rlist + i))) != 0)
					break;
			}
		}
	}
	return rc;
}

/*
 * PBS_resc() - internal common code for sending resource requests
 *
 *	Formats and sends the requests for pbs_rescquery(), pbs_rescreserve(),
 *	and pbs_rescfree().   Note, while the request is overloaded for all
 *	three, each has its own expected reply format.
 */

static int PBS_resc(c, reqtype, rescl, ct, rh)
	int	   c;
	int	   reqtype;
	char     **rescl;
	int	   ct;
	resource_t rh;
{
	int rc;
	int sock;

	sock = connection[c].ch_socket;

	/* setup DIS support routines for following DIS calls */

	DIS_tcp_setup(sock);

	if ( (rc = encode_DIS_ReqHdr(sock, reqtype, pbs_current_user)) ||
	     (rc = encode_DIS_Resc(sock, rescl, ct, rh)) ||
	     (rc = encode_DIS_ReqExtend(sock, (char *)0)) )   {
		connection[c].ch_errtxt = strdup(dis_emsg[rc]);
		return (pbs_errno = PBSE_PROTOCOL);
	}
	if (DIS_tcp_wflush(sock)) {
		return (pbs_errno = PBSE_PROTOCOL);
	}
	return (0);
}

/*
 * pbs_rescquery() - query the availability of resources
 */

int pbs_rescquery(c, resclist, num_resc, 
		available, allocated, reserved, down)
	int    c;
	char **resclist;	/* In - list of queries */
	int    num_resc;	/* In - number in list  */
	int   *available; 	/* Out - number available per query */
	int   *allocated; 	/* Out - number allocated per query */
	int   *reserved;	/* Out - number reserved  per query */
	int   *down;		/* Out - number down/off  per query */
{
	int i;
	struct batch_reply *reply;
	int rc = 0;


	if (resclist == 0) {
		connection[c].ch_errno = PBSE_RMNOPARAM;
		return (pbs_errno = PBSE_RMNOPARAM);
	}

	/* send request */

	if ((rc = PBS_resc(c, PBS_BATCH_Rescq, resclist, num_resc, (resource_t)0)) != 0)
		return (rc);

	/* read in reply */

	reply = PBSD_rdrpy(c);
	if ((rc = connection[c].ch_errno) == PBSE_NONE) {

		/* copy in available and allocated numbers */

		for (i=0; i < num_resc; ++i) {
		    *(available + i) = *(reply->brp_un.brp_rescq.brq_avail +i);
		    *(allocated + i) = *(reply->brp_un.brp_rescq.brq_alloc +i);
		    *(reserved  + i) = *(reply->brp_un.brp_rescq.brq_resvd +i);
		    *(down      + i) = *(reply->brp_un.brp_rescq.brq_down  +i);
		}
	}
	
	PBSD_FreeReply(reply);
	return (rc);
}

/*
 * pbs_reserve() - reserver resources
 *
 */

int pbs_rescreserve(c, rl, num_resc, prh)
	int	  c;		/* connection */
	char	**rl;		/* list of resources */
	int	  num_resc;	/* number of items in list */
	resource_t *prh;	/* ptr to resource reservation handle */
{
	int	ct = 0;
	char  **pr;
	int	rc;
	struct batch_reply *reply;

	if (rl == NULL) {
		connection[c].ch_errno = PBSE_RMNOPARAM;
		return (pbs_errno = PBSE_RMNOPARAM);
	}
	if (prh == NULL) {
		connection[c].ch_errno = PBSE_RMBADPARAM;
		return (pbs_errno = PBSE_RMBADPARAM);
	}

	/* send request */
	
	if ((rc = PBS_resc(c, PBS_BATCH_ReserveResc, rl, num_resc, *prh)) != 0)
		return (rc);

	/*
	 * now get reply, if reservation successful, the reservation handle,
	 * resource_t, is in the  aux field
	 */

	reply = PBSD_rdrpy(c);

	if ( ((rc = connection[c].ch_errno) == PBSE_NONE) ||
	     (rc == PBSE_RMPART) ) {
		*prh = reply->brp_auxcode;
	}
	PBSD_FreeReply(reply);

	return (rc);
}

/*
 * pbs_release() - release a resource reservation 
 *
 *	To encode we send same info as for reserve except that the resource
 *	list is empty.
 */

int pbs_rescrelease(c, rh)
	int	   c;
	resource_t rh;
{
	struct batch_reply *reply;
	int	rc;

	if ((rc = PBS_resc(c, PBS_BATCH_ReleaseResc, (char **)0, 0, rh)) != 0)
		return (rc);

	/* now get reply */

	reply = PBSD_rdrpy(c);

	PBSD_FreeReply(reply);

	return (connection[c].ch_errno);
}

/*
 * The following routines are provided as a convience in converting
 * older schedulers which did addreq() of "totpool", "usepool", and
 * "avail".
 *
 * The "update" flag if non-zero, causes a new resource query to be sent
 * to the server.  If zero, the existing numbers are used.
 */

/* totpool() - return total number of nodes */

int totpool(int con, int update)
{
	if (update)
		if (pbs_rescquery(con, &resc_nodes, 1, &nodes_avail,
				&nodes_alloc, &nodes_resrv, &nodes_down) != 0) {
			return (-1);
		}
	return (nodes_avail + nodes_alloc + nodes_resrv + nodes_down);
}


/* usepool() - return number of nodes in use, includes reserved and down */

int usepool(int con, int update)
{
	if (update)
		if (pbs_rescquery(con, &resc_nodes, 1, &nodes_avail,
				&nodes_alloc, &nodes_resrv, &nodes_down) != 0) {
			return (-1);
		}
	return (nodes_alloc + nodes_resrv + nodes_down);
}

/*
 * avail - returns answer about available of a specified node set
 *	"yes"	if available (job could be run)
 *	"no"	if not currently available
 *	"never"	if can never be satified
 *	"?"	if error in request
 */

char *avail(int con, char *resc)
{
	int av;
	int al;
	int res;
	int dwn;
	
	if (pbs_rescquery(con, &resc, 1, &av, &al, &res, &dwn) != 0)
		return ("?");

	else if (av > 0)
		return ("yes");
	else if (av == 0)
		return ("no");
	else
		return ("never");
}

