/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
 
static char ident[] = "@(#) $RCSfile: tcp_dis.c,v $ $Revision: 2.1 $";


#include <pbs_config.h>   /* the master config generated by configure */

#include <errno.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <sys/types.h>
#include <stdlib.h>

#if defined(FD_SET_IN_SYS_SELECT_H)
#  include <sys/select.h>
#endif

#include "dis.h"
#include "dis_init.h"

#define THE_BUF_SIZE 8192

struct tcpdisbuf {
	char *tdis_leadp;
	char *tdis_trailp;
	char *tdis_eod;
	char  tdis_thebuf[THE_BUF_SIZE];
};

struct	tcp_chan {
	struct	tcpdisbuf	readbuf;
	struct	tcpdisbuf	writebuf;
};

static	struct	tcp_chan	**tcparray = NULL;
static	int			tcparraymax = 0;

time_t	pbs_tcp_timeout = 30;
int	pbs_tcp_interrupt = 0;

/*
 * tcp_pack_buff - pack existing data into front of buffer
 *
 *	Moves "uncommited" data to front of buffer and adjusts pointers.
 *	Does a character by character move since data may over lap.
 */

static void tcp_pack_buff(tp)
	struct	tcpdisbuf	*tp;
{
	size_t amt;
	size_t start;
	int i;

	start = tp->tdis_trailp - tp->tdis_thebuf;
	if (start != 0) {
		amt  = tp->tdis_eod - tp->tdis_trailp;
		for (i=0; i<amt; ++i) {
			*(tp->tdis_thebuf + i) =
				*(tp->tdis_thebuf + i + start);
		}
		tp->tdis_leadp  -= start;
		tp->tdis_trailp -= start;
		tp->tdis_eod    -= start;
	}
}

/*
 * tcp_read - read data from tcp stream to "fill" the buffer
 *	Update the various buffer pointers.
 *
 *	Return:	>0 number of characters read
 *		 0 if EOD (no data currently avalable)
 *		-1 if error
 *		-2 if EOF (stream closed)
 */

static int tcp_read(fd)
	int fd;
{
	int i;
	fd_set	readset;
	struct	timeval timeout;
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->readbuf;

	/* must compact any uncommitted data into bottem of buffer */

	tcp_pack_buff(tp);

	/*
	 * we don't want to be locked out by an attack on the port to
	 * deny service, so we time out the read, the network had better
	 * deliver promptly
	 */
	do {
		timeout.tv_sec = pbs_tcp_timeout;
		timeout.tv_usec = 0;
		FD_ZERO(&readset);
		FD_SET(fd, &readset);
		i = select(FD_SETSIZE, &readset, (fd_set *)0,
			(fd_set *)0, &timeout);
		if (pbs_tcp_interrupt)
			break;
	} while ((i == -1) && (errno == EINTR));
	if (i <= 0)
		return i;
	
	while ((i = read(fd, tp->tdis_eod,
			tp->tdis_thebuf +
			THE_BUF_SIZE -
			tp->tdis_eod)) == -1) {
		if (errno != EINTR)
			break;
	}
	if (i > 0)
		tp->tdis_eod += i;
	return ((i == 0) ? -2 : i);
}

/*
 * DIS_tcp_wflush - flush tcp/dis write buffer
 *
 *	Writes "committed" data in buffer to file discriptor,
 *	packs remaining data (if any), resets pointers
 *	Returns: 0 on success, -1 on error
 */

int DIS_tcp_wflush(fd)
	int fd;
{
	size_t	ct;
	int	i;
	char	*pb;
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->writebuf;
	pb = tp->tdis_thebuf;

	ct = tp->tdis_trailp - tp->tdis_thebuf;
	while ((i = write(fd, pb, ct)) != ct) {
		if (i == -1)  {
			if (errno != EINTR)
				return (-1);
			else
				continue;
		}
		ct -= i;
		pb += i;
	}
	tp->tdis_eod = tp->tdis_leadp;
	tcp_pack_buff(tp);
	return 0;
}

/*
 * DIS_buf_clear - reset tpc/dis buffer to empty
 */

static void
DIS_tcp_clear(tp)
	struct	tcpdisbuf	*tp;
{
	tp->tdis_leadp  = tp->tdis_thebuf;
	tp->tdis_trailp = tp->tdis_thebuf;
	tp->tdis_eod    = tp->tdis_thebuf;
}

void DIS_tcp_reset(fd, i)
	int	fd;
	int	i;
{
	struct	tcp_chan	*tcp;

	tcp = tcparray[fd];
	DIS_tcp_clear( i==0 ? &tcp->readbuf : &tcp->writebuf );
}

/*
 * tcp_rskip - tcp/dis suport routine to skip over data in read buffer
 *
 *	Returns: 0 on success, -1 on error
 */

static int tcp_rskip(fd, ct)
	int	fd;
	size_t	ct;
{
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->readbuf;
	if (tp->tdis_leadp - tp->tdis_eod < ct)
		/* this isn't the best thing to do, but this isn't used, so */
		return (-1);
	tp->tdis_leadp += ct;
	return 0;
}

/*
 * tcp_getc - tcp/dis support routine to get next character from read buffer
 *
 *	Return:	>0 number of characters read
 *		-1 if EOD or error
 *		-2 if EOF (stream closed)
 */

static int tcp_getc(fd)
	int fd;
{
	int	x;
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->readbuf;
	if (tp->tdis_leadp >= tp->tdis_eod) {
		/* not enought data, try to get more */
		x = tcp_read(fd);
		if (x <= 0)
			return ((x == -2) ? -2 : -1);	/* Error or EOF */
	}
	return ((int)*tp->tdis_leadp++);
}

/*
 * tcp_gets - tcp/dis support routine to get a string from read buffer
 *
 *	Return:	>0 number of characters read
 *		 0 if EOD (no data currently avalable)
 *		-1 if error
 *		-2 if EOF (stream closed)
 */

static int tcp_gets(fd, str, ct)
	int	fd;
	char	*str;
	size_t	ct;
{
	int	x;
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->readbuf;
	while (tp->tdis_eod - tp->tdis_leadp < ct) {
		/* not enought data, try to get more */
		x = tcp_read(fd);
		if (x <= 0)
			return x;	/* Error or EOF */
	}
	(void)memcpy((char *)str, tp->tdis_leadp, ct);
	tp->tdis_leadp += ct;
	return (int)ct;
}

/*
 * tcp_puts - tcp/dis support routine to put a counted string of characters
 *	into the write buffer.
 *
 *	Returns: >= 0, the number of characters placed
 *		 -1 if error
 */

static int tcp_puts(fd, str, ct)
	int fd;
	const char *str;
	size_t ct;
{
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->writebuf;
	if ((tp->tdis_thebuf + THE_BUF_SIZE - tp->tdis_leadp) < ct) {
		/* not enough room, try to flush committed data */
		if ((DIS_tcp_wflush(fd) < 0) ||
				((tp->tdis_thebuf +
				THE_BUF_SIZE -
				tp->tdis_leadp) < ct))
			return -1;		/* error */
	}
	(void)memcpy(tp->tdis_leadp, (char *)str, ct);
	tp->tdis_leadp += ct;
	return ct;
}

/*
 * tcp_rcommit - tcp/dis support routine to commit/uncommit read data 
 */

static int tcp_rcommit(fd, commit_flag)
	int fd;
	int commit_flag;
{
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->readbuf;
	if (commit_flag) {
		/* commit by moving trailing up */
		tp->tdis_trailp = tp->tdis_leadp;
	} else {
		/* uncommit by moving leading back */
		tp->tdis_leadp = tp->tdis_trailp;
	}
	return 0;
}

/*
 * tcp_wcommit - tcp/dis support routine to commit/uncommit write data 
 */

static int tcp_wcommit(fd, commit_flag)
	int fd;
	int commit_flag;
{
	struct	tcpdisbuf	*tp;

	tp = &tcparray[fd]->writebuf;
	if (commit_flag) {
		/* commit by moving trailing up */
		tp->tdis_trailp = tp->tdis_leadp;
	} else {
		/* uncommit by moving leading back */
		tp->tdis_leadp = tp->tdis_trailp;
	}
	return 0;
}

void DIS_tcp_funcs()
{
	if (dis_getc != tcp_getc) {
		dis_getc = tcp_getc;
		dis_puts = tcp_puts;
		dis_gets = tcp_gets;
		disr_skip = tcp_rskip;
		disr_commit = tcp_rcommit;
		disw_commit = tcp_wcommit;
	}
}

/*
 * DIS_tcp_setup - setup supports routines for dis, "data is strings", to
 * 	use tcp stream I/O.  Also initializes an array of pointers to
 *	buffers and a buffer to be used for the given fd.
 */

void DIS_tcp_setup(fd)
	int	fd;
{
	struct	tcp_chan	*tcp;

	/* set DIS function pointers to tcp routines */
	DIS_tcp_funcs();

	if (fd >= tcparraymax) {
		int	hold = tcparraymax;

		tcparraymax = fd+10;
		if (tcparray == NULL) {
			tcparray = (struct tcp_chan **)calloc(tcparraymax,
					sizeof(struct tcp_chan *));
		}
		else {
			tcparray = (struct tcp_chan **)realloc(tcparray,
					tcparraymax*sizeof(struct tcp_chan *));
			memset(&tcparray[hold], '\0',
				(tcparraymax-hold)*sizeof(struct tcp_chan *));
		}
	}
	tcp = tcparray[fd];
	if (tcp == NULL) {
		tcp = tcparray[fd] =
			(struct tcp_chan *)malloc(sizeof(struct tcp_chan));
	}

	/* initialize read and write buffers */
	DIS_tcp_clear(&tcp->readbuf);
	DIS_tcp_clear(&tcp->writebuf);
}
