/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/stat.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <limits.h>
#include "portability.h"
#include "log.h"

static char ident[] = "@(#) $RCSfile: chk_file_sec.c,v $ $Revision: 2.1 $";

#ifndef	S_ISLNK
#define	S_ISLNK(m)	(((m) & S_IFMT) == S_IFLNK)
#endif

/*
 * chk_file_sec() - Check file/directory security
 *      Part of the PBS System Security "Feature"
 *
 *	To be secure, all directories (and final file) in path must be:
 *		owned by uid < 10
 *		owned by group < 10 if group writable
 *		not have world writable unless stick bit set & this is allowed.
 *
 * Returns 0 if ok
 *      errno value if not ok, including:
 *              EPERM if not owned by root
 *              ENOTDIR if not file/directory as specified
 *              EACCESS if permissions are not ok
 */

int chk_file_sec(path, isdir, sticky, disallow, fullpath)
	char *path;		/* path to check */
	int   isdir;		/* 1 = path is directory, 0 = file */
	int   sticky;		/* allow write on directory if sticky set */
	int   disallow;		/* perm bits to disallow */
	int   fullpath;		/* recursively check full path */
{
	int    i;
	char  *error_buf;
	char  *pc;
	int    rc = 0;
	struct stat sbuf;
	char   shorter[_POSIX_PATH_MAX];
	char   symlink[_POSIX_PATH_MAX];
	
	if ((*path == '/') && fullpath) {

	    /* check full path starting at root */

	    (void)strcpy(shorter, path);
	    pc  = strrchr(shorter, (int)'/');
	    if ( (pc != (char *)0) && (pc != shorter) ) {

		/*
		 * push "dirname" onto stack, stack will pop back from
		 * root to the given file/directory
 		 */

		*pc = '\0';
		if (rc = chk_file_sec(shorter,1,sticky,S_IWGRP|S_IWOTH,1)) {
			return (rc);
		}
	    }
	}

	if (lstat(path, &sbuf) == -1) {
		rc = errno;
		goto chkerr;
	}

	if (S_ISLNK(sbuf.st_mode) != 0) {
		i = readlink(path, symlink, _POSIX_PATH_MAX);
		if (i < 0) {
			rc = errno;
			goto chkerr;
		}
		*(symlink+i) = '\0';
		if (*symlink == '/') {
			return (chk_file_sec(symlink,isdir,sticky,disallow,fullpath));
		} else {
			(void)strcat(shorter, "/");
			(void)strcat(shorter, symlink);
			return (chk_file_sec(shorter,isdir,sticky,disallow,fullpath));
		}
	}
			
	i = sbuf.st_mode & (S_IRWXU | S_IRWXG | S_IRWXO);
	if (sbuf.st_uid > 10) {
		rc = EPERM;

	} else if ( ((isdir == 1) && (S_ISDIR(sbuf.st_mode)==0)) ||
	     ((isdir == 0) && (S_ISREG(sbuf.st_mode)==0)) ) {
		rc = ENOTDIR;

	} else if (i & disallow)	{
		/* if group write, gid must be less than 10 */
		if ( (i & disallow & S_IWGRP) && (sbuf.st_gid > 9) )
			rc = EPERM;
		/* if world write, sticky bit must be set and "sticky" ok */
		if (i & disallow & S_IWOTH) {
			if ((S_ISDIR(sbuf.st_mode) == 0) || 
			    (((sbuf.st_mode & S_ISVTX) == 0) || (sticky != 1)) )
				rc =  EACCES;
		}
	}

chkerr:
	if (rc != 0) {
		if ((error_buf = malloc(1024)) == 0) {
			log_err(rc, "chk_file_sec", "Malloc failed");		
		} else {
			(void)sprintf(error_buf,
					"Security violation with \"%s\"",
					path);
			log_err(rc, "chk_file_sec", error_buf);
			(void)free(error_buf);
		}
	}

	return (rc);
}
