/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * pbs_log.c - contains functions to log error and event messages to
 *	the log file.
 *
 * Functions included are:
 *	log_open()
 *	log_err()
 *	log_record()
 *	log_close()
 */

static char ident[] = "@(#) $RCSfile: pbs_log.c,v $ $Revision: 2.1 $";

#include <pbs_config.h>   /* the master config generated by configure */

#include "portability.h"
#include "pbs_error.h"

#include <sys/param.h>
#include <sys/types.h>
#include <limits.h>
#include <time.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "log.h"
#if SYSLOG
#include <syslog.h>
#endif

/* Global Data */

extern int errno;
char log_buffer[LOG_BUF_SIZE];
char log_directory[_POSIX_PATH_MAX/2];

extern char *msg_daemonname;

/* Local Data */

static int	     log_auto_switch = 0;
static int	     log_open_day;
static FILE	    *logfile;		/* open stream for log file */
static volatile int  log_opened = 0;
#if SYSLOG
static int	     syslogopen = 0;
#endif	/* SYSLOG */

/* 
 * the order of these names MUST match the defintions of 
 * PBS_EVENTCLASS_* in log.h
 */
static char *class_names[] = {
		"n/a",
		"Svr",
		"Que",
		"Job",
		"Req",
		"Fil",
		"Act",
		"node" };

/* External functions called */

/*
 * mk_log_name - make the log name used by MOM
 *	based on the date: yyyymmdd
 */

static char *mk_log_name(pbuf)
	char *pbuf;
{
	struct tm *ptm;
	time_t time_now;

	time_now = time((time_t *)0);
	ptm = localtime(&time_now);
	(void)sprintf(pbuf, "%s/%04d%02d%02d", log_directory, ptm->tm_year+1900,
		      ptm->tm_mon+1, ptm->tm_mday);
	log_open_day = ptm->tm_yday;	/* Julian date log opened */
	return (pbuf);
}

/*
 * log_open() - open the log file for append.
 *
 *	Opens a (new) log file. 
 *	If a log file is already open, and the new file is successfully opened,
 *	the old file is closed.  Otherwise the old file is left open.
 */

int log_open(filename, directory)
	char *filename;		/* abs filename or NULL */
	char *directory;	/* normal log directory */
{
	char  buf[_POSIX_PATH_MAX];
	int   fds;


	if (log_opened > 0)	
		return (-1);	/* already open */

	(void)strncpy(log_directory, directory, _POSIX_PATH_MAX/2-1);

	if ((filename == (char *)0) || (*filename == '\0')) {
		filename = mk_log_name(buf);
		log_auto_switch = 1;
	} else if (*filename != '/') {
		return (-1);	/* must be absolute path */
	}

	if ((fds = open(filename, O_CREAT|O_WRONLY|O_APPEND, 0644)) < 0) {
		log_opened = -1;	/* note that open failed */
		return (-1);
	} 
	if (fds < 3) {
		log_opened = fcntl(fds, F_DUPFD, 3);	/* overload variable */
		if (log_opened < 0)
			return (-1);
		(void)close(fds);
		fds = log_opened;
	}
	logfile = fdopen(fds, "a");
	(void)setvbuf(logfile, NULL, _IOLBF, 0);	/* set line buffering */
	log_opened = 1;			/* note that file is open */

	log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, "Log", "Log opened");

	return (0);
}

/*
 * log_err - log an internal error
 *	The error is recorded to the pbs log file and to syslogd if it is
 *	available.  If the error file has not been opened and if syslog is
 *	not defined, then the console is opened.
 */

void log_err(errnum, routine, text)
	int	errnum;
	char   *routine;
	char   *text;
{
	char buf[LOG_BUF_SIZE], *errmsg;
	int  i;

	if (errnum == -1) {
		buf[0] = '\0';
	} else {
		if ((errmsg = strerror(errnum)) == NULL &&
				(errmsg = pbse_to_txt(errnum)) == NULL)
			errmsg = "";
		(void)sprintf(buf, "%s (%d) in ", errmsg, errnum);
	}
	(void)strcat(buf, routine);
	(void)strcat(buf, ", ");
	i = LOG_BUF_SIZE - (int)strlen(buf) - 2;
	(void)strncat(buf, text, i);
	buf[LOG_BUF_SIZE -1] = '\0';

	if (log_opened == 0)  {
#if ! SYSLOG
		(void)log_open("/dev/console", log_directory);
#endif	/* not SYSLOG */
	}
	if (isatty(2))
		(void)fprintf(stderr, "%s: %s\n", msg_daemonname, buf);
			
	if (log_opened > 0)
		(void)log_record(PBSEVENT_ERROR | PBSEVENT_FORCE,
				PBS_EVENTCLASS_SERVER, 
				msg_daemonname, buf);

#if SYSLOG
	if (syslogopen == 0) {
		openlog(msg_daemonname, LOG_NOWAIT, LOG_DAEMON);
		syslogopen = 1;
	}
	syslog(LOG_ERR|LOG_DAEMON, buf);
#endif	/* SYSLOG */
	return;
}

/*
 * log_record - log a message to the log file
 *	The log file must have been opened by log_open().
 *
 *	The caller should ensure proper formating of the message if "text"
 *	is to contain "continuation lines".
 */

void log_record(eventtype, objclass, objname, text)
	int   eventtype;
	int   objclass;
	char *objname;
	char *text;
{
	time_t now;
	struct tm *ptm;
	int    rc;
	FILE  *savlog;

	if (log_opened < 1)
		return;

	now = time((time_t *)0);	/* get time for message */
	ptm = localtime(&now);

	/* Do we need to switch the log? */

	if (log_auto_switch && (ptm->tm_yday != log_open_day)) {
		log_close(1);
		log_open((char *)0, log_directory);
	}

	rc = fprintf(logfile,
		"%02d/%02d/%04d %02d:%02d:%02d;%04x;%10.10s;%s;%s;%s\n",
		ptm->tm_mon+1, ptm->tm_mday, ptm->tm_year+1900,
		ptm->tm_hour, ptm->tm_min, ptm->tm_sec,
		eventtype & ~PBSEVENT_FORCE,
		msg_daemonname,
		class_names[objclass],
		objname,
		text);
	(void)fflush(logfile);
	if (rc < 0) {
		rc = errno;
		clearerr(logfile);
		savlog = logfile;
		logfile = fopen("/dev/console", "w");
		log_err(rc, "log_record", "PBS cannot write to its log");
		fclose(logfile);
		logfile = savlog;
	}
}

/*
 * log_close - close the current open log file
 */

void log_close(msg)
	int msg;
{
	if (log_opened == 1) {
		log_auto_switch = 0;
		if (msg) {
			log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER,
				   "Log", "Log closed");
		}
		(void)fclose(logfile);
		log_opened = 0;
	}
#if SYSLOG
	if (syslogopen)
		closelog();
#endif	/* SYSLOG */
}
