/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <netdb.h>
#include "portability.h"
#include "server_limits.h"
#include "net_connect.h"

static char ident[] = "@(#) $RCSfile: net_client.c,v $ $Revision: 2.1 $";

/*
 * client_to_svr - connect to a server
 *
 *	Perform socket/tcp/ip stuff to connect to a server.
 *
 *	Returns: >=0 the socket obtained, or
 *		 PBS_NET_RC_FATAL (-1) if fatal error, just quit, or
 *		 PBS_NET_RC_RETRY (-2) if temp error, should retry
 *
 * Note, the server's host address and port are was chosen as parameters
 * rather than their names to possibly save extra look-ups.  It seems likely
 * that the caller "might" make several calls to the same host or different
 * hosts with the same port.  Let the caller keep the addresses arround
 * rather than look it up each time.
 */

int client_to_svr(hostaddr, port, local_port)
	pbs_net_t	hostaddr;	/* Internet addr  of host */
	unsigned int    port;		/* port to which to connect */
	int	        local_port;	/* not 0 if use local reserved port */
{
	struct sockaddr_in local;
	struct sockaddr_in remote;
	int		 sock;
	unsigned short	 tryport;

	local.sin_family = AF_INET;
	local.sin_addr.s_addr = 0;

	/* get socket					*/

	sock = socket(AF_INET, SOCK_STREAM, 0);
	if (sock < 0) {
		return (PBS_NET_RC_FATAL);
	} else if (sock >= PBS_NET_MAX_CONNECTIONS) {
		(void)close(sock);		/* too many connections */
		return (PBS_NET_RC_RETRY);
	}

	/*	If local privilege port requested, bind to one	*/
	/*	Must be root privileged to do this		*/

	if (local_port) {
		tryport = IPPORT_RESERVED - 1;
		local.sin_port = htons(tryport);

		while (bind(sock,(struct sockaddr *)&local,sizeof(local)) < 0) {
			if ((errno != EADDRINUSE) && 
			    (errno != EADDRNOTAVAIL)) {
				(void)close(sock);
				return (PBS_NET_RC_FATAL);
			} else if (--tryport < (unsigned short)(IPPORT_RESERVED/2)) {
				(void)close(sock);
				return (PBS_NET_RC_RETRY);
			}
			local.sin_port = htons(tryport);
		}
	}
			
/*	connect to specified server host and port	*/

	remote.sin_addr.s_addr = htonl(hostaddr);
	remote.sin_port = htons((unsigned short)port);
	remote.sin_family = AF_INET;
	if (connect(sock, (struct sockaddr *)&remote, sizeof(remote)) < 0) {
		switch (errno) {
		    case EINTR:
		    case EADDRINUSE:
		    case ETIMEDOUT:
		    case ECONNREFUSED:
			close(sock);
			return (PBS_NET_RC_RETRY);
		    default:
			close(sock);
			return (PBS_NET_RC_FATAL);
		}
			
	} else {
		return (sock);
	}
}
