/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <netinet/in.h>
#include <sys/socket.h>

#include <errno.h>
#include <fcntl.h>
#include <stdio.h>
#include <string.h>
#include <termios.h>
#include <unistd.h>
#ifdef sun
#include <sys/stream.h>
#include <sys/stropts.h>
#endif /* sun */
#if defined(HAVE_SYS_IOCTL_H)
#include <sys/ioctl.h>
#endif
#if !defined(sgi) && !defined(_AIX) && !defined(linux)
#include <sys/tty.h>
#endif  /* ! sgi */
#include "portability.h"
#include "pbs_ifl.h"
#include "server_limits.h"

static char ident[] = "@(#) $RCSfile: mom_inter.c,v $ $Revision: 2.1 $";

static char cc_array[PBS_TERM_CCA];
static struct winsize wsz;

extern int mom_reader_go;

/*
 * read_net - read data from network till received amount expected
 *
 *	returns >0 amount read
 *		-1 on error
 */
static int read_net(sock, buf, amt)
	int    sock;
	char  *buf;
	int    amt;
{
	int got;
	int total = 0;

	while (amt > 0) {
		got = read(sock, buf, amt);
		if (got > 0) {	/* read (some) data */
			amt   -= got;
			buf   += got;
			total += got;
		} else if (got == 0)
			break;
		else
			return (-1);
	}
	return (total); 
}

/*
 * rcvttype - receive the terminal type of the real terminal
 *
 *	Sent over network as "TERM=type_string"
 */

char *rcvttype(sock)
	int sock;
{
	static char buf[PBS_TERM_BUF_SZ];

	/* read terminal type as sent by qsub */

	if ( (read_net(sock, buf, PBS_TERM_BUF_SZ) != PBS_TERM_BUF_SZ) ||
	     (strncmp(buf, "TERM=", 5) != 0) )
		return ((char *)0);

	/* get the basic control characters from qsub's termial */

	if (read_net(sock, cc_array, PBS_TERM_CCA) != PBS_TERM_CCA) {
		return ((char *)0);
	}
	
	return (buf);
}

/* 
 * set_termcc - set the basic modes for the slave terminal, and set the
 *	control characters to those sent by qsub.
 */

void set_termcc(fd)
{
	struct termios slvtio;

#ifdef	PUSH_STREAM
	(void)ioctl(fd, I_PUSH, "ptem");
	(void)ioctl(fd, I_PUSH, "ldterm");
#endif	/* PUSH_STREAM */

	if (tcgetattr(fd, &slvtio) < 0)
		return;		/* cannot do it, leave as is */

#ifdef IMAXBEL
	slvtio.c_iflag = (BRKINT|IGNPAR|ICRNL|IXON|IXOFF|IMAXBEL);
#else
	slvtio.c_iflag = (BRKINT|IGNPAR|ICRNL|IXON|IXOFF);
#endif
	slvtio.c_oflag = (OPOST|ONLCR);
#if defined(ECHOKE) && defined(ECHOCTL)
	slvtio.c_lflag = (ISIG|ICANON|ECHO|ECHOE|ECHOK|ECHOKE|ECHOCTL);
#else
	slvtio.c_lflag = (ISIG|ICANON|ECHO|ECHOE|ECHOK);
#endif
	slvtio.c_cc[VEOL]   = '\0';
	slvtio.c_cc[VEOL2]  = '\0';
	slvtio.c_cc[VSTART] = '\021';		/* ^Q */
	slvtio.c_cc[VSTOP]  = '\023';		/* ^S */
#if defined(VDSUSP)
	slvtio.c_cc[VDSUSP] = '\031';		/* ^Y */
#endif
#if defined(VREPRINT)
	slvtio.c_cc[VREPRINT] = '\022';		/* ^R */
#endif
	slvtio.c_cc[VLNEXT] = '\017';		/* ^V */

	slvtio.c_cc[VINTR]  = cc_array[0];
	slvtio.c_cc[VQUIT]  = cc_array[1];
	slvtio.c_cc[VERASE] = cc_array[2];
	slvtio.c_cc[VKILL]  = cc_array[3];
	slvtio.c_cc[VEOF]   = cc_array[4];
	slvtio.c_cc[VSUSP]  = cc_array[5];
	(void)tcsetattr(fd, TCSANOW, &slvtio);
}
	


/*
 * rcvwinsize - receive the window size of the real terminal window
 *
 *	Sent over network as "WINSIZE rn cn xn yn"  where .n is numeric string
 */

int rcvwinsize(sock)
	int sock;
{
	char buf[PBS_TERM_BUF_SZ];

	if (read_net(sock, buf, PBS_TERM_BUF_SZ) != PBS_TERM_BUF_SZ)
		return (-1);
	if (sscanf(buf, "WINSIZE %hu,%hu,%hu,%hu", &wsz.ws_row, &wsz.ws_col,
					&wsz.ws_xpixel, &wsz.ws_ypixel) != 4)
		return (-1);
	return 0;
}

int setwinsize(pty)
	int pty;
{
	if (ioctl(pty, TIOCSWINSZ, &wsz) < 0) {
		perror("ioctl TIOCSWINSZ");
		return (-1);
	}
	return (0);
}


/*
 * reader process - reads from the remote socket, and writes
 *	to the master pty
 */
int mom_reader(s, ptc)
	int s;
	int ptc;
{
	char buf[1024];
	int c;

	/* read from the socket, and write to ptc */
	while (mom_reader_go) {
		c = read(s, buf, sizeof(buf));
		if (c > 0) {
			int wc;
			char *p = buf;
	
			while (c) {
				if ((wc = write(ptc, p, c)) < 0) {
					if (errno == EINTR) {
						continue;
					}
					return (-1);
				}
				c -= wc;
				p += wc;
			}
		} else if (c == 0) {
			return (0);
		} else if (c < 0) {
			if (errno == EINTR)
				continue;
			else {
				return (-1);
			}
		}
	}
	return 0;
}


/*
 * Writer process: reads from master pty, and writes
 * data out to the rem socket
 */
int
mom_writer(s, ptc)
	int s;
	int ptc;
{
	char buf[1024];
	int c;

	/* read from ptc, and write to the socket */
	while (1) {
		c = read(ptc, buf, sizeof(buf));
		if (c > 0) {
			int wc;
			char *p = buf;
	
			while (c) {
				if ((wc = write(s, p, c)) < 0) {
					if (errno == EINTR) {
						continue;
					}
					return (-1);
				}
				c -= wc;
				p += wc;
			}
		} else if (c == 0) {
			return (0);
		} else if (c < 0) {
			if (errno == EINTR)
				continue;
			else {
				return (-1);
			}
		}
	}
}

/*
 * conn_qsub - connect to the qsub that submitted this interactive job
 */

int conn_qsub(hostname, port)
	char *hostname;
	long  port;
{
	pbs_net_t hostaddr;

	if ((hostaddr = get_hostaddr(hostname)) == (pbs_net_t)0)
		return (-1);
	return (client_to_svr(hostaddr, (unsigned int)port, 0));
}
