/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include	<assert.h>
#include	<stdio.h>
#include	<stdlib.h>
#include	<unistd.h>
#include	<string.h>
#include	<ctype.h>
#include	<errno.h>
#include	<time.h>
#include	<limits.h>
#include	<netdb.h>
#include	<sys/types.h>
#include	<sys/param.h>
#include	<sys/times.h>
#include	<sys/stat.h>
#include	<netinet/in.h>
#include	<sys/time.h>

#include 	"pbs_ifl.h"
#include	"pbs_error.h"
#include	"log.h"
#include	"net_connect.h"
#include	"rpp.h"
#include	"dis.h"
#include	"dis_init.h"
#include 	"pbs_nodes.h"
#include	"resmon.h"

static char ident[] = "@(#) $RCSfile: mom_server.c,v $ $Revision: 2.3 $";


/* Global Data Items */

extern	unsigned int	default_server_port;
extern	char		mom_host[];
extern	char		*path_jobs;
extern	char		*path_home;
extern	int		pbs_errno;
extern	unsigned int	pbs_mom_port;
extern	unsigned int	pbs_rm_port;
extern	unsigned int	pbs_tm_port;
extern	time_t		time_now;
extern	int		internal_state;

int			server_stream = -1;	/* XXX */

void	state_to_server A_((int));

/*
 * Tree search generalized from Knuth (6.2.2) Algorithm T just like
 * the AT&T man page says.
 *
 * The node_t structure is for internal use only, lint doesn't grok it.
 *
 * Written by reading the System V Interface Definition, not the code.
 *
 * Totally public domain.
 */
/*LINTLIBRARY*/

/*
**	Modified by Tom Proett <proett@nas.nasa.gov> for PBS.
*/

typedef struct node_t {
	u_long		key;
	struct node_t	*left, *right;
} node;
node		*okclients = NULL;	/* tree of ip addrs */

/* find value in tree, return 1 if found, 0 if not */
int
tfind(key, rootp)
	const u_long 	key;		/* key to be located */
	node		**rootp;	/* address of tree root */
{
	register node	*q;

	if (rootp == (struct node_t **)0)
		return 0;
	while (*rootp != (struct node_t *)0) {	/* Knuth's T1: */
		if (key == (*rootp)->key)	/* T2: */
			return 1;		/* we found it! */
		rootp = (key < (*rootp)->key) ?
			&(*rootp)->left :	/* T3: follow left branch */
			&(*rootp)->right;	/* T4: follow right branch */
	}
	return 0;
}

void
tinsert(key, rootp)
	const u_long 	key;		/* key to be located */
	node		**rootp;	/* address of tree root */
{
	register node	*q;

	if (rootp == (struct node_t **)0)
		return;
	while (*rootp != (struct node_t *)0) {	/* Knuth's T1: */
		if (key == (*rootp)->key)	/* T2: */
			return;			/* we found it! */
		rootp = (key < (*rootp)->key) ?
			&(*rootp)->left :	/* T3: follow left branch */
			&(*rootp)->right;	/* T4: follow right branch */
	}
	q = (node *) malloc(sizeof(node));	/* T5: key not found */
	if (q != (struct node_t *)0) {		/* make new node */
		*rootp = q;			/* link new node to old */
		q->key = key;			/* initialize new node */
		q->left = q->right = (struct node_t *)0;
	}
	return;
}

void
tfree(rootp)
	node		**rootp;
{
	if (rootp == NULL || *rootp == NULL)
		return;
	tfree(&(*rootp)->left);
	tfree(&(*rootp)->right);
	free(*rootp);
	*rootp = NULL;
}

/*
**	Start a standard inter-server message.
*/
int
is_compose(stream, command)
     int	stream;
     int	command;
{
	int	ret;

	if (stream < 0)
		return DIS_EOF;
	DIS_rpp_reset();

	ret = diswsi(stream, IS_PROTOCOL);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(stream, IS_PROTOCOL_VER);
	if (ret != DIS_SUCCESS)
		goto done;
	ret = diswsi(stream, command);
	if (ret != DIS_SUCCESS)
		goto done;
	return DIS_SUCCESS;

 done:
	DBPRT(("is_compose: send error %s\n", dis_emsg[ret]))
	return ret;
}

/*
**	Input is coming from another server over a DIS rpp stream.
**	Read the stream to get a Inter-Server request.
*/
void
is_request(stream, version)
     int	stream;
     int	version;
{
	static	char		id[] = "is_request";
	int			command = 0;
	int			ret = DIS_SUCCESS;
	int			i;
	u_long			ipaddr;
	short			port;
	struct	sockaddr_in	*addr;
	void			init_addrs();

	DBPRT(("%s: stream %d version %d\n", id, stream, version))
	if (version != IS_PROTOCOL_VER) {
		sprintf(log_buffer, "protocol version %d unknown", version);
		log_err(-1, id, log_buffer);
		rpp_close(stream);
		return;
	}

	/* check that machine is okay to be a server */
	addr = rpp_getaddr(stream);
	port = ntohs((unsigned short)addr->sin_port);
	ipaddr = ntohl(addr->sin_addr.s_addr);
	if (port >= IPPORT_RESERVED || !tfind(ipaddr, &okclients)) {
		sprintf(log_buffer, "bad connect from %s",
			netaddr(addr));
		log_err(-1, id, log_buffer);
		rpp_close(stream);
		return;
	}

	command = disrsi(stream, &ret);
	if (ret != DIS_SUCCESS)
		goto err;

	switch (command) {

	case IS_NULL:		/* a ping from the server */
		DBPRT(("%s: IS_NULL\n", id))
		if (internal_state & INUSE_DOWN) {
			state_to_server(1);
		}
		break;

	case IS_HELLO:		/* server wants a return ping */
		DBPRT(("%s: IS_HELLO, state=0x%x\n", id, internal_state))
		server_stream = stream;		/* save stream to server XXX */
		is_compose(stream, IS_HELLO);
		rpp_flush(stream);
		if (internal_state != 0)
			state_to_server(1);
		break;

	case IS_CLUSTER_ADDRS:
		DBPRT(("%s: IS_CLUSTER_ADDRS\n", id))
		for (;;) {
			ipaddr = disrul(stream, &ret);
			if (ret != DIS_SUCCESS)
				break;
			DBPRT(("%s:\t%ld.%ld.%ld.%ld\n", id,
				(ipaddr & 0xff000000) >> 24,
				(ipaddr & 0x00ff0000) >> 16,
				(ipaddr & 0x0000ff00) >> 8,
				(ipaddr & 0x000000ff) ))
			tinsert(ipaddr, &okclients);
		}
		if (ret != DIS_EOD)
			goto err;
		break;

	default:
		sprintf(log_buffer, "unknown command %d sent", command);
		log_err(-1, id, log_buffer);
		goto err;
	}

	rpp_eom(stream);
	return;

 err:
	/*
	** We come here if we got a DIS read error or a protocol
	** element is missing.
	*/
	sprintf(log_buffer, "%s from %s", dis_emsg[ret], netaddr(addr));
	log_err(-1, id, log_buffer);
	rpp_close(stream);

	return;
}

/*
 * check_busy() - 
 *	If current load average ge max_load_val and busy not already set
 *		set it
 *	If current load average lt ideal_load_val and busy currently set
 *		unset it
 */
void check_busy(mla)
	double	mla;
{
	extern int   internal_state;
	extern float ideal_load_val;
	extern float max_load_val;

	if ( (mla >= max_load_val) && ((internal_state & INUSE_BUSY) == 0) )
	    internal_state |= INUSE_BUSY | UPDATE_MOM_STATE;
	else if ((mla < ideal_load_val) && ((internal_state & INUSE_BUSY) != 0))
	    internal_state  = (internal_state & ~INUSE_BUSY) | UPDATE_MOM_STATE;
}

/*
 * state_to_server() - if UPDATE_MOM_STATE is set, send state message to
 *	the server.
 */
void state_to_server(force)
	int force;
{
	if (force || (internal_state & UPDATE_MOM_STATE)) {
		if (server_stream < 0)
			return;

		if (is_compose(server_stream, IS_UPDATE) != DIS_SUCCESS) {
			return;		
		} else if (diswui(server_stream, internal_state & ~UPDATE_MOM_STATE) != DIS_SUCCESS) {
			return;
		}
		rpp_flush(server_stream);
		
		internal_state &= ~UPDATE_MOM_STATE;
		DBPRT(("updateing state 0x%x to server\n", internal_state))
	}
}
