/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#define PBS_MOM 1
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/stat.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include "libpbs.h"
#include "list_link.h"
#include "server_limits.h"
#include "attribute.h"
#include "job.h"
#include "log.h"
#include "mom_mach.h"
#include "mom_func.h"

static char ident[] = "@(#) $RCSfile: prolog.c,v $ $Revision: 2.2 $";

#define PBS_PROLOG_TIME 30

unsigned int pe_alarm_time = PBS_PROLOG_TIME;
static pid_t  child;
static int run_exit;

extern int pe_input A_((char *jobid));

/*
 * resc_to_string - convert resources_[list or used] to a single string
 */

static char *resc_to_string(pattr, buf, buflen)
	attribute *pattr;	/* the attribute to convert */
	char      *buf;		/* the buffer into which to convert */
	int	   buflen;	/* the length of the above buffer */
{
	int       need;
	svrattrl *patlist;
	list_head svlist;

	CLEAR_HEAD(svlist);
	*buf = '\0';

	if (encode_resc(pattr,&svlist,"x",(char *)0,ATR_ENCODE_CLIENT) <= 0)
		return (buf);

	patlist = (svrattrl *)GET_NEXT(svlist);
	while (patlist) {
	    need = strlen(patlist->al_resc) + strlen(patlist->al_value) + 3;
	    if (need < buflen) {
		(void)strcat(buf, patlist->al_resc);
		(void)strcat(buf,"=");
		(void)strcat(buf, patlist->al_value);
		buflen -= need;
	    }
	    patlist = (svrattrl *)GET_NEXT(patlist->al_link);
	    if (patlist)
		(void)strcat(buf, ",");
	}
	return (buf);
}

/*
 * pelog_err - record error for run_pelog()
 */

static int pelog_err(pjob, file, n, text)
	job  *pjob;
	char *file;
	int   n;
	char *text;
{
	(void)sprintf(log_buffer, "pro/epilogue failed, file: %s, exit: %d, %s\n", file, n, text);
	LOG_EVENT(PBSEVENT_ERROR, PBS_EVENTCLASS_JOB,
		  pjob->ji_qs.ji_jobid, log_buffer);
	return (n);
}

/*
 * pelogalm() - alarm handler for run_pelog()
 */

static void pelogalm(sig)
	int sig;
{
	kill(child, SIGKILL);
	run_exit = -4;
}

/*
 * run_pelog() - Run the Prologue/Epilogue script
 *
 *	Script is run under uid of root, prologue and the epilogue have:
 *		- argv[1] is the jobid
 *		- argv[2] is the user's name
 *		- argv[3] is the user's group name
 *		- the input file is a architecture dependent file
 *		- the output and error are the job's output and error
 *	The epilogue also has:
 *		- argv[4] is the job name
 *		- argv[5] is the session id
 *		- argv[6] is the list of resource limits specified
 *		- argv[7] is the list of resources used
 *		- argv[8] is the queue in which the job resides
 *		- argv[9] is the Account under which the job run
 */

int run_pelog(which, pelog, pjob, pe_io_type)
	int   which;
	char *pelog;
	job  *pjob;
	int   pe_io_type;
{
	struct sigaction act;
	char	        *arg[11];
	int		 fds1;
	int		 fds2;
	int		 fd_input;
	char		 resc_list[2048];
	char		 resc_used[2048];
	struct stat	 sbuf;
	char		 sid[20];
	int		 waitst;

	if (stat(pelog, &sbuf) == -1) {
		if (errno == ENOENT)
			return (0);
		else
			return (pelog_err(pjob, pelog, errno, "cannot stat"));
	} else {
		if ( (sbuf.st_uid != 0) ||
		     (! S_ISREG(sbuf.st_mode)) ||
		     (sbuf.st_mode & (S_IRUSR|S_IXUSR) != (S_IRUSR | S_IXUSR))||
		     (sbuf.st_mode & (S_IWGRP | S_IWOTH)) ) 
			return (pelog_err(pjob, pelog, -1, "Permission Error"));
	}
	fd_input = pe_input(pjob->ji_qs.ji_jobid);
	if (fd_input < 0)
		return (pelog_err(pjob, pelog, -2, "no pro/epilogue input file"));


	run_exit = 0;
	child = fork();
	if (child > 0) {	/* parent */
		(void)close(fd_input);
		act.sa_handler = pelogalm;
		sigemptyset(&act.sa_mask);
		act.sa_flags = 0;
		sigaction(SIGALRM, &act, 0);
		alarm(pe_alarm_time);
		while (wait(&waitst) < 0) {
			if (errno != EINTR) {
				run_exit = -3;
				break;
			}
		}
		alarm(0);
		act.sa_handler = SIG_DFL;
		sigaction(SIGALRM, &act, 0);
		if (run_exit == 0) {
			if (WIFEXITED(waitst)) {
				run_exit = WEXITSTATUS(waitst);
			}
		}
	
	} else {		/* child */
		if (fd_input != 0) {
			(void)close(0);
			(void)dup(fd_input);
			(void)close(fd_input);
		}
		if (pe_io_type == PE_IO_TYPE_NULL) {
			/* No Output, force to /dev/null */
			fds1 = open("/dev/null", O_WRONLY, 0600);
			fds2 = open("/dev/null", O_WRONLY, 0600);
		} else if (pe_io_type == PE_IO_TYPE_STD) {
			/* open job standard out/error */
			fds1 = open_std_file(pjob, StdOut, O_WRONLY | O_APPEND,
					    pjob->ji_qs.ji_un.ji_momt.ji_exgid);
			fds2 = open_std_file(pjob, StdErr, O_WRONLY | O_APPEND,
					    pjob->ji_qs.ji_un.ji_momt.ji_exgid);
		}

		if (pe_io_type != PE_IO_TYPE_ASIS) {
		/* If PE_IO_TYPE_ASIS, leave as is, already open to job */
			if (fds1 != 1) {
				(void)close(1);
				(void)dup(fds1);
				(void)close(fds1);
			}
			if (fds2 != 2) {
				(void)close(2);
				(void)dup(fds2);
				(void)close(fds2);
			}
		}

		/* for both prologue and epilogue */

		arg[0] = pelog;
		arg[1] = pjob->ji_qs.ji_jobid;
		arg[2] = pjob->ji_wattr[(int)JOB_ATR_euser].at_val.at_str;
		arg[3] = pjob->ji_wattr[(int)JOB_ATR_egroup].at_val.at_str;

		/* for epilogue only */

		if (which == PE_EPILOGUE) {
			arg[4] = pjob->ji_wattr[(int)JOB_ATR_jobname].at_val.at_str;
			(void)sprintf(sid, "%ld", pjob->ji_wattr[(int)JOB_ATR_session_id].at_val.at_long);
			arg[5] = sid;
			arg[6] = resc_to_string(&pjob->ji_wattr[(int)JOB_ATR_resource], resc_list, 2048);
			arg[7] = resc_to_string(&pjob->ji_wattr[(int)JOB_ATR_resc_used], resc_used, 2048);
			arg[8] = pjob->ji_wattr[(int)JOB_ATR_in_queue].at_val.at_str;
			arg[9] = pjob->ji_wattr[(int)JOB_ATR_account].at_val.at_str;
			arg[10] = 0;

		} else {
			arg[4] = (char *)0;
		}
			
		execv(pelog, arg);

		log_err(errno, "run_pelog", "execle of prologue failed");
		exit (255);
	}
	if (run_exit)
		(void)pelog_err(pjob, pelog,run_exit,"nonzero p/e exit status");
	return (run_exit);
}
