/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <grp.h>
#include <limits.h>
#include <pwd.h>
#include <signal.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <dirent.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include "dis.h"
#include "libpbs.h"
#include "pbs_error.h"
#include "server_limits.h"
#include "list_link.h"
#include "credential.h"
#include "attribute.h"
#include "resource.h"
#include "job.h"
#include "batch_request.h"
#include "mom_mach.h"
#include "mom_func.h"
#include "log.h"
#ifdef _CRAY
#include <sys/category.h>
#endif

static char ident[] = "@(#) $RCSfile: requests.c,v $ $Revision: 2.9 $";

/* External Global Data Items */

extern unsigned int	default_server_port;
extern int		exiting_tasks;
extern list_head	svr_alljobs;
extern char		mom_host[];
extern char		*msg_err_unlink;
extern char		*path_checkpoint;
extern char		*path_spool;
extern char		*path_undeliv;
extern attribute_def	job_attr_def[];
extern char		*msg_jobmod;
extern char		*msg_manager;
extern time_t		time_now;
extern int		resc_access_perm;	/* see encode_resc() */
						/* in attr_fn_resc.c */

/* Local Data Items */

static uid_t  useruid;
static gid_t  usergid;
static int    ngroup;
static int   *groups;
static char  *output_retained = "Output retained on that host in: ";
static char   rcperr[MAXPATHLEN];	/* file to contain rcp error */

/*
 * fork_to_user - fork mom and go to user's home directory
 *		  also sets up the global useruid and usergid in the child
 *
 *	WARNING: valid only if called when preq points to a cpyfiles structure
 */

static pid_t fork_to_user(preq)
	struct batch_request *preq;
{
	struct group   *grpp;
	pid_t		pid;
	job	       *pjob;
	struct passwd  *pwdp;
	static int	fgrp[NGROUPS_MAX];

	pid = fork_me(preq->rq_conn);
	if (pid > 0)  {
		free_br(preq);	/* parent - note leave connection open   */
		return (pid);
	} else if (pid < 0)
		return (-PBSE_SYSTEM);

	/* The Child */

	if ( (pjob = find_job(preq->rq_ind.rq_cpyfile.rq_jobid)) &&
	     (pjob->ji_grpcache != 0) ) {

		/* used the good stuff cached in the job structure */

		useruid = pjob->ji_qs.ji_un.ji_momt.ji_exuid;
		usergid = pjob->ji_qs.ji_un.ji_momt.ji_exgid;
		ngroup  = pjob->ji_grpcache->gc_ngroup;
		groups  = pjob->ji_grpcache->gc_groups;
		(void)chdir(pjob->ji_grpcache->gc_homedir);

#ifdef _CRAY
		/* set account id */
		if (pjob->ji_wattr[(int)JOB_ATR_account].at_flags & ATR_VFLAG_SET) {
			acctid(0, nam2acid(pjob->ji_wattr[(int)JOB_ATR_account].at_val.at_str));
		}
#endif  	/* _CRAY */
	} else {

		/* Need to look up the uid, gid, and home directory */

		if ((pwdp = getpwnam(preq->rq_ind.rq_cpyfile.rq_user)) ==
							(struct passwd *)0) {
			return (-PBSE_BADUSER);
		}
		useruid = pwdp->pw_uid;

		if (preq->rq_ind.rq_cpyfile.rq_group[0] == '\0') {
		    usergid = pwdp->pw_gid;	/* default to login group */
		} else {
		    if ((grpp = getgrnam(preq->rq_ind.rq_cpyfile.rq_group)) ==
							(struct group *)0) {
			return (-PBSE_BADUSER);
		    }
		    usergid = grpp->gr_gid;
		}
		ngroup  = init_groups(pwdp->pw_name, usergid, NGROUPS_MAX,fgrp);
		if (ngroup < 0)  ngroup = 0;
		groups  = fgrp;
		(void)chdir(pwdp->pw_dir); /* change to user`s home directory */
	}
	return (pid);
}

/*
 * add_bad_list -  add bad file message to bad file list
 */

static void add_bad_list(pbl, newtext, nl)
	char **pbl;
	char  *newtext;
	int    nl;
{
	int	needed = 0;
	char   *pnew;

	if (*pbl) {
		needed += strlen(*pbl) + strlen(newtext) + nl + 1;
		pnew = realloc(*pbl, needed);
	} else {
		needed += strlen(newtext) + nl + 1;
		pnew = malloc(needed);
		if (pnew)
			*pnew = '\0';
	}
	if (pnew == 0)
		return;

	*pbl = pnew;
	while (nl--)				/* prefix new-lines */
		(void)strcat(*pbl, "\n");
	(void)strcat(*pbl, newtext);
	return;
}

#define RT_BLK_SZ 4096 

static int return_file(pjob, which, sock)
	job	      *pjob;
	enum job_file  which;
	int	       sock;
{
	int		      amt;
	char		      buf[RT_BLK_SZ];
	int		      fds;
	char		     *filename;
	struct batch_request *prq;
	int		      rc = 0;
	int		      seq = 0;

	filename = std_file_name(pjob, which, &amt); /* amt is place holder */
	fds = open(filename, O_RDONLY, 0);
	if (fds < 0)
		return (0);

	prq = alloc_br(PBS_BATCH_MvJobFile);
	if (prq == (struct batch_request *)0)
		return PBSE_SYSTEM;

	(void)strcpy(prq->rq_host, mom_host);
	(void)strcpy(prq->rq_ind.rq_jobfile.rq_jobid, pjob->ji_qs.ji_jobid);

	while ( (amt = read(fds, buf, RT_BLK_SZ)) > 0 ) {
		/* prq->rq_ind.rq_jobfile.rq_sequence = seq++; */
		/* prq->rq_ind.rq_jobfile.rq_type = (int)which; */
		/* prq->rq_ind.rq_jobfile.rq_size = amt; */
		/* prq->rq_ind.rq_jobfile.rq_data = buf; */


		DIS_tcp_setup(sock);
		if ( (rc = encode_DIS_ReqHdr(sock, PBS_BATCH_MvJobFile,
					     pbs_current_user)) ||
		     (rc = encode_DIS_JobFile(sock, seq++, buf, amt,
					      pjob->ji_qs.ji_jobid, which)) ||
		     (rc = encode_DIS_ReqExtend(sock, (char *)0)) ) {
			break;
		}
		
		DIS_tcp_wflush(sock);

		if ( (DIS_reply_read(sock, &prq->rq_reply) != 0) ||
		     (prq->rq_reply.brp_code != 0) ) {
			(void)close(fds);
			rc = -1;
			break;
		}
		
	}
	free_br(prq);
	(void)close(fds);
	if (rc == 0) (void)unlink(filename);
	return (rc);
}

/*
 * wchost_match - wild card host name match
 *
 *	return	1 if can"idate" matches master name
 *		0 if not a match
 *	master name may be wild carded at beginning
 */
static wchost_match(can, master)
	const char *can;
	const char *master;
{
	const char *pc;
	const char *pm;

	pc = can + strlen(can) - 1;
	pm = master + strlen(master) - 1;
	while ((pc > can) && (pm > master)) {
		if (*pc != *pm)
			return 0;
		pc--;
		pm--;
	}

	/* comparison of one or both reached the start of the string */
	if (pm == master) {
		if (*pm == '*')
			return 1;
		else if ((pc == can) && (*pc == *pm))
			return 1;
	}
	return 0;
}

static int told_to_cp(host, oldpath, newpath)
	char  *host;
	char  *oldpath;
	char **newpath;
{
	int    i;
	int    nh;
	static char newp[MAXPATHLEN+1];
	extern struct cphosts *pcphosts;

	for (nh = 0; nh < cphosts_num; nh++) {
	    if (wchost_match(host, (pcphosts+nh)->cph_hosts)) {
		i = strlen((pcphosts+nh)->cph_from);
		if (strncmp((pcphosts+nh)->cph_from, oldpath, i) == 0) {
		    (void)strcpy(newp, (pcphosts+nh)->cph_to);
		    (void)strcat(newp, oldpath+i);
		    *newpath = newp;
		    return 1;
		}
	    }
	}
	return 0;
}

/*
 * local_or_remote() - is the specified path to a local or remote file
 *	checks to see if there is a hostname which matches this host
 *
 *	returns: 1 if remote and 0 if local
 *	also updates the path pointer to just the path name if local
 */

static int local_or_remote(path)
	char **path;
{
	int   len;
	char *pcolon;

	pcolon = strchr(*path, (int)':');
	if (pcolon == (char *)0)
		return 0;

	*pcolon = '\0';
	len = strlen(*path);
	if ( (strcmp("localhost",*path)==0)  ||
             (( strncmp(mom_host,*path,len) == 0) && 
	      ( (mom_host[len] == '\0') || (mom_host[len] == '.'))) ) {
		/* we have a host match, file is local */
		*pcolon = ':';
		*path = pcolon+1;
		return 0;
	} else if (told_to_cp(*path, pcolon+1, path)) {
		/* path updated in told_to_cp() */
		return 0;
	} else {
		/* remote file */
		*pcolon = ':';
		return 1;
	}
}

/*
 * is_file_same() - are two paths pointing to the same file
 *	returns: 1 if are the same
 *		 0 if not the same (or cannot tell)
 */

static int is_file_same(file1, file2)
	char *file1;
	char *file2;
{
	struct stat sb1, sb2;

	if ((stat(file1, &sb1) == 0) && (stat(file2, &sb2) == 0)) {

		if ( (sb1.st_dev == sb2.st_dev) && (sb1.st_ino == sb2.st_ino))
			return 1;

	}
	return 0;
}


void req_deletejob(preq)
	struct batch_request *preq;
{
	job *pjob;

	pjob = find_job(preq->rq_ind.rq_delete.rq_objname);
	if (pjob) {
		mom_deljob(pjob);
        	reply_ack(preq);
	} else {
		req_reject(PBSE_UNKJOBID, 0, preq);
	}
}

/*
 * req_holdjob - checkpoint and terminate job
 */

void req_holdjob(preq)
	struct batch_request *preq;
{
	job *pjob;
	int  rc;

#if MOM_CHECKPOINT == 1
	/* If checkpoint supported, do it and terminate the job */
	/* otherwise, return PBSE_NOSUP				*/

	pjob = find_job(preq->rq_ind.rq_hold.rq_orig.rq_objname);
	if (pjob == (job *)0) {
        	rc = PBSE_UNKJOBID;
	} else {
		if ((rc = start_checkpoint(pjob, 1, preq)) != 0)
        		req_reject(rc, 0, preq);    /* unable to start chkpt */

	}
	/* note, normally the reply to the server is in start_checkpoint() */

#else	/* MOM_CHECKPOINT */

	req_reject(PBSE_NOSUP, 0, preq);
#endif	/* MOM_CHECKPOINT */
}

/*
 *	Write text into a job's output file,
 *	Return a PBS error code.
 */

int message_job(pjob, jft, text)
    job		*pjob;
    enum	job_file	jft;
    char	*text;
{
	char		*pstr = NULL;
	int		len;
	int		fds;

	if (pjob == NULL)
		return PBSE_UNKJOBID;

	/* must be Mother Superior for this to make sence */
	if ((pjob->ji_qs.ji_svrflags & JOB_SVFLG_HERE) == 0)
		return PBSE_MOMREJECT;

	len = is_joined(pjob);
	if (len == -1)
		jft = StdErr;	/* only have stderr open */
	else if (len == 1)
		jft = StdOut;	/* only have stdout open */

	if ((fds = open_std_file(pjob, jft, O_WRONLY|O_APPEND, 
				   pjob->ji_qs.ji_un.ji_momt.ji_exgid)) < 0)
		return PBSE_MOMREJECT;

	len = strlen(text);
	if ( text[len-1] != '\n' ) {
		if ((pstr = malloc(len+2)) == NULL)
			return PBSE_INTERNAL;

		(void)strcpy(pstr, text);
		pstr[len++] = '\n';	/* append new-line */
		text = pstr;
	}
	(void)write(fds, text, len);
	(void)close(fds);
	if (pstr)
		free(pstr);
	return PBSE_NONE;
}

/*
 * req_messagejob - Append message to job's output/error file
 */

void req_messagejob(preq)
	struct batch_request *preq;
{
	int	      ret = 0;
	job	     *pjob;

	pjob = find_job(preq->rq_ind.rq_message.rq_jid);
	if ( (preq->rq_ind.rq_message.rq_file == PBS_BATCH_FileOpt_Default) ||
	     (preq->rq_ind.rq_message.rq_file & PBS_BATCH_FileOpt_OFlg)) {
		ret = message_job(pjob, StdOut,preq->rq_ind.rq_message.rq_text);
	}

	if ( (preq->rq_ind.rq_message.rq_file & PBS_BATCH_FileOpt_EFlg) &&
	     (ret == 0) ) {
		ret = message_job(pjob, StdErr,
			preq->rq_ind.rq_message.rq_text);
	}

	if (ret == PBSE_NONE)
        	reply_ack(preq);
	else
		req_reject(ret, 0, preq);
	return;
}


/*
 * req_modifyjob - service the Modify Job Request
 *
 *	This request modifys a job's attributes.
 */

void req_modifyjob(preq)
	struct batch_request *preq;
{
	int		 bad = 0;
	int		 i;
	attribute	 newattr[(int)JOB_ATR_LAST];
	attribute	*pattr;
	job		*pjob;
	svrattrl	*plist;
	int		 rc;

	 pjob = find_job(preq->rq_ind.rq_modify.rq_objname);
	if (pjob == (job *)0) {
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}

	plist = (svrattrl *)GET_NEXT(preq->rq_ind.rq_modify.rq_attr);
	if (plist == (svrattrl *)0) {	/* nothing to do */
		reply_ack(preq);
		return;
	}

	/* modify the jobs attributes */

	bad = 0;
	pattr = pjob->ji_wattr;

	/* call attr_atomic_set to decode and set a copy of the attributes */

	rc = attr_atomic_set(plist, pattr, newattr, job_attr_def, JOB_ATR_LAST,
			     -1, ATR_DFLAG_MGWR | ATR_DFLAG_MOM, &bad);
	if (rc) {
		/* leave old values, free the new ones */
		for (i=0; i<JOB_ATR_LAST; i++)
			job_attr_def[i].at_free(newattr+i);
		req_reject(rc, 0, preq);
		return;
	}

	/* OK, now copy the new values into the job attribute array */

	for (i=0; i<JOB_ATR_LAST; i++) {
		if (newattr[i].at_flags & ATR_VFLAG_MODIFY) {

			if (job_attr_def[i].at_action)  
				(void)job_attr_def[i].at_action(&newattr[i],
						        pjob, ATR_ACTION_ALTER);
			job_attr_def[i].at_free(pattr+i);
			if ((newattr[i].at_type == ATR_TYPE_LIST) ||
			    (newattr[i].at_type == ATR_TYPE_RESC)) {
				list_move(&newattr[i].at_val.at_list, 
					  &(pattr+i)->at_val.at_list);
			} else {
				*(pattr+i) = newattr[i];
			}
			(pattr+i)->at_flags = newattr[i].at_flags;
		}
	}
	/* note, the newattr[] attributes are on the stack, they goaway auto */

	if (rc == 0)
		rc = mom_set_limits(pjob, SET_LIMIT_ALTER);
	if (rc) {
		req_reject(rc, bad, preq);
		return;
	}

	(void)job_save(pjob, SAVEJOB_FULL);
	(void)sprintf(log_buffer, msg_manager, msg_jobmod,
		      preq->rq_user, preq->rq_host);
	LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, pjob->ji_qs.ji_jobid,
		  log_buffer);
	reply_ack(preq);
}

void req_shutdown(preq)
	struct batch_request *preq;
{
	req_reject(PBSE_NOSUP, 0, preq);
}

#ifdef _CRAY
/*
 * cray_susp_resum - special cray suspend/resume function
 */
static void cray_susp_resum(pjob, which, preq)
	job	*pjob;
	int	 which;
	struct batch_request *preq;
{
	int 	i;
	int	ct;
	task	*ptask;
	pid_t   pid;
	long	sess;
	int	sock;

	sock = preq->rq_conn;
	pid = fork_me(sock);
	if (pid > 0) {

		/* record pid in job for when child terminates */

		pjob->ji_momsubt = pid;
		if (which == 1) {
			pjob->ji_mompost = post_suspend;

			/* save stop time for adjusting walltime */
			pjob->ji_momstat = time_now;
		} else {
			pjob->ji_mompost = post_resume;
		}
		free_br(preq);
		return;

	} else if (pid == -1) {
		/* fork failed - still the main mom */
		req_reject(PBSE_SYSTEM, errno, preq);
		return;
	}

	/* child of MOM, cannot update job struct */

	for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
			ptask != NULL;
			ptask = (task *)GET_NEXT(ptask->ti_jobtask)) {
		sess = ptask->ti_qs.ti_sid;

		for (ct=0; ct<3; ct++)  {
			i = (which == 1) ?
				suspend(C_JOB, sess) :
				resume(C_JOB, sess);
			if (i == 0)
				break;
			if ((errno != EAGAIN) && (errno != EINTR))
				break;
		}
		if (i == -1) {	/* error */
			req_reject(PBSE_SYSTEM, errno, preq);
			exit(1);
		}
	}
	reply_ack(preq);
	exit(0);
}
#endif	/* _CRAY */

/*
 * req_signaljob - issue (kill) a specified signal to a job
 *	Signal may be either a numeric string or a signal name
 *	with or without the "SIG" prefix.
 */

void req_signaljob(preq)
	struct batch_request *preq;
{
	job  *pjob;
	int   sig;
	char *sname;
	struct sig_tbl *psigt;
	extern struct sig_tbl sig_tbl[];

	pjob = find_job(preq->rq_ind.rq_signal.rq_jid);
	if (pjob == NULL) {
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}

	sname = preq->rq_ind.rq_signal.rq_signame;

#ifdef _CRAY	/* suspend/resume on Cray only */
	if (strcmp(sname, SIG_SUSPEND) == 0) {
		if (pjob->ji_qs.ji_substate != JOB_SUBSTATE_RUNNING) {
			req_reject(PBSE_BADSTATE, 0, preq);
		} else {
			cray_susp_resum(pjob, 1, preq);
		}
		return;
	} else if (strcmp(sname, SIG_RESUME) == 0) {
		if (pjob->ji_qs.ji_substate != JOB_SUBSTATE_SUSPEND) {
			LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
				  pjob->ji_qs.ji_jobid,
				  "resume on job not suspended");
		}
		cray_susp_resum(pjob, 0, preq);
		return;
	}
#endif	/* _CRAY */

	if (isdigit((int)*sname))
		sig = atoi(sname);
	else {
		if (!strncmp("SIG", sname, 3))
			sname += 3;
		psigt = sig_tbl;
		while (psigt->sig_name) {
			if (!strcmp(sname, psigt->sig_name)) 
				break;
			psigt++;
		}
		sig = psigt->sig_val;
	}
	if (sig < 0) {
		req_reject(PBSE_UNKSIG, 0, preq);
		return;
	}
			
	if ((kill_job(pjob, sig) == 0) && (sig == 0)) {
		/* SIGNUL and no procs found, force job to exiting */
		/* force issue of (another) job obit */
		(void)sprintf(log_buffer,
			"Job recycled into exiting on SIGNULL from substate %d",
			 pjob->ji_qs.ji_substate);
		LOG_EVENT(PBSEVENT_ERROR, PBS_EVENTCLASS_JOB,
			  pjob->ji_qs.ji_jobid, log_buffer);
		pjob->ji_qs.ji_substate = JOB_SUBSTATE_EXITING;
		exiting_tasks = 1;
	}

	if ( (sig == SIGKILL) &&
		     (pjob->ji_qs.ji_substate == JOB_SUBSTATE_EXITING) ) {
		/* force issue of (another) job obit */
		(void)sprintf(log_buffer,
			"Job recycled into exiting on SIGKILL from substate %d",
			 pjob->ji_qs.ji_substate);
		LOG_EVENT(PBSEVENT_ERROR, PBS_EVENTCLASS_JOB,
			  pjob->ji_qs.ji_jobid, log_buffer);
		pjob->ji_qs.ji_substate = JOB_SUBSTATE_EXITING;
		exiting_tasks = 1;
	}
        reply_ack(preq);
	return;
}

static enum job_atr mom_rtn_list[] = {
	JOB_ATR_errpath,
	JOB_ATR_outpath,
	JOB_ATR_session_id,
	JOB_ATR_altid,
	(enum job_atr) -1
};


void
encode_used(pjob, phead)
    job		*pjob;
    list_head	*phead;
{
	unsigned long		lnum;
	int			i;
	attribute		*at;
	attribute_def		*ad;
	resource		*rs;

	at = &pjob->ji_wattr[JOB_ATR_resc_used];
	ad = &job_attr_def[JOB_ATR_resc_used];
	if ((at->at_flags & ATR_VFLAG_SET) == 0)
		return;

	for (rs = (resource *)GET_NEXT(at->at_val.at_list);
			rs != (resource *)0;
			rs = (resource *)GET_NEXT(rs->rs_link)) {

		resource_def	*rd = rs->rs_defin;
		attribute	val;
		int		rc;

		if ((rd->rs_flags & resc_access_perm) == 0)
			continue;

		val = rs->rs_value;	/* copy resource attribute */

		/* count up sisterhood too */
		lnum = 0;
		if (pjob->ji_resources != NULL) {
			if (strcmp(rd->rs_name, "cput") == 0) {
				for (i=0; i<pjob->ji_numnodes-1; i++) {
					noderes	*nr = &pjob->ji_resources[i];
					lnum += nr->nr_cput;
				}
			}
			else if (strcmp(rd->rs_name, "mem") == 0) {
				for (i=0; i<pjob->ji_numnodes-1; i++) {
					noderes	*nr = &pjob->ji_resources[i];
					lnum += nr->nr_mem;
				}
			}
		}
		val.at_val.at_long += lnum;

		rc = rd->rs_encode(&val, phead,
				ad->at_name, rd->rs_name,
				ATR_ENCODE_CLIENT);
		if (rc < 0)
			break;
	}
	return;
}

/*
 * req_stat_job - return the status of one (if id is specified) or all
 *	jobs (if id is the null string).
 */

void req_stat_job(preq)
	struct batch_request *preq;
{
	int			all;
	char			*name;
	job			*pjob;
	int			index;
	int			nth = 0;
	struct	batch_reply	*preply = &preq->rq_reply;
	struct	brp_status	*pstat;
	attribute		*at;
	attribute_def		*ad;

	/*
	 * first, validate the name of the requested object, either
	 * a single job or all jobs
	 */

	name = preq->rq_ind.rq_status.rq_id;
	if ((*name == '\0') || (*name == '@')) {
		all = 1;
		pjob = (job *)GET_NEXT(svr_alljobs);
	} else {
		all = 0;
		pjob = find_job(name);
		if (pjob == (job *)0) {
			req_reject(PBSE_UNKJOBID, 0, preq);
			return;
		}
	}

	preply->brp_choice = BATCH_REPLY_CHOICE_Status;
	CLEAR_HEAD(preply->brp_un.brp_status);

	/* pass user-client privilege to encode_resc() */
	resc_access_perm = preq->rq_perm & ATR_DFLAG_RDACC;

	for (;pjob; pjob = all ? (job *)GET_NEXT(pjob->ji_alljobs) : NULL) {
		if ((pjob->ji_qs.ji_svrflags & JOB_SVFLG_HERE) == 0)
			continue;	/* not Mohter Superior */

		if (pjob->ji_qs.ji_substate != JOB_SUBSTATE_RUNNING)
			continue;

		(void)mom_set_use(pjob);

		/* allocate reply structure and fill in header portion */
		pstat = (struct brp_status *)malloc(sizeof(struct brp_status));
		assert(pstat != (struct brp_status *)0);
		CLEAR_LINK(pstat->brp_stlink);
		pstat->brp_objtype = MGR_OBJ_JOB;
		(void)strcpy(pstat->brp_objname, pjob->ji_qs.ji_jobid);
		CLEAR_HEAD(pstat->brp_attr);
		append_link(&preply->brp_un.brp_status,
				&pstat->brp_stlink, pstat);

		/* add attributes to the status reply */
		for (index = 0; (int)mom_rtn_list[index] >= 0; ++index) {
			nth = (int)mom_rtn_list[index];
			at = &pjob->ji_wattr[nth];
			ad = &job_attr_def[nth];

			if (at->at_flags & ATR_VFLAG_MODIFY) {
				(void)ad->at_encode(at, &pstat->brp_attr,
					ad->at_name, NULL,
					ATR_ENCODE_CLIENT);

				/* turn off modify so only sent if changed */
				at->at_flags &= ~ATR_VFLAG_MODIFY;
			}
		}

		/* now do resources used */
		encode_used(pjob, &pstat->brp_attr);
	}
	reply_send(preq);
}

/*
 * del_files - delete the files in a copy files or delete files request
 *
 *	WARNING WARNING WARNING WARNING WARNING WARNING WARNING 
 *
 *	fork_to_user() must be called first so that useruid/gid is set up
 */

static int del_files(preq, pbadfile)
	struct batch_request *preq;
	char                **pbadfile;
{
	int		 asroot = 0;
	struct rqfpair  *pair;
	int		 rc = 0;
	char		 path[MAXPATHLEN+1];
	char		*pp;
	char		*prmt;
	struct stat	 sb;

	/*
	 * Should be running in the user's home directory.
	 * Build up path of file using local name only, then unlink it.
	 * The first set of files may have the STDJOBFILE
	 * flag set, which we need to unlink as root, the others as the user.
	 * This is changed from the past.  We no longer delete
	 * checkpoint files here.
	 */

	for ( pair=(struct rqfpair *)GET_NEXT(preq->rq_ind.rq_cpyfile.rq_pair);
	      pair;
	      pair = (struct rqfpair *)GET_NEXT(pair->fp_link) ) {

		prmt = pair->fp_rmt;
		path[0] = '\0';
		if (pair->fp_flag == STDJOBFILE) {

			/* the job's standard out or error */

#ifndef NO_SPOOL_OUTPUT
			(void)strcpy(path, path_spool);
#endif	/* NO_SPOOL_OUTPUT */

		} else if (asroot == 0) {
			(void)setgroups(ngroup, (gid_t *)groups);
			(void)setgid(usergid);
			(void)setuid(useruid);   /* run as the user */
			asroot = 1;
		}
	
		(void)strcat(path, pair->fp_local);

		if (local_or_remote(&prmt) == 0) {
			/* local file, is the source == destination? */
			/* if so, don't delete it		     */
			if (is_file_same(prmt, path) == 1) {
				continue;
			}
		}
		if (stat(path, &sb) == 0) {
			if (S_ISDIR(sb.st_mode)) {

				/* have a directory, must append last segment */
				/* of source name to it for  the unlink	      */

				(void)strcat(path, "/");
				pp = strrchr(prmt, (int)'/');
				if (pp) {
					++pp;
				} else if (pp = strrchr(prmt, (int)':')) {
					++pp;
				} else {
					pp = prmt;
				}
				(void)strcat(path, pp);
			}
		} else {
			(void)sprintf(log_buffer, "cannot stat %s", path);
			LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_REQUEST,
				  "del_files", log_buffer);
		}
			
		if (remtree(path) == -1) {
		    if (errno != ENOENT) {
			(void)sprintf(log_buffer,
			     "Unable to delete file %s for user %s, error = %d",
			     path, preq->rq_ind.rq_cpyfile.rq_user, errno);
			LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_REQUEST,
				  "del_files", log_buffer);
			add_bad_list(pbadfile, log_buffer, 2);
			rc = errno;
		    }
#ifdef DEBUG
		} else {
		    (void)sprintf(log_buffer, "Deleted file %s for user %s",
				  path, preq->rq_ind.rq_cpyfile.rq_user);
		    LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_FILE,
			      "del_files", log_buffer);
#endif
		}
	}
	return (rc);
}

void req_rerunjob(preq)
	struct batch_request *preq;
{
	job		*pjob;
	unsigned int	 port;
	int		 rc;
	int		 sock;
	char		*svrport;

	pjob = find_job(preq->rq_ind.rq_rerun);
	if (pjob == (job *)0) {
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}

	/* fork to send files back */

	if ( (rc = fork_me(preq->rq_conn)) > 0) {
		free_br(preq);	/* parent - note leave connection open   */
		return;
	} else if (rc < 0) {
		req_reject(-rc, 0, preq);
		return;
	}

	/* Child process ...  for each standard file generate and */
	/* send a Job Files request(s).				  */

	svrport = strchr(pjob->ji_wattr[(int)JOB_ATR_at_server].at_val.at_str, (int)':');
	if (svrport)
		port = atoi(svrport+1);
	else
		port = default_server_port;
	sock = client_to_svr(pjob->ji_qs.ji_un.ji_momt.ji_svraddr, port, 1);
	if (sock < 0) {
		LOG_EVENT(PBSEVENT_ERROR,PBS_EVENTCLASS_REQUEST,
			  "req_rerun", "no contact with the server");
		req_reject(PBSE_NOSERVER, 0, preq);
		exit(0);
	}


	if ( ((rc = return_file(pjob, StdOut, sock)) != 0) ||
	     ((rc = return_file(pjob, StdErr, sock)) != 0) ||
	     ((rc = return_file(pjob, Chkpt, sock)) != 0) ) 
		req_reject(rc, 0, preq);
	else
		reply_ack(preq);
	(void)close(sock);
	exit(0);
}

/*
 * sys_copy - issue system call to copy file
 *
 *	Check error and retry as required
 */

static int sys_copy(rmtflg, ag2, ag3, conn)
	int   rmtflg;
	char *ag2;
	char *ag3;
	int   conn;
{
	char *ag0;
	char *ag1;
	int i;
	static char *myid = "sys_copy";
	int loop;
	int rc;

	sprintf(rcperr, "%srcperr.%d", path_spool, getpid());

	for (loop = 1; loop < 5; ++loop) {

	    if (rmtflg == 0) {	/* local copy */
		ag0 = "/bin/cp";
		ag1 = "-r";
#ifdef SCP_PATH
	    } else if ((loop % 2) == 1) {	/* remote, try scp */
		ag0 = SCP_PATH;
		ag1 = "-Br";
#endif	/* SCP_PATH */
	    } else {
		ag0 = RCP_PATH;
		ag1 = "-r";
	    }

	    if ((rc = fork()) > 0) {

		/* Parent - wait for copy to complete */

		while ( ((i = wait(&rc)) < 0) && (errno == EINTR) ) ;
		if (i == -1)	{
			rc = (20000+errno);	/* 200xx is error on wait */
		} else if (WIFEXITED(rc))   {
			if ((rc = WEXITSTATUS(rc)) == 0) {
				return (rc);		/* good,  stop now */
			}
		} else if (WIFSTOPPED(rc))  {
			rc = (30000+WSTOPSIG(rc));	/* 300xx is stopped */
		} else if (WIFSIGNALED(rc)) {
			rc = (40000+WTERMSIG(rc));	/* 400xx is signaled */
		}

	    } else if (rc < 0) {

		rc = errno + 10000;	/* error on fork (100xx), retry */

	    } else {

		int	 fd;

		/* child - exec the copy command */

		rpp_terminate();
		(void)close(conn);

		/* redirect stderr to make error from rcp available to MOM */
		if ((fd = open(rcperr, O_RDWR | O_CREAT, 0644)) < 0) {
			(void)sprintf(log_buffer, "can't open %s, error = %d",
				rcperr, errno);
			LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_FILE,
					myid, log_buffer);
			exit(12);
		};
		if (fd != 2) {
			(void)dup2(fd, 2);
			(void)close(fd);
		}

		execl(ag0, ag0, ag1, ag2, ag3, (char *)0);
		(void)sprintf(log_buffer, "command: %s %s %s %s exec failed %d",
			      ag0, ag1, ag2, ag3, errno);
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_FILE,myid,log_buffer);
		exit (13);	/* 13, an unluckly number */
	    }

	    /* copy did not work, try again */

	    (void)sprintf(log_buffer, "command: %s %s %s %s status=%d, try=%d",
			  ag0, ag1, ag2, ag3, rc, loop );
	    LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_FILE,myid,log_buffer);
	    if ((loop % 2) == 0)	/* don't sleep between scp and rcp */
	    	sleep(loop/2 * 10 + 1);

	}
	return (rc);	/* tried a bunch of times, just give up */
}
			
/*
 * req_cpyfile - process the Copy Files request from the server to dispose
 *	of output from the job.  This is done by a child of MOM since it
 *	might take time.
 *
 *	The supplied PBS means of moving the file is by "rcp".
 * 	A site may wish to change this.
 */

void req_cpyfile(preq)
	struct batch_request *preq;
{
	char		 arg2[MAXPATHLEN+1];
	char		 arg3[MAXPATHLEN+1];
	int		 bad_files = 0;
	char		*bad_list = (char *)0;
	int		 dir;		
	int		 from_spool;
	int		 len;
	char		 localname[MAXPATHLEN+1];  /* used only for in-bound */
	struct rqfpair  *pair;
	char		*prmt;
	int		 rc;
	int		 rmtflag;
	char		 undelname[MAXPATHLEN+1];
#ifdef  _CRAY
	char		 tmpdirname[MAXPATHLEN+1];
#endif 	/* _CRAY */


	if ((rc = (int)fork_to_user(preq)) > 0) {
		return;		/* parent - continue with someother task */
	} else if (rc < 0) {
		req_reject(-rc, 0, preq);
		return;
	}

	/*
	 * Child process ...
	 * Now running in the user's home directory.
	 * Become the user and build up cp/rcp command(s), one per file pair
	 */

	(void)setgroups(ngroup, (gid_t *)groups);
	(void)setgid(usergid);     /* run as the group */
	(void)setuid(useruid);     /* run as the user  */
	dir  = preq->rq_ind.rq_cpyfile.rq_dir;

	for ( pair=(struct rqfpair *)GET_NEXT(preq->rq_ind.rq_cpyfile.rq_pair);
	      pair != 0;
	      pair = (struct rqfpair *)GET_NEXT(pair->fp_link) ) {

		from_spool = 0;
		prmt   = pair->fp_rmt;
			
		if (local_or_remote(&prmt) == 0) {
			/* destination host is this host, use cp */
			rmtflag = 0;
		} else {
			/* destination host is another, use (pbs_)rcp */
			rmtflag = 1;
		}

		/*
		 * Which way to copy, In or Out ...
		 */

		if (dir == STAGE_DIR_OUT) {

			/*
			 * out bound copy ...
			 * build "from" path name, local to this system
			 */

			localname[0] = '\0';
			if (pair->fp_flag == STDJOBFILE) {

#ifndef NO_SPOOL_OUTPUT
				/* stdout | stderr from MOM's spool area */

				(void)strcpy(localname, path_spool);
				from_spool = 1;	/* flag as being in spool dir */
				/*
				 * note, if NO_SPOOL_OUTPUT is defined, the
				 * output is in the user's home directory where
				 * we currently are.
				 */
#endif	/* NO_SPOOL_OUTPUT */

#if MOM_CHECKPOINT == 1
			} else if (pair->fp_flag == JOBCKPFILE) {
				(void)strcpy(localname, path_checkpoint);
#endif	/* MOM_CHECKPOINT */
			} 
			(void)strcat(localname, pair->fp_local);

#if SRFS
			/* Is this file part of $BIGDIR or $FASTDIR ? */

			if (!strncmp(localname,"/BIGDIR",7)) {
				sprintf(tmpname,"%s/%s",
					tmpdirname(var_value("BIGDIR",
					    preq->rq_ind.rq_cpyfile.rq_jobid)),
					&localname[7]);
				strcpy(localname,tmpname);
			} else if (!strncmp(localname,"/FASTDIR",8)) {
				sprintf(tmpname,"%s/%s",
					tmpdirname(var_value("BIGDIR",
					     preq->rq_ind.rq_cpyfile.rq_jobid)),
					&localname[8]);
				strcpy(localname,tmpname);
			}
#endif	/* SRFS */


			/* Is the file there?  If not, don`t trying copy*/

			if (access(localname, F_OK|R_OK) < 0)
			    if (errno == ENOENT) {
				continue;
			    }

			(void)strcpy(arg2, localname);

			/* take (remote) destination name from request */

			arg3[0] = '\0';
			if (rmtflag) {
				/* using rcp, need to prepend the owner name */
				strcat(arg3,preq->rq_ind.rq_cpyfile.rq_owner);
				strcat(arg3, "@");
			}
			(void)strcat(arg3, prmt);


		} else {	/* in bound (stage-in) file */

			/* take (remote) source name from request */

			arg2[0] = '\0';
			if (rmtflag) {
				/* using rcp, need to prepend the owner name */
				strcat(arg2,preq->rq_ind.rq_cpyfile.rq_owner);
				strcat(arg2, "@");
			}
			(void)strcat(arg2, prmt);

			(void)strcpy(arg3, pair->fp_local);
		}

		if ( (rmtflag == 0) && (is_file_same(arg2, arg3) == 1) ) {
			/* local file, source == destination, don't copy */
			continue;
		}

		if ((rc = sys_copy(rmtflag,arg2,arg3, preq->rq_conn)) != 0) {
 		    FILE	*fp;

		    /* copy failed */
		    bad_files = 1;
		    (void)sprintf(log_buffer,"Unable to copy file %s %s %s",
				  pair->fp_local,
				  (dir == STAGE_DIR_IN) ? "from" : "to",
				  pair->fp_rmt);
		    add_bad_list(&bad_list, log_buffer, 2);
		    log_record(PBSEVENT_ADMIN, PBS_EVENTCLASS_FILE,
			       pair->fp_local, log_buffer);

 		    /* copy message from rcp as well */
 		    if ((fp = fopen(rcperr, "r")) != NULL) {
			add_bad_list(&bad_list, ">>> error from copy", 1);
 		    	while (fgets(log_buffer, LOG_BUF_SIZE, fp) != NULL) {
				len = strlen(log_buffer) - 1;

				if (log_buffer[len] == '\n')
					log_buffer[len] = '\0';
				add_bad_list(&bad_list, log_buffer, 1);
				log_record(PBSEVENT_ADMIN, PBS_EVENTCLASS_FILE,
					pair->fp_local, log_buffer);
			}
 		    	fclose(fp);
			add_bad_list(&bad_list, ">>> end error output", 1);
 		    }
		
		    if (dir == STAGE_DIR_IN) {

			/* delete the stage_in files that were just copied in */

			(void)del_files(preq, &bad_list);
#ifndef NO_SPOOL_OUTPUT
		    } else if (from_spool == 1) {	/* copy out of spool */

			/* Copying out files and in spool area ...	*/
			/* move to "undelivered" directory		*/

			(void)strcpy(localname, path_spool);
			(void)strcat(localname, pair->fp_local);
			(void)strcpy(undelname, path_undeliv);
			(void)strcat(undelname, pair->fp_local);
			if (rename(localname, undelname) == 0) {
			    add_bad_list(&bad_list, output_retained, 1);
			    add_bad_list(&bad_list, undelname, 0);

			} else {
			    (void)sprintf(arg3, "Unable to rename %s to %s",
					  localname, undelname);
			    log_err(errno, "req_cpyfile", arg3);
			}
#endif	/* NO_SPOOL_OUTPUT */
		    }

		    if (dir == STAGE_DIR_IN) {
			(void)unlink(rcperr);
			break;
		    }
		} else {

		    /* Copy in/out succeed */

		    if (dir == STAGE_DIR_OUT) {
			/* have copied out, ok to remove local one */

			if (remtree(localname) < 0) {
				(void)sprintf(arg3, msg_err_unlink, "stage out",
					      localname);
				log_err(errno, "req_cpyfile", arg3);
				add_bad_list(&bad_list, arg3, 2);
				bad_files = 1;
			}
		    }
		}
		(void)unlink(rcperr);
	}
	if (bad_files) {
	    reply_text(preq, PBSE_NOCOPYFILE, bad_list);
	} else {
	    reply_ack(preq);
	}
	exit(0);	/* remember, we are the child, exit not return */
}

/*
 * req_delfile - delete the specifled output/staged files 
 */

void req_delfile(preq)
	struct batch_request *preq;
{
	int		 rc;
	char		*bad_list = (char *)0;

	if ((rc = (int)fork_to_user(preq)) > 0) {
		return;		/* parent - continue with someother task */
	} else if (rc < 0) {
		req_reject(-rc, 0, preq);
		return;
	}

	/* Child process ... delete the files */

	if (rc = del_files(preq, &bad_list)) {
		reply_text(preq, rc, bad_list);
	} else {
		reply_ack(preq);
	}
	exit(0);	/* remember, we are the child, exit not return */
}

#if MOM_CHECKPOINT == 1
/*
 * Checkpoint the job.
 *
 *	If abort is TRUE, kill it too.  Return a PBS error code.
 */

int mom_checkpoint_job(pjob, abort)
    job		*pjob;
    int		abort;
{
	static char	id[] = "mom_checkpoint_job";
	int		hasold = 0;
	int		sesid = -1;
	int		ckerr;
	int		i;
	struct stat	statbuf;
	char		path[MAXPATHLEN+1];
	char		oldp[MAXPATHLEN+1];
	char		file[MAXPATHLEN+1], *name;
	int		filelen;
	task		*ptask;
	extern	char	task_fmt[];

	assert(pjob != NULL);

	strcpy(path, path_checkpoint);
	strcat(path, pjob->ji_qs.ji_fileprefix);
	strcat(path, JOB_CKPT_SUFFIX);

	if (stat(path, &statbuf) == 0) {
		(void)strcpy(oldp, path);   /* file already exists, rename it */
		(void)strcat(oldp, ".old");
		if (rename(path, oldp) < 0)
			return (errno);
		hasold = 1;
	}
	if (mkdir(path, 0755) == -1)
		goto fail;

	filelen = strlen(path);
	strcpy(file, path);
	name = &file[filelen];

#ifdef	_CRAY
	/*
	 * if job is suspended and if <abort> is set, resume job first,
	 * this is so job will be "Q"ueued and then back into "R"unning
	 * when restarted.
	 */
	if ((pjob->ji_qs.ji_svrflags & JOB_SVFLG_Suspend) && abort) {
		for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
				ptask != NULL;
				ptask = (task *)GET_NEXT(ptask->ti_jobtask)) {
			sesid = ptask->ti_qs.ti_sid;
			if (ptask->ti_qs.ti_status != TI_STATE_RUNNING)
				continue;

			/*
			** What to do if some resume's work and others don't?
			*/
			if ((ckerr = resume(C_JOB, sesid)) == 0) {
				post_resume(pjob, ckerr);
	    		} else {
				sprintf(log_buffer,
					"chkpnt failed: errno=%d sid=%d",
					errno, sesid);
				LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
					pjob->ji_qs.ji_jobid, log_buffer);
				return (errno);
			}
		}
	}
#endif	/* _CRAY */

	for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
			ptask != NULL;
			ptask = (task *)GET_NEXT(ptask->ti_jobtask)) {

		sesid = ptask->ti_qs.ti_sid;
		if (ptask->ti_qs.ti_status != TI_STATE_RUNNING)
			continue;
		sprintf(name, task_fmt, ptask->ti_qs.ti_task);
		if (mach_checkpoint(ptask, file, abort) == -1)
			goto fail;
	}

	/* Checkpoint successful */

	(void)job_save(pjob, SAVEJOB_FULL);  /* to save resources_used so far */
	sprintf(log_buffer, "checkpointed to %s", path);
	log_record(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
		   pjob->ji_qs.ji_jobid, log_buffer);
	if (hasold) 
		(void)remtree(oldp);

       	return (PBSE_NONE);

 fail:
	/*
	** A checkpoint has failed.  Log and return error.
	*/
	ckerr = errno;
	sprintf(log_buffer,"chkpnt failed:errno=%d sid=%d", errno, sesid);
	LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,pjob->ji_qs.ji_jobid,
		  log_buffer);

	/*
	** See if any checkpoints worked and abort is set.
	** If so, we need to restart these tasks so the whole job is
	** still running.  This has to wait until we reap the
	** aborted task(s).
	*/
	if (abort)
		return PBSE_CKPSHORT;

	/*
	** Clean up files.
	*/
	(void)remtree(path);
	if (hasold) {
		if (rename(oldp, path) == -1)
			pjob->ji_qs.ji_svrflags &= ~JOB_SVFLG_CHKPT;
	}
	if (ckerr == EAGAIN)
		return (PBSE_CKPBSY);
	else
		return(ckerr);
}

/* 
 * post_chkpt - post processor for start_checkpoint()
 *
 *	Called from scan_for_terminated() when found in ji_mompost;
 *	This sets the "has checkpoint image" bit in the job.
 */

void post_chkpt(pjob, ev)
	job *pjob;
	int  ev;
{
	char		path[MAXPATHLEN+1];
	DIR		*dir;
	struct	dirent	*pdir;
	extern	char	*path_checkpoint;
	tm_task_id	tid;
	task		*ptask;
	int		abort = pjob->ji_flags & MOM_CHKPT_ACTIVE;

	exiting_tasks = 1;	/* make sure we call scan_for_exiting() */
	pjob->ji_flags &= ~MOM_CHKPT_ACTIVE;
	if (ev == 0) {
		pjob->ji_qs.ji_svrflags |= JOB_SVFLG_CHKPT;
		return;
	}

	/*
	** If we get here, an error happened.  Only try to recover
	** if we had abort set.
	*/
	if (abort == 0)
		return;
	/*
	** Set a flag for scan_for_exiting() to be able to
	** deal with a failed checkpoint rather than doing
	** the usual processing.
	*/
	pjob->ji_flags |= MOM_CHKPT_POST;

	/*
	** Set the TI_FLAGS_CHKPT flag for each task that
	** was checkpointed and aborted.
	*/
	strcpy(path, path_checkpoint);
	strcat(path, pjob->ji_qs.ji_fileprefix);
	strcat(path, JOB_CKPT_SUFFIX);

	dir = opendir(path);
	if (dir == NULL)
		return;
	while ((pdir = readdir(dir)) != NULL) {
		if (pdir->d_name[0] == '.')
			continue;
		tid = atoi(pdir->d_name);
		if (tid == 0)
			continue;
		ptask = task_find(pjob, tid);
		if (ptask == NULL)
			continue;
		ptask->ti_flags |= TI_FLAGS_CHKPT;
	}
	closedir(dir);
	return;
}

/*
 * start_checkpoint - start a checkpoint going
 *
 *	checkpoint done from a child because it takes a while 
 */

int start_checkpoint(pjob, abort, preq)
	job *pjob;
	int  abort;
	struct batch_request *preq;	/* may be null */
{
	svrattrl *pal;
	pid_t     pid;
	int       rc;
	int       sock = -1;
	attribute tmph;

	if (mom_does_chkpnt() == 0) 	/* no checkpoint, reject request */
		return(PBSE_NOSUP);

	/* now set up as child of MOM */
	if (preq)
		sock = preq->rq_conn;
	pid = fork_me(sock);
	if (pid > 0) {

		/* parent, record pid in job for when child terminates */

		pjob->ji_momsubt = pid;
		pjob->ji_mompost = post_chkpt;
		if (preq)
			free_br(preq);
		/*
		** If we are going to have tasks dieing, set a flag.
		*/
		if (abort)
			pjob->ji_flags |= MOM_CHKPT_ACTIVE;

	} else if (pid < 0) {
		return (PBSE_SYSTEM);		/* error on fork */
	} else {
		/* child - does the checkpoint */
		int	  hok = 1;

		clear_attr(&tmph, &job_attr_def[(int)JOB_ATR_hold]);
		if (preq) {
			pal = (svrattrl *)
				GET_NEXT(preq->rq_ind.rq_hold.rq_orig.rq_attr);
			if (pal) {
				hok = job_attr_def[(int)JOB_ATR_hold].at_decode(
							&tmph,
							pal->al_name,
							(char *)0,
							pal->al_value);
			}
		}
		rc = mom_checkpoint_job(pjob, abort);
		if (preq != (struct batch_request *)0) {
			/*
			** rc may be 0, req_reject is used
			** to pass auxcode
			*/
			req_reject(rc, PBS_CHKPT_MIGRATE, preq);
		}
		if ( (rc == 0) && (hok == 0) )
			rc = site_mom_postchk(pjob, (int)tmph.at_val.at_long);

		exit(rc);	/* zero exit tells main chkpnt ok */
	}
	return (0);		/* parent return */
}
#endif	/* MOM_CHECKPOINT */
