/*
This program is confidential and proprietary to MRJ Technology Solutions
and may not be reproduced, published or disclosed to others without
written authorization from MRJ.

Copyright (c) 1998 MRJ Technology Solutions, Inc. All Rights Reserved.

This program is derived from prior work some of which was originally
written as a joint project between the Numerical Aerospace Simulation
(NAS) Systems Division of NASA Ames Research Center and the National
Energy Research Supercomputer Center (NERSC) of Lawrence Livermore
National Laboratory.
*/
#include <pbs_config.h>   /* the master config generated by configure */

#include <assert.h>
#include <limits.h>
#include <stdio.h>
#include <stdlib.h>
#include <stddef.h>
#include <unistd.h>
#include <ctype.h>
#include <procinfo.h>
#include <jm_client.h>
#include <errno.h>
#include <fcntl.h>
#include <pwd.h>
#include <strings.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/param.h>
#include <sys/resource.h>
#include <sys/uio.h>
#include <sys/file.h>
#include <sys/stat.h>
#include <sys/statfs.h>
#include <sys/user.h>
#include <sys/proc.h>
#include <sys/stream.h>
#include <sys/stropts.h>

#include "pbs_error.h"
#include "portability.h"
#include "list_link.h"
#include "server_limits.h"
#include "attribute.h"
#include "resource.h"
#include "job.h"
#include "log.h"
#include "mom_mach.h"
#include "pbs_nodes.h"
#include "resmon.h"
#include "../rm_dep.h"


/*
**	System dependent code to gather information for the resource
**	monitor for an IBM sp-2 running AIX 4.
**
**	Resources not supported by this code:
**		sessions	list of sessions in the system
**		pids		list of pids in a session
**		ncpus		number of cpus
**		nsessions	number of sessions in the system
**		nusers		number of users in the system
**		jnodes		number of nodes used by a session
**		size		size of a file or filesystem in KB
**		walltime	wall clock time for a pid
**		loadave		current load average
**
*/

static char ident[] = "@(#) sp2/$RCSfile: mom_mach.c,v $ $Revision: 2.2 $";

#ifndef TRUE
#define FALSE	0
#define TRUE	1
#endif	/* TRUE */

struct	JM_JOB_STATUS		*job_tbl = NULL;
static	struct	procsinfo	*proc_tbl = NULL;
static	int			nproc = 0;
static	int			njob = -1;
int				proctot = 0;

extern	struct	pbs_err_to_txt	pbs_err_to_txt[];
extern	time_t			time_now;
char *	nodelist	A_(( char * ));

/*
** external functions and data
*/
extern	struct	config		*search A_((struct config *, char *));
extern	struct	rm_attribute	*getattr A_((char *));
extern	int			rm_errno;
extern	unsigned	int	reqnum;
extern	double	wallfactor;

/*
** local functions
*/
char	*ncpus		A_((struct rm_attribute *attrib));
char	*jnodes		A_((struct rm_attribute *attrib));
char	*walltime	A_((struct rm_attribute *attrib));

extern char *loadave	A_((struct rm_attribute *attrib));

/*
** local resource list storage
*/
struct	config	dependent_config[] = {
	{ "ncpus",	ncpus },
	{ "jnodes",	jnodes },
	{ "walltime",	walltime },
	{ "loadave",	loadave },
	{ NULL,		nullproc },
};

struct nlist nl[] = {
	{ "avenrun" }
};

#define ASIZE		10
#define	KSYM_LOAD	0

time_t			wait_time = 10;
int			kd = -1;
static	int		sock = -1;

extern	char	*ret_string;
extern	char	extra_parm[];
extern	char	no_parm[];
char		nokernel[] = "kernel not available";
char		noproc[] = "process %d does not exist";

void
dep_initialize()
{
	char	*id = "dep_initialize";
	int	i, rc;

	log_record(PBSEVENT_SYSTEM, 0, id, "dependent");

	if ((kd = open("/dev/kmem", O_RDONLY)) == -1) {
		log_err(errno, id, "open");
		return;
	}

	proc_tbl = malloc(ASIZE*sizeof(struct procsinfo));
	proctot = ASIZE;

	rc = knlist(nl, sizeof(nl)/sizeof(struct nlist), sizeof(struct nlist));
	if (rc == -1) {
		log_err(errno, id, "knlist");
		return;
	}
#ifdef	DEBUG
	for (i=0; i<sizeof(nl)/sizeof(struct nlist); i++)
		printf("%s: %s @ %x\n", id, nl[i].n_name, nl[i].n_value);
#endif
	return;
}

void
dep_cleanup()
{
	char	*id = "dep_cleanup";

	log_record(PBSEVENT_SYSTEM, 0, id, "dependent");

	if (close(kd) == -1)
		log_err(errno, id, "close");
	kd = -1;

	if (proc_tbl) {
		free(proc_tbl);
		proc_tbl = NULL;
	}

	return;
}

void
end_proc()
{
	return;
}

/*
 * Internal long decoding routine.
 *
 *	Accepts a resource pointer and a pointer to the unsigned long integer
 *	to receive the decoded value.  It returns a PBS error code, and the
 *	decoded value.  Time is in seconds in the unsigned long integer.
 */

static int getnum(pres, ret)
    resource		*pres;
    unsigned long	*ret;
{

	if (pres->rs_value.at_type != ATR_TYPE_LONG)
		return (PBSE_ATTRTYPE);
	if (pres->rs_value.at_val.at_long < 0)
	        return (PBSE_BADATVAL);
	*ret = pres->rs_value.at_val.at_long;

	return (PBSE_NONE);
}

/*
**	Check that node_name matches one of the names in node_list
**	separated by '+'s.  The match is only done up to the first
**	'.' character.
**	Return 1 for a match and 0 for no match.
*/
int
match(node_name, node_list)
    char	*node_name;
    char	*node_list;
{
	char		*cp, *name;

	/*
	** start a loop through "node_list"
	** making one pass only
	*/
	name = node_name;
	cp = node_list;
	while (*cp) {
		/*
		** step through both name and
		** nodes matching each char
		*/
		for (; *name; name++, cp++) {
			if (*name != *cp)
				break;
		}
		/*
		** once we are done with the
		** test loop above, check
		** to see if a match is found
		** by having gotten to the
		** end of name and the end
		** of one node name in "node_list"
		*/
		if ((*name == '\0' || *name == '.') &&
		    (*cp == '+' || *cp == '\0' || *cp == '.'))
			return 1;

		/*
		** if we get here, there was no match
		** skip the rest of the node name
		** (and plus sign) and reset name to
		** the one we are looking for
		*/
		while (*cp != '+' && *cp != '\0')
			cp++;
		if (*cp == '+')
			cp++;
		name = node_name;
	}
	return 0;
}

static
int
injob(pjob, sesid)
    job			*pjob;
    pid_t		sesid;
{
        task		*ptask;

	for (ptask = (task *)GET_NEXT(pjob->ji_tasks);
			ptask;
			ptask = (task *)GET_NEXT(ptask->ti_jobtask)) {
		if (ptask->ti_qs.ti_sid <= 1)
			continue;
		if (ptask->ti_qs.ti_sid == sesid)
			return TRUE;
	}
	return FALSE;
}

/*
 * Internal session number of cpu decoding routine.
 *
 *      Accepts a job pointer.  Returns TRUE if the nodes used
 *      by any session fall outside those allowed.
 */
static unsigned long nodes_ses(pjob)
    job		*pjob;
{
	char		*id = "nodes_ses";
	int		i, j, k;
	resource	*pres;
	attribute	*at;
	resource_def	*rd;
	char		*nodes;
	char		*badnodes = "";

	at = &pjob->ji_wattr[(int)JOB_ATR_resc_used];
	assert(at->at_type == ATR_TYPE_RESC);

	DBPRT(("%s: entered\n", id))

	rd = find_resc_def(svr_resc_def, "nodes", svr_resc_size);
	assert(rd != NULL);
	pres = find_resc_entry(at, rd);

	/*
	** The variable "nodes" is the string of plus sign separated
	** node names saved earlier in set_use() or the empty string.
	** We want to check the nodes used by any process in the job
	** to these and return TRUE if they are not a subset.
	*/
	nodes = pres ? pres->rs_value.at_val.at_str : "";
	for (i=0; i<nproc; i++) {
		register struct procsinfo	*pp = &proc_tbl[i];

		if (pp->pi_state == SNONE)
			continue;

		if (!injob(pjob, pp->pi_sid))
			continue;

		/*
		** found a process in the job
		** loop to see if any JM job shows this
		** proc as its client pid
		*/
		DBPRT(("%s: pid=%d\n", id, pp->pi_pid))
		for (j=0; j<njob; j++) {
			struct	JM_JOB_STATUS	*jp = &job_tbl[j];
			int	len, hit;
			char	*end;

			if (jp->jm_client_pid != pp->pi_pid)
				continue;

			/*
			** the plot thickens
			** a JM job has been found to be part of this session
			** check to see if the nodes associated with
			** the job are shown in "nodes"
			*/
			DBPRT(("%s: job pid %d nodes %d\n",
				id, pp->pi_pid, jp->jm_num_nodes))
			hit = 0;
			for (k=0; k<jp->jm_num_nodes; k++) {
				struct	JM_NODE_IN_JOB
					*np = &jp->jm_node_in_job[k];

				/*
				** if we find a match, everything is okay,
				** the job is staying within its limits
				*/
				if (match(np->jm_node_name, nodes))
					continue;
				/*
				** if we get here without a match
				** a node being used in the actual job
				** was compared to every node this
				** job is allowed to use (nodes)
				** and it was not in the set
				*/
				hit = 1;

				/*
				** see if this node is already listed
				** in the "badnodes" list
				*/
				if (match(np->jm_node_name, badnodes))
					continue;

				/*
				** not in the "badnodes" list so we need
				** to add it
				*/
				if ((len = strlen(badnodes)) == 0) {
					badnodes = strdup(np->jm_node_name);
					continue;
				}

				badnodes = realloc(badnodes,
					len + strlen(np->jm_node_name) + 4);
				strcat(badnodes, "+");
				strcat(badnodes, np->jm_node_name);
			}
			if (hit == 0)
				continue;

			sprintf(log_buffer, "rouge pid %d using node(s) %s",
				jp->jm_client_pid,
				jp->jm_node_in_job[0].jm_node_name);
			for (k=1; k<jp->jm_num_nodes; k++) {
				struct	JM_NODE_IN_JOB
					*np = &jp->jm_node_in_job[k];
				int	n = strlen(log_buffer);

				end = &log_buffer[n];
				sprintf(end, "+%s", np->jm_node_name);
			}

			log_record(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
				pjob->ji_qs.ji_jobid, log_buffer);
		}
	}

	if (strlen(badnodes)) {
		sprintf(log_buffer,
			"node(s) %.450s outside allowed list %.450s",
				badnodes, nodes);
		DBPRT(("%s: %s\n", id, log_buffer))
		free(badnodes);
		return(TRUE);
	}
	else
		return (FALSE);
}

extern char *msg_momsetlim;

/*
 * Internal error routine
 */
int error(string, value)
    char	*string;
    int		value;
{
	int		i = 0;
	char		*message;
	extern	FILE	*nuse;

	assert(string != NULL);
	assert(*string != '\0');
	assert(value > PBSE_);			/* minimum PBS error number */
	assert(value <= PBSE_NOSYNCMSTR);	/* maximum PBS error number */
	assert(pbs_err_to_txt[i].err_no != 0);

	do {
		if (pbs_err_to_txt[i].err_no == value)
			break;
	} while (pbs_err_to_txt[++i].err_no != 0);

	assert(pbs_err_to_txt[i].err_txt != NULL);
	message = *pbs_err_to_txt[i].err_txt;
	assert(message != NULL);
	assert(*message != '\0');
	(void)fprintf(stderr, msg_momsetlim, string, message);
	(void)fflush(stderr);
	fclose(nuse);
	nuse = NULL;

	return (value);
}

/*
 * Establish system-enforced limits for the job.
 *
 *	Run through the resource list, checking the values for all items
 *	we recognize.
 *
 *	If set_mode is SET_LIMIT_SET, then also set hard limits for the
 *	system enforced limits (not-polled).
 *	If anything goes wrong with the process, return a PBS error code
 *	and print a message on standard error.  A zero-length resource list
 *	is not an error.
 *
 *	If set_mode is SET_LIMIT_SET the entry conditions are:
 *	    1.	MOM has already forked, and we are called from the child.
 *	    2.	The child is still running as root.
 *	    3.  Standard error is open to the user's file.
 *
 *	If set_mode is SET_LIMIT_ALTER, we are beening called to modify
 *	existing limits.  Cannot alter those set by setrlimit (kernel)
 *	because we are the wrong process.  
 */
int mom_set_limits(pjob, set_mode)
    job			*pjob;
    int			 set_mode;	/* SET_LIMIT_SET or SET_LIMIT_ALTER */
{
	char		*id = "mom_set_limits";
	char		*pname;
	int		retval;
	unsigned long	value;	/* place in which to build resource value */
	resource	*pres;

	DBPRT(("%s: entered\n", id))
	assert(pjob != NULL);
	assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);
	pres = (resource *)
	    GET_NEXT(pjob->ji_wattr[(int)JOB_ATR_resource].at_val.at_list);

/*
 * Cycle through all the resource specifications,
 * setting limits appropriately.
 */

	while (pres != NULL) {
		assert(pres->rs_defin != NULL);
		pname = pres->rs_defin->rs_name;
		assert(pname != NULL);
		assert(*pname != '\0');

		if (strcmp(pname, "walltime") == 0) {		/* Check */
			retval = getnum(pres, &value);
			if (retval != PBSE_NONE)
			        return (error(pname, retval));
			DBPRT(("%s: found walltime %d\n", id, value))
		} else if (strcmp(pname, "neednodes") == 0) {
			/*
			** nothing to to here since nodes has
			** been set before the prolog
			*/
		} else if (strcmp(pname, "nice") == 0) {	/* set nice */
		    if (set_mode == SET_LIMIT_SET)  {
			if (nice((int)pres->rs_value.at_val.at_long) == -1)
				return (error(pname, PBSE_BADATVAL));
			DBPRT(("%s: found nice %d\n",
				id, (int)pres->rs_value.at_val.at_long))
		    }
		} else if ((pres->rs_defin->rs_flags & ATR_DFLAG_RMOMIG) == 0) 
			/* don't recognize and not marked as ignore by mom */
			return (error(pname, PBSE_UNKRESC));
		pres = (resource *)GET_NEXT(pres->rs_link);
	}

	return (PBSE_NONE);
}

/*
 * State whether MOM main loop has to poll this job to determine if some
 * limits are being exceeded.
 *
 *	Sets flag TRUE if polling is necessary, FALSE otherwise.  Actual
 *	polling is done using the mom_over_limit machine-dependent function.
 *	For the SP-2, we always want to poll to make sure users are not
 *	running a job which uses nodes outside our control.
 */

int mom_do_poll(pjob)
    job			*pjob;
{
	char		*id = "mom_do_poll";
	char		*pname;
	resource	*pres;

	DBPRT(("%s: entered\n", id))
	assert(pjob != NULL);
	assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);

	return (TRUE);
}

/*
 * Setup for polling.
 */
int mom_open_poll()
{
	char		*id = "mom_open_poll";
	struct	pbsnode	*np;

	DBPRT(("%s: entered\n", id))

	proc_tbl = malloc(ASIZE*sizeof(struct procsinfo));
	proctot = ASIZE;

	return (PBSE_NONE);
}

/*
**	Get process table.
**	Return PBS error code.
*/
int
getprocstat()
{
	char		*id = "getprocstat";
	struct	procsinfo	*pp;
	int		num, addnum;
	pid_t		pid;

	DBPRT(("%s: entered\n", id))

	addnum = proctot;
	nproc = 0;
	pid = 0;
	pp = proc_tbl;

	while ((num = getprocs(pp, sizeof(struct procsinfo),
			NULL, sizeof(struct fdsinfo),
			&pid, addnum)) > 0) {
		DBPRT(("%s: loop start: got %d\n", id, num))

		nproc += num;
		if (num < addnum)
			break;

		proctot += ASIZE;
		addnum = ASIZE;
		proc_tbl = realloc(proc_tbl, proctot*sizeof(struct procsinfo));
		pp = &proc_tbl[nproc];
	}
	if (num == -1) {
		log_err(errno, id, "getprocs");
		return PBSE_SYSTEM;
	}
	DBPRT(("%s: nproc = %d\n", id, nproc))

	return (PBSE_NONE);
}

int
rm_getproc()
{
	static	uint	lastproc = 0;


	if (lastproc == reqnum)		/* don't need new proc table */
		return nproc;

	if (getprocstat() != PBSE_NONE)
		return 0;

	lastproc = reqnum;
	return nproc;
}

#define       JMRETRY 2
/*
**	Get job manager info.
**	Return the number of jobs or -1 on error;
*/
int
getjobstat()
{
	char	*id = "getjobstat";
	int	cnt;
	int	sock;

	if (job_tbl)
		jmq_jobs_free(&job_tbl, njob);

	njob = 0;
	for (cnt=0; cnt < JMRETRY; cnt++) {
		if ((sock = jm_connect_ub(NULL)) >= 0)
			break;

		switch (errno) {
		case EINTR:
			break;
		default:
			log_err(errno, id, "jm_connect_ub failed");
			return -1;
		}
	}
	if (cnt == JMRETRY) {
		log_err(errno, id, "jm_connect_ub retry exhausted");
		return -1;
	}

	for (cnt=0; cnt < JMRETRY; cnt++) {
		DBPRT(("%s: jm socket %d cnt %d\n", id, sock, cnt))
		if ((njob = jmq_jobs_status(sock, &job_tbl)) >= 0)
			break;

		sprintf(log_buffer, "jmq_jobs_status failed %d", njob);
		log_err(errno, id, log_buffer);
		log_buffer[0] = '\0';
	}
	jm_disconnect(sock);
/*	SDRCloseSession(); */
#ifdef	DEBUG
{
	int		i, j, k;

	for (i=0; i<njob; i++) {
		struct  JM_JOB_STATUS   *jp = &job_tbl[i];

		printf("-------------------------------\n");
		printf("name: %s\n", jp->jm_user_name);
		printf("desc: %s\n", jp->jm_job_description);
		printf("%s pid=%d id=%d type=%d jnodes=%d\n",
			jp->jm_time_allocated,
			jp->jm_client_pid,
			jp->jm_job_id,
			jp->jm_adapter_type,
			jp->jm_num_nodes);
		for (j=0; j<jp->jm_num_nodes; j++) {
			struct JM_NODE_IN_JOB *np = &jp->jm_node_in_job[j];

			printf("\t+++++++++++++++++++\n");
			printf("\tnode %s\n", np->jm_node_name);
			printf("\tusage=(%d/%d) tasks=%d\n\t",
				np->jm_cpu_usage,
				np->jm_adapter_usage,
				np->jm_num_virtual_tasks);
			for (k=0; k<np->jm_num_virtual_tasks; k++) {
				int     vid = np->jm_virtual_task_ids[k];

				printf("(%d) ", vid);
			}
			printf("\n");
		}
	}
}
#endif
	return njob;
}

static
int
node_cmp(s1, s2)
    char	*s1, *s2;
{
	for (; *s1 && *s2; s1++, s2++) {
		if (*s1 != *s2)
			break;
	}

	if (*s1 && *s2)				/* chars did not match */
		return 1;
	if (*s1 == '\0' && *s2 == '\0')		/* all chars matched */
		return 0;

	if (*s1 == '.' || *s2 == '.')		/* short net name match */
		return 0;
	else
		return 1;
}

/*
**	Remove from consideration any node which is shown
**	by the IBM Job Manager to be busy.
*/
int
dep_inuse()
{
	struct	pbsnode	*np;
	int		i, j;
	extern	struct	JM_JOB_STATUS	*job_tbl;
	extern	struct	pbsnode		*pbsnlist;
	extern	int	rm_numnodes;

	if (getjobstat() == -1)
		return -1;

	for (np=pbsnlist; np; np=np->next) {
		if (np->inuse)
			continue;

		for (i=0; i<njob; i++) {
			struct  JM_JOB_STATUS   *jp = &job_tbl[i];

			for (j=0; j<jp->jm_num_nodes; j++) {
				struct JM_NODE_IN_JOB *jmnp =
					&jp->jm_node_in_job[j];

				if (node_cmp(np->last->name,
						jmnp->jm_node_name) == 0)
					break;
			}
			if (j < jp->jm_num_nodes)
				break;
		}
		if (i < njob) {		/* inuse by system */
			np->inuse |= FLAG_JOB;
			rm_numnodes--;
		}
	}
	return 0;
}

int
rm_getjob()
{
	static	uint	lastproc = 0;

	if (lastproc == reqnum)	/* don't need new table */
		return njob;

	return (getjobstat());
}

/*
 * Declare start of polling loop.
 *
 *	Until the next call to mom_get_sample, all mom_over_limit calls will
 *	use the same data.  Returns a PBS error code.
 */

int mom_get_sample()
{
	char	*id = "mom_get_sample";

	DBPRT(("%s: entered\n", id))

        if (getprocstat() != PBSE_NONE)
                return PBSE_SYSTEM;

	if (getjobstat() == -1)
		return PBSE_SYSTEM;

	return PBSE_NONE;
}

/*
 * Measure job resource usage and compare with its limits.
 *
 *	If it has exceeded any well-formed polled limit return TRUE.
 *	Otherwise, return FALSE.
 */

int mom_over_limit(pjob)
    job			*pjob;
{
	char		*id = "mom_over_limit";
	char		*pname;
	int		retval;
	unsigned long	value;
	resource	*pres;

	assert(pjob != NULL);
	assert(pjob->ji_wattr[(int)JOB_ATR_resource].at_type == ATR_TYPE_RESC);

	DBPRT(("%s: entered\n", id))
	if (nodes_ses(pjob))
		return (TRUE);

	pres = (resource *)
	    GET_NEXT(pjob->ji_wattr[(int)JOB_ATR_resource].at_val.at_list);
	for ( ; pres != NULL; pres = (resource *)GET_NEXT(pres->rs_link)) {
		assert(pres->rs_defin != NULL);
		pname = pres->rs_defin->rs_name;
		assert(pname != NULL);
		assert(*pname != '\0');

		if (strcmp(pname, "walltime") == 0) {
			retval = getnum(pres, &value);
			if (retval != PBSE_NONE)
				continue;
			num = (unsigned long)((double)(time_now - pjob->ji_qs.ji_stime) * wallfactor);
			if (num > value) {
				sprintf(log_buffer,
					"walltime %d exceeded limit %d",
					num, value);
				return (TRUE);
			}
		}
	}

	return (FALSE);
}

/*
 * Update the job attribute for resources used.
 *
 *	The first time this is called for a job, set up resource entries for
 *	each resource that can be reported for this machine.  Fill in the
 *	correct values.  Return an error code.
 *
 *	Assumes that the session ID attribute has already been set.
 */
int mom_set_use(pjob)
    job			*pjob;
{
	char			*id = "mom_set_use";
	resource		*pres;
	attribute		*at;
	resource_def		*rd;
	unsigned long		*lp;

	assert(pjob != NULL);
	DBPRT(("%s: entered\n", id))

	at = &pjob->ji_wattr[(int)JOB_ATR_resc_used];
	assert(at->at_type == ATR_TYPE_RESC);

	at->at_flags |= ATR_VFLAG_MODIFY;
	if ((at->at_flags & ATR_VFLAG_SET) == 0) {
		at->at_flags |= ATR_VFLAG_SET;

		rd = find_resc_def(svr_resc_def, "nodes", svr_resc_size);
		assert(rd != NULL);
		pres = add_resource_entry(at, rd);
		assert(pres != NULL);
		if ((pres->rs_value.at_val.at_str =
				nodelist(pjob->ji_qs.ji_jobid)) != NULL)
			pres->rs_value.at_flags |= ATR_VFLAG_SET;
		pres->rs_value.at_type = ATR_TYPE_STR;

		rd = find_resc_def(svr_resc_def, "walltime", svr_resc_size);
		assert(rd != NULL);
		pres = add_resource_entry(at, rd);
		assert(pres != NULL);
		pres->rs_value.at_flags |= ATR_VFLAG_SET;
		pres->rs_value.at_type = ATR_TYPE_LONG;
		pres->rs_value.at_val.at_long = 0;
	}

	rd = find_resc_def(svr_resc_def, "walltime", svr_resc_size);
	assert(rd != NULL);
	pres = find_resc_entry(at, rd);
	assert(pres != NULL);
	pres->rs_value.at_val.at_long = (long)((double)(time_now - pjob->ji_qs.ji_stime) * wallfactor);

	return (PBSE_NONE);
}


/*
 *	Kill a task session.
 *	Call with the task and a signal number.
 */
int kill_task(ptask, sig)
    task	*ptask;
    int		sig;
{
	char	*id = "kill_task";
	int	ct = 0;
	int	i;
	int	sesid;

	sesid = ptask->ti_qs.ti_sid;
	DBPRT(("%s entered sesid %d\n", id, sesid))

	if (sesid <= 1)
		return 0;

	if (getprocstat() != PBSE_NONE)
		return 0;

	for (i=0; i<nproc; i++) {
		register struct procsinfo	*pp = &proc_tbl[i];

		if (pp->pi_state == SNONE)
			continue;

		if (sesid != pp->pi_sid)
			continue;

		DBPRT(("%s: send signal %d to pid %d\n", id, sig, pp->pi_pid))
		(void)kill(pp->pi_pid, sig);
		++ct;
	}
	return ct;
}

/*
 * Clean up everything related to polling.
 *
 */
int mom_close_poll()
{
	DBPRT(("mom_close_poll entered\n"))

	return (PBSE_NONE);
}

/*
 * mom_does_chkpnt - return 1 if mom supports checkpoint
 *			    0 if not
 */

int mom_does_chkpnt()
{
	return (0);
}

/*
 * Checkpoint the job.
 *
 *	If abort is true, kill it too.
 */

int mach_checkpoint(ptask, file, abort)
    task	*ptask;
    char	*file;
    int		abort;
{
       	return (-1);
}

/*
 * Restart the job from the checkpoint file.
 *
 *	Return -1 on error or sid if okay.
 */

long mach_restart(ptask, file)
    task	*ptask;
    char	*file;
{
	return (-1);
}

int
kvm_read(fd, addr, buf, size)
    int		fd;
    long	addr;
    char	*buf;
    int		size;
{
	int	ret;

	if (lseek(fd, addr, SEEK_SET) != addr)
		return -1;
	if ((ret = read(fd, buf, size)) == -1)
		return -1;
	return ret;
}

char	*
cput(attrib)
struct	rm_attribute	*attrib;
{
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
}

char	*
mem(attrib)
struct	rm_attribute	*attrib;
{
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
}

char	*
idletime(attrib)
struct	rm_attribute	*attrib;
{
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
}

char	*
sessions(attrib)
struct	rm_attribute	*attrib;
{
#if 1
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
#else
	char			*id = "sessions";
	int			nproc;
	int			i, j;
	char			*fmt;
	int			njids = 0;
	pid_t			*jids, jobid;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((nproc = rm_getproc()) == 0) {
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	if ((jids = calloc(nproc, sizeof(pid_t))) == NULL) {
		log_err(errno, id, "no memory");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	/*
	** Search for session
	*/
	for (i=0; i<nproc; i++) {
		register struct procsinfo	*pp = &proc_tbl[i];
		if (pp->pi_state == SNONE)
			continue;
		if (pp->pi_suid == 0)
			continue;

		jobid = pp->pi_sid;
		if (jobid == 0)
			continue;

		DBPRT(("%s: pid %d sid %u\n", id, pp->pi_pid, jobid))
		for (j=0; j<njids; j++) {
			if (jids[j] == jobid)
				break;
		}
		if (j == njids)			/* not found */
			jids[njids++] = jobid;	/* so add it to list */
	}

	fmt = ret_string;
	for (j=0; j<njids; j++) {
		checkret(&fmt, 100);
		sprintf(fmt, " %d", jids[j]);
		fmt += strlen(fmt);
	}
	free(jids);
	return ret_string;
#endif
}

char	*
nsessions(attrib)
struct	rm_attribute	*attrib;
{
#if 1
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
#else
	char	*result, *ch;
	int	num = 0;

	if ((result = sessions(attrib)) == NULL)
		return result;

	for (ch=result; *ch; ch++) {
		if (*ch == ' ')		/* count blanks */
			num++;
	}
	sprintf(ret_string, "%d", num);
	return ret_string;
#endif
}

char	*
pids(attrib)
struct	rm_attribute	*attrib;
{
#if 1
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
#else
	char			*id = "pids";
	pid_t			jobid;
	int			nproc;
	int			i;
	char			*fmt;
	int			num_pids = 0;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((jobid = (pid_t)atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (getattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "session") != 0) {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((nproc = rm_getproc()) == 0) {
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	/*
	** Search for members of session
	*/
	fmt = ret_string;
	for (i=0; i<nproc; i++) {
		register struct procsinfo	*pp = &proc_tbl[i];

		if (pp->pi_state == SNONE)
			continue;

		if (jobid != pp->pi_sid)
			continue;

		checkret(&fmt, 100);
		sprintf(fmt, " %d", pp->pi_pid);
		fmt += strlen(fmt);
		num_pids++;

		DBPRT(("%s[%d]: pid %d sid %u\n",
		       id, num_pids, pp->pi_pid, pp->pi_sid))
	}
	if (num_pids == 0) {
		rm_errno = RM_ERR_EXIST;
		return NULL;
	}
	return ret_string;
#endif
}

char	*
nusers(attrib)
struct	rm_attribute	*attrib;
{
#if 1
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
#else
	char			*id = "nusers";
	int			nproc;
	int			i, j;
	int			nuids = 0;
	uid_t			*uids, uid;

	if (attrib) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((nproc = rm_getproc()) == 0) {
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	if ((uids = calloc(nproc, sizeof(uid_t))) == NULL) {
		log_err(errno, id, "no memory");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	for (i=0; i<nproc; i++) {
		register struct procsinfo	*pp = &proc_tbl[i];

		if (pp->pi_state == SNONE)
			continue;
		if ((uid = pp->pi_suid) == 0)
			continue;

		DBPRT(("%s: pid %d uid %u\n", id, pp->pi_pid, uid))
		for (j=0; j<nuids; j++) {
			if (uids[j] == uid)
				break;
		}
		if (j == nuids)			/* not found */
			uids[nuids++] = uid;	/* so add it to list */
	}

	sprintf(ret_string, "%d", nuids);
	free(uids);
	return ret_string;
#endif
}

char	*
ncpus(attrib)
struct rm_attribute	*attrib;
{
	if (attrib) {
		log_err(-1, "ncpus", extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	sprintf(ret_string, "%d", sysconf(_SC_NPROCESSORS_ONLN));
	return ret_string;
}

char	*
jnodes(attrib)
struct	rm_attribute	*attrib;
{
	char		*id = "jnodes";
	pid_t		jobid;
	int		nproc;
	int		i, j;
	int		nodcnt;
	int		found = 0;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((jobid = (pid_t)atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (getattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "session") != 0) {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if ((nproc = rm_getproc()) == 0) {
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	if (rm_getjob() == -1) {
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	/*
	** Search for members of session
	*/
	nodcnt = 0;
	for (i=0; i<nproc; i++) {
		register struct procsinfo	*pp = &proc_tbl[i];

		if (pp->pi_state == SNONE)
			continue;

		if (jobid != pp->pi_sid)
			continue;

		DBPRT(("%s: pid %d sid %u\n", id, pp->pi_pid, pp->pi_sid))
		found = 1;
		for (j=0; j<njob; j++) {
			struct	JM_JOB_STATUS	*jp = &job_tbl[j];

			if (jp->jm_client_pid == pp->pi_pid) {
				DBPRT(("%s: job pid %d jnodes %d\n",
					id, pp->pi_pid, jp->jm_num_nodes))

				nodcnt += jp->jm_num_nodes;
			}
		}
	}
	if (found) {
		sprintf(ret_string, "%d", nodcnt);
		return ret_string;
	}

	rm_errno = RM_ERR_EXIST;
	return NULL;
}

char	*
size_fs(param)
char	*param;
{
	char		*id = "size_fs";
	FILE		*mf;
	struct	mntent	*mp;
	struct	statfs	fsbuf;

	if (param[0] != '/') {
		sprintf(log_buffer, "%s: not full path filesystem name: %s",
			id, param);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (statfs(param, &fsbuf) == -1) {
		log_err(errno, id, "statfs");
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	/* in KB */
	sprintf(ret_string, "%ukb", (fsbuf.f_bsize * fsbuf.f_bavail) >> 10);
	return ret_string;
}

char	*
size_file(param)
char	*param;
{
	char		*id = "size_file";
	struct	stat	sbuf;

	if (param[0] != '/') {
		sprintf(log_buffer, "%s: not full path filesystem name: %s",
			id, param);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (stat(param, &sbuf) == -1) {
		log_err(errno, id, "stat");
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	sprintf(ret_string, "%ukb", sbuf.st_size >> 10); /* KB */
	return ret_string;
}

char	*
size(attrib)
struct	rm_attribute	*attrib;
{
	char	*id = "size";
	char	*param;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if (getattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	param = attrib->a_value;
	if (strcmp(attrib->a_qualifier, "file") == 0)
		return (size_file(param));
	else if (strcmp(attrib->a_qualifier, "fs") == 0)
		return (size_fs(param));
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
}

char	*
walltime(attrib)
struct	rm_attribute	*attrib;
{
#if 1
	rm_errno = RM_ERR_UNKNOWN;
	return NULL;
#else
	char			*id = "walltime";
	pid_t			value;
	int			i, nproc, job, found = 0;
	time_t			now, start;

	if (attrib == NULL) {
		log_err(-1, id, no_parm);
		rm_errno = RM_ERR_NOPARAM;
		return NULL;
	}
	if ((value = (pid_t)atoi(attrib->a_value)) == 0) {
		sprintf(log_buffer, "bad param: %s", attrib->a_value);
		log_err(-1, id, log_buffer);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}
	if (getattr(NULL)) {
		log_err(-1, id, extra_parm);
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if (strcmp(attrib->a_qualifier, "proc") == 0)
		job = 0;
	else if (strcmp(attrib->a_qualifier, "session") == 0)
		job = 1;
	else {
		rm_errno = RM_ERR_BADPARAM;
		return NULL;
	}

	if ((nproc = rm_getproc()) == 0) {
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}
	if ((now = time(NULL)) <= 0) {
		log_err(errno, id, "time");
		rm_errno = RM_ERR_SYSTEM;
		return NULL;
	}

	start = now;
	for (i=0; i<nproc; i++) {
		struct	procsinfo	*pp = &proc_tbl[i];

		if (pp->pi_state == SNONE)
			continue;

		if (job) {
			if (value != pp->pi_sid)
				continue;
		}
		else {
			if (value != pp->pi_pid)
				continue;
		}

		found = 1;
		DBPRT(("%s: pid %d start %d\n",
			id, pp->pi_pid, pp->pi_start))
		start = MIN(start, pp->pi_start);
	}
	if (found) {
		sprintf(ret_string, "%ld", (long)((double)(now - start) * walltimefactor));
		return ret_string;
	}

	rm_errno = RM_ERR_EXIST;
	return NULL;
#endif
}

#define	FSCALE	(1<<16)

int
get_la(rv)
	float	*rv;
{
	char	*id = "get_la";
	long	load;

	if (kd == NULL) {
		log_err(-1, id, nokernel);
		return (rm_errno = RM_ERR_SYSTEM);
	}
	if (kvm_read(kd, nl[KSYM_LOAD].n_value, (char *)&load,
		     sizeof(load)) != sizeof(load)) {
		log_err(errno, id, "kvm_read");
		return (rm_errno = RM_ERR_SYSTEM);
	}

	*rv = (float)load/FSCALE;
	return 0;
}

/*
 * dummy synbol for unknown IBM job manager routine,
 * wish IBM would give out some information about it...
 */
void ppslog()
{
}
