/*
 * FILE: dedqueue.c
 * Contains routines needed to process dedicated time and/or scheduled
 * down time.
 */
#include <sys/types.h>
#include <errno.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

/* PBS header files */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"

/* Scheduler header files */
#include "toolkit.h"
#include "gblxvars.h"
#include "msgs.h"

static int schedule_dedicated(Queue *dedq);

int 
schd_handle_dedicated_time(Queue *dedq)
{
    char   *id = "schd_handle_dedicated_time";
    Job    *jobs;
    Outage *outages;
    char   *comment, *shorthost;
    char   *ptr;
    char    buffer[256];
    int     ran;

    if (!(schd_ENFORCE_DEDTIME && schd_TimeNow >= schd_ENFORCE_DEDTIME)) {
	comment = schd_booltime2val(schd_ENFORCE_DEDTIME);
	strcpy(buffer, schd_JobMsg[NO_DED_TIME]);
	if (ptr = strchr(comment, '(')) {
	    strcat(buffer, " ");
	    strcat(buffer, ptr);
	}
	comment = buffer;
	DBPRT(("%s: dedicated time is not enforced.\n", id));
    } else {
	outages = schd_host_outage(dedq->exechost, 0);
	/* Get the short hostname for the outages string. */
	shorthost = schd_shorthost(dedq->exechost);

	if (outages != NULL) {

	    /* Are we currently in dedicated time? */
	    if ((schd_TimeNow >= outages->beg_time) &&
		(schd_TimeNow < outages->end_time)) 
	    {
		(void)sprintf(log_buffer, 
		    "currently dedtime for %s (%s/%s - %s/%s)", 
		    shorthost ? shorthost : outages->exechost, 
		    outages->beg_datestr, outages->beg_timestr, 
		    outages->end_datestr, outages->end_timestr);
		log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, 
		    log_buffer);
		DBPRT(("%s: %s\n", id, log_buffer));

		DBPRT(("%s: schedule_dedicated(%s@%s)\n", id, 
		    dedq->qname, dedq->exechost));
		ran = schedule_dedicated(dedq);

		return (ran);
	    } else {
		/* Dedtime is being enforced, and a dedtime is upcoming. */

	
		(void)sprintf(log_buffer, 
		    "upcoming dedtime for %s (%s/%s - %s/%s)", 
		    shorthost ? shorthost : outages->exechost, 
		    outages->beg_datestr, outages->beg_timestr, 
		    outages->end_datestr, outages->end_timestr);
		log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, 
		    log_buffer);
		DBPRT(("%s: %s\n", id, log_buffer));

		sprintf(buffer, "%s (%s %s %s)", schd_JobMsg[WAIT_FOR_DEDTIME],
		    shorthost ? shorthost : outages->exechost, 
		    outages->beg_datestr, outages->beg_timestr);
		if (shorthost)
		    free(shorthost);

		comment = buffer;
		DBPRT(("%s: %s\n", id, comment));
	    }
	} else {
	    /* Dedtime is being enforced, but no dedtime is upcoming. */
	    sprintf(buffer, "%s (none scheduled for %s)", 
		schd_JobMsg[WAIT_FOR_DEDTIME], shorthost);
	    comment = buffer;
	    DBPRT(("%s: waiting for dedicated time.\n", id));
	}
    }

    /* Set the comment in each of the dedicated time jobs
     * noting why they are waiting.
     * Don't care if it's NULL -- if so, no jobs are waiting.
     */
    for (jobs = dedq->jobs; jobs != NULL; jobs = jobs->next) {
	schd_comment_job(jobs, comment, JOB_COMMENT_OPTIONAL);
    }

    return (0);
}

/*
 * Dedicated-time job scheduling algorithm.
 *
 */
static int
schedule_dedicated(Queue *dedq)
{
    char   *id = "schedule_dedicated";
    Job    *job;
    char    reason[256];
    int     queue_full, ran = 0;

    /* 
     * Determine if the dedicated queue is full.  This isn't a fatal error -
     * if it is, just comment each job that this is the case.  'reason' will
     * still be set in the check below.
     */
    queue_full = schd_check_queue_limits(dedq, reason);

    if (dedq->rsrcs == NULL)
	dedq->rsrcs = schd_get_resources(dedq->exechost);

    if (dedq->rsrcs == NULL) {
	DBPRT(("%s: schd_get_resources(%s) failed!\n", id, dedq->exechost));
	return (-1);
    }
    /*
     * Choose the first N jobs that "fit" into the queue's resource limits.
     * prevents the call to actually run the job, but allows the scheduler 
     * to set the comment field to the correct string for each pending job.
     */

    DBPRT(("%s: Considering jobs:\n", id));
    for (job = dedq->jobs; job != NULL; job = job->next) {

	/* We are only interested in jobs in state "Q" == Queued */
	if (job->state != 'Q')
	    continue;

	/* 
	 * If queue_limits() says that no jobs may be run on the queue, set 
	 * the comment field of each job.  'reason' will still be set from
	 * the call to 'check_queue_limits()' above.
	 */
	if (queue_full) {
	    DBPRT(("queue_full -- rejected [%s]\n", reason));
	    schd_comment_job(job, reason, JOB_COMMENT_OPTIONAL);
	    continue;
	}

	if (!schd_job_fits_queue(job, dedq, reason)) {
	    DBPRT(("job %s does not fit %s : %s\n", job->jobid, dedq->qname, 
		reason));
	    schd_reject_job(job, reason);
	    continue;
	}

	/*
	 * Check that the resources requested by this job will not exceed 
	 * the system's available resources.
	 */
	if (schd_resource_limits(job, dedq->rsrcs, reason)) {
	    DBPRT(("rejected by resource_limits() [%s]\n", reason));
	    schd_comment_job(job, reason, JOB_COMMENT_OPTIONAL);
	    continue;
	}

	DBPRT(("dedicated_can_run('%s', '%s'):\n", job->jobid, dedq->qname));
	/* Zero 'when' argument indicates that we are asking about "now". */
	if (!schd_dedicated_can_run(job, dedq, 0, reason)) {
	    DBPRT(("No (%s).\n", reason));
	    schd_comment_job(job, reason, JOB_COMMENT_OPTIONAL);
	    continue;
	}

	/*
	 * Check that the resources requested by this job will not 
	 * exceed the queue's available resources, and that it will
	 * fit within the dedicated time.
	 */
	if (schd_user_limits(job, dedq, reason)) {
	    DBPRT(("rejected by user_limits() [%s]\n", reason));
	    schd_comment_job(job, reason, JOB_COMMENT_OPTIONAL);
	    continue;
	}


	/* 
	 * Job is okay.
	 * 
	 * Run the job on the queue, and set its comment to indicate when
	 * it began running.  Then add its expected resource usage to the
	 * queue's limits and resources.
	 */

	if (schd_run_job_on(job, dedq, dedq->exechost, SET_JOB_COMMENT)) {
	    (void)sprintf(log_buffer, "Unable to run job '%s' on queue '%s'.", 
		job->jobid, dedq->qname);
	    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, log_buffer);
	    return (-1);
	}

	/* Account for the job's impact on the queue.  */
	schd_charge_job(job, dedq, dedq->rsrcs);

	ran++;	/* Count number of jobs running on the queue. */
    }

    DBPRT(("%s: ran %d jobs on queue %s@%s.\n", id, ran, dedq->qname, 
	dedq->exechost));
    return (ran);
}
