/* $Id: file_chgs.c,v 1.1 1999/11/15 23:34:47 hender Exp $ */

/*
 * Routines to check if files have been changed, deleted, created, etc. since
 * the last invocation.  Used to automatically load new configuration and
 * statistics data.
 */

#include <sys/stat.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include <pbs_error.h>
#include <pbs_ifl.h>
#include <log.h>

#include "toolkit.h"

struct filestatus {
    struct filestatus *next;
    char    *filename;
    time_t   ctime;
    int      exists;
};
typedef struct filestatus FileStatus;

static FileStatus *filestats = NULL;

int
schd_register_file(char *filename)
{
    char    *id = "schd_register_file";
    FileStatus *stats, *tail, *new = NULL;

    /*
     * Look for the tail of the list.  While walking the list, check to see
     * that the filename is not already registered.
     */
    tail = NULL;
    for (stats = filestats; stats != NULL; stats = stats->next) {
	if (strcmp(filename, stats->filename) == 0) {
	    sprintf(log_buffer, "%s: file %s already registered.", id, 
		filename);
	    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, log_buffer);
	    DBPRT(("%s: %s\n", id, log_buffer));
	    return (-1);
	}
	tail = stats;
    }

    /* Create space for the new record. */
    new = (FileStatus *) malloc (sizeof (FileStatus));
    if (new == NULL) {
	sprintf(log_buffer, 
	    "%s: out of memory allocating FileStatus for file %s", 
	    id, filename);
	log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, log_buffer);
	return (-1);
    }

    /* Clear the record out -- this clears the ctime and next pointer. */
    memset (new, 0, sizeof(FileStatus));

    /* Keep a copy of the filename around. */
    new->filename = schd_strdup(filename);
    if (new->filename == NULL) {
	log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id,
	    "schd_strdup(filename)");
	return (-1);
    }

    /* 
     * If this is not the first element, tack it on the end of the list.
     * Otherwise, start the list with it.
     */
    if (tail)
	tail->next = new;
    else
	filestats = new;

    (void)sprintf(log_buffer, "%s: file %s registered.", id, filename);
    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, log_buffer);

    /*
     * Load the new element with the initial values for the file.  Ignore
     * the return value - only setting up the timestamp and file existence
     * status are important.
     */
    (void)schd_file_has_changed(filename, 1);

    return (0);
}

int
schd_file_has_changed(char *filename, int reset_stamp)
{
    char   *id = "schd_file_has_changed";

    FileStatus *stats;

    struct stat stat_buffer;
    int     exists;

    /* Assume that the file has not changed, and that it exists. */
    exists      = 1;

    if (filename == NULL) {
	DBPRT(("%s: filename is null\n", id));
	return (-1);
    }

    for (stats = filestats; stats != NULL; stats = stats->next) {
	if (strcmp(filename, stats->filename) == 0)
	    break;
    }

    if (stats == NULL) {
	DBPRT(("%s: filename %s not registered\n", id, filename));
	return (-1);
    }

    /* Get the file modification times from the filesystem. */
    if (stat(filename, &stat_buffer) == -1) {
	if (errno == ENOENT) {
	    exists = 0;
	} else {
	    (void)sprintf(log_buffer,
		"%s: stat(%s) failed: %d", id, filename, errno);
	    log_record(PBSEVENT_SYSTEM, PBS_EVENTCLASS_SERVER, id, log_buffer);
	    return (-1);
	}
    }

    /*
     * Has file has changed state?
     */
    if (exists != stats->exists) {
	stats->exists = exists;
	if (exists && reset_stamp)
	    stats->ctime = stat_buffer.st_ctime;
	return (1);
    }

    /*
     * If the ctime is different from the previously recorded one, the
     * file has changed.  stat(2) indicates that ctime will be changed
     * on every write, truncate, etc.  Update the ctime value for the
     * next call.
     */
    if (exists && (stat_buffer.st_ctime != stats->ctime)) {
	if (reset_stamp)
	    stats->ctime = stat_buffer.st_ctime;
	return (1);
    }

    /*
     * Either file still does not exist, or it has not been changed since
     * the last call.
     */

    return (0);
}

/*
 * "Forget" about the file named by 'filename', or all files if 'filename'
 * is a NULL pointer.  Returns the number of files removed from the watch
 * list, or -1 if the file is not found (or the list was empty).
 */
int
schd_forget_file(char *filename)
{
    /* char   *id = "schd_forget_file"; */
    FileStatus *stats, *prev, *next;
    int  count, found;

    count = 0;
    found = 0;
    prev  = NULL;

    /*
     * Remove entries in the list of file stats being watched that match
     * the supplied filename, or all entries if 'filename' is NULL.
     */

    for (stats = filestats; stats != NULL; stats = next) {

	next = stats->next;

	if (filename && (strcmp(filename, stats->filename) != 0)) {
	    prev = stats;
	    continue;
	}

	found ++;

	if (stats == filestats) {
	    filestats = stats->next;
	} else {
	    prev->next = stats->next;
	}

	/* Free the schd_strdup()'d filename */
	free(stats->filename);
	free(stats);
	count ++;
    }

    if (found)
	return (count);
    else
	return (-1);
}
