/* $Id: queue_limits.c,v 1.1 1999/11/15 23:34:53 hender Exp $ */

/*
 * There are two routines here.  The first, queue_limits(), evaluates the 
 * limits of a queue to see if the queue can support running another job.  
 * The other, user_limits(), evaluates a job against any imposed user 
 * limit for the specified queue.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <string.h>

/* PBS header files */

#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"

/* Scheduler header files */

#include "toolkit.h"
#include "gblxvars.h"

/*
 * Determine if a job *can* run in this queue.  This is distinct from if
 * it *should* be run in the queue.  
 *
 * A job *can* fit in a queue if its requested resources are not greater 
 * than the queue's maximums.
 *
 * A job *should* be run only if its requested resources do not exceed the
 * queue's *available* resources.
 */
int
schd_job_fits_queue(Job *job, Queue *queue, char *reason)
{
    /* char   *id = "schd_job_fits_queue"; */
    int     priority;

    /* Some of the limits are not tested for priority jobs.
     */
    priority = job->flags & JFLAGS_PRIORITY;

    /* 
     * Compare the job's requested resources against the queue's limits. 
     */
    if (!priority && (queue->wallt_min != UNSPECIFIED) && 
	(job->walltime < queue->wallt_min))
    {
	if (reason)
	    (void)sprintf(reason, 
		"Does not meet queue '%s' walltime minimum (%s).", 
		    queue->qname, schd_sec2val(queue->wallt_min));

        return (0);
    }

    if (!priority && (queue->wallt_max != UNSPECIFIED) && 
	(job->walltime > queue->wallt_max))
    {
	if (reason)
	    (void)sprintf(reason, 
		"Would exceed queue '%s' walltime limit (%s).", queue->qname, 
		    schd_sec2val(queue->wallt_max));

        return (0);
    }

    if (!priority && (queue->nodes_min != UNSPECIFIED) && 
	(job->nodes < queue->nodes_min)) 
    {
	if (reason)
	    (void)sprintf(reason, 
		"Does not meet queue '%s' node minimum (%d).", queue->qname, 
		queue->nodes_min);

        return (0);
    }

    if (!priority && (queue->nodes_max != UNSPECIFIED) && 
	(job->nodes > queue->nodes_max)) 
    {
	if (reason)
	    (void)sprintf(reason, "Would exceed queue '%s' node limit (%d).",
		queue->qname, queue->nodes_max);

        return (0);
    }

    /* 
     * The job _can_ fit in this queue.  This doesn't mean it *will* fit
     * in the queue as it currently exists, but it *would* fit if the queue
     * was completely empty.
     */
    return (1);		
}

int
schd_job_can_queue(Job *job) 
{
    QueueList *qptr;

    /* No?  Try the batch queues, if any. */
    if (schd_BatchQueues) {
	for (qptr = schd_BatchQueues; qptr != NULL; qptr = qptr->next) {
	    if (schd_job_fits_queue(job, qptr->queue, NULL))
		return (1);
	}
    }

    /* 
     * No queues were found that could run this job.  Reject it, since it
     * will never be able to run.
     */

    return (0);
}

/* 
 * Check queue limits to see if a job should be run out of this queue.
 */
int 
schd_check_queue_limits(Queue *queue, char *reason)
{
    /* char   *id = "schd_check_queue_limits"; */

    /* Is the queue enabled?  If not, note it and fail. */
    if (queue->flags & QFLAGS_DISABLED) {
	if (reason)
	    sprintf(reason, "Queue %s not enabled", queue->qname);
	return (-1);
    }

    /* Is the queue started?  If not, note it and fail. */
    if (queue->flags & QFLAGS_STOPPED) {
	if (reason)
	    sprintf(reason, "Queue %s not started", queue->qname);
	return (-1);
    }

    /* Is the queue below the run limit?  If so, note it and fail. */
    if ((queue->maxrun != UNSPECIFIED) && 
	(queue->running >= queue->maxrun)) 
    {
	if (reason)
	    sprintf(reason, "Queue %s run limit (%d) reached", queue->qname,
		queue->maxrun);

	/*
	 * Assert the MAXRUN bit, since the queue is essentially full.
	 */
	queue->flags |= QFLAGS_MAXRUN;

	return (-1);
    }

    /* Has the queue nodes limit been reached?  If so, note it and fail. */
    if ((queue->nodes_max != UNSPECIFIED) && 
	(queue->nodes_assn >= queue->nodes_max)) 
    {
	if (reason)
	    sprintf(reason, "Queue %s node limit (%d) reached", queue->qname,
		queue->nodes_max);

	/*
	 * Assert the QFLAGS_FULL bit, since a queue resource has been
	 * reached.
	 */
	queue->flags |= QFLAGS_FULL;

	return (-1);
    }

    /*
     * There are queued jobs and no queue limits have been reached.  Set 
     * the user run limit and return okay.
     */
    return (0);
}
