/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <pbs_error.h>
#include <pbs_ifl.h>
#include <log.h>
#include "state_count.h"
#include "constant.h"
#include "misc.h"

static char *ident = "$Id: state_count.c,v 2.1 1999/04/16 19:29:23 hender Exp $";


/*
 *
 *	print_state_count - print out a state_count struct
 *
 *	  sc - the struct to print
 *
 *	returns nothing
 *
 */
void print_state_count( state_count *sc )
{
  printf("running: %d\n", sc -> running);
  printf("queued: %d\n", sc -> queued);
  printf("transit: %d\n", sc -> transit);
  printf("exiting: %d\n", sc -> exiting);
  printf("held: %d\n", sc -> held);
  printf("waiting: %d\n", sc -> waiting);
  printf("total: %d\n", sc -> total);
}

/*
 *
 *	init_state_count - initalize state count struct
 *
 *	  sc - the struct to initalize
 *
 *	returns nothing
 *
 */
void init_state_count( state_count *sc )
{
  sc -> running = 0;
  sc -> queued = 0;
  sc -> transit = 0;
  sc -> exiting = 0;
  sc -> held = 0;
  sc -> waiting = 0;
  sc -> total = 0;
}

/*
 *
 *	count_states - count the jobs in each state and set the state counts
 *
 *	  jobs - array of jobs
 *	  sc   - state count structure passed by reference
 *
 *	returns nothing
 *
 */
void count_states( job_info **jobs, state_count *sc )
{
  int i;

  if( jobs  != NULL )
  {
    for( i = 0; jobs[i] != NULL; i++ )
    {
      if( jobs[i] -> is_queued )
	sc -> queued++;
      else if( jobs[i] -> is_running )
	sc -> running++;
      else if( jobs[i]-> is_transit )
	sc -> transit++;
      else if( jobs[i] -> is_exiting )
	sc -> exiting++;
      else if( jobs[i] -> is_held )
	sc -> held++;
      else if( jobs[i] -> is_waiting )
	sc -> waiting++;
      else
	log( PBSEVENT_JOB, PBS_EVENTCLASS_JOB, jobs[i] -> name, "Job in unknown state");
    }
  }

  sc -> total = sc -> queued + sc -> running + sc -> transit + sc -> exiting + sc -> held + sc -> waiting;

}

/*
 *
 *	total_states - add the states from sc2 to the states in sc1
 *		       i.e. sc1 += sc2
 *
 *	  sc1 - the accumlator
 *	  sc2 - what is being totaled
 *
 *	returns nothing
 *
 */
void total_states( state_count *sc1, state_count *sc2 )
{
  sc1 -> running += sc2 -> running;
  sc1 -> queued += sc2 -> queued;
  sc1 -> held += sc2 -> held;
  sc1 -> waiting += sc2 -> waiting;
  sc1 -> exiting += sc2 -> exiting;
  sc1 -> transit += sc2 -> transit;
  sc1 -> total += sc2 -> total;
}

