static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: libsched.a
       File: getconfig.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    This module contains the necessary functions for
    processing the config file and setting the 
    appropriate variables.  These functions are:
  
    getconfig:  main loop for processing the config file
    opnconfig:  open the configuration file for read
    clsconfig:  close the configuration file
    setcfgopt:  validate the configuration file and
                set appropriate variables

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <string.h>

#include "pbs_ifl.h"
#include "log.h"

/*
 *  Scheduler header files
 */
#include "toolkit.h"
#include "gblxvars.h"

#define	LINESIZE	1024	/* maximum line size */

/*
 *  This module contains the necessary functions for
 *  processing the config file and setting the 
 *  appropriate variables.  These functions are:
 *
 *  getconfig:  main loop for processing the config file
 *  opnconfig:  open the configuration file for read
 *  clsconfig:  close the configuration file
 *  setcfgopt:  validate the configuration file and
 *              set appropriate variables
 */

static	FILE *configfile = NULL;
static	char  line[LINESIZE];
static  int   opnconfig();
static  int   clsconfig();
static  int   setcfgopt(char *, char *);
/*
int           getconfig();
*/

/*
 *  Main loop for processing the configuration file
 */
int getconfig()
{
char	*id = "getconfig";
char	cfg_option[LINESIZE];
char	cfg_arg[LINESIZE];

	/*
	 *  Open the config file
	 */
	if (configfile == NULL)
		if (opnconfig() != 0)
			return(-1);

	cfg_arg[0] = '\0';

	/*
	 *  go through the config file line by line
	 */
	while(fgets(line,LINESIZE,configfile)) {
		if(strlen(line) < 2)
			continue;	/* skip blank lines */

		if(line[0] == '#')
			continue;	/* skip comments    */
		
		line[strlen(line)-1] = '\0'; /* remove newline */

		/*
		 *  Split the option and the arguement
		 */
		(void) sscanf(line,"%s %s",cfg_option,cfg_arg);

		/*
		 *  Set the configurable options
		 */
		if(setcfgopt(cfg_option,cfg_arg)) {
			clsconfig();
			return(-1);
		}

		cfg_arg[0] = '\0';
	}
	/*
	 *  Close the config file
	 */
	clsconfig();
	read_thresh();
	prtconfig();
	return(0);
}

/*
 *  Open the config file.  By default /usr/local/pbs/etc/pbssched_config
 *  is the config file.  This can be bypassed by setting the
 *  environment variable PBSSCHED_CONFIG to a different config
 *  file.
 */
static int opnconfig()
{
char	*id = "opnconfig";
char	*fn;

	if (clsconfig() != 0)
		return(-1);

	if ((fn = getenv("PBSSCHED_CONFIG")) == NULL)
		fn = CONFIGFILE;

	if ((configfile = fopen(fn, "r")) == NULL)
		return (-1);

	return(0);
}

/*
 *  Close the config file
 */
static int clsconfig()
{
char	*id = "clsconfig";

	if (configfile != NULL) {
		if (fclose(configfile) != 0)
			return(-1);
	}

	configfile = NULL;
	return(0);
}

/*
 *  Now that we have an option and its arguement,
 *  validate it and set the appropriate variables.
 */
static int setcfgopt(cfg_option,cfg_arg)
char	*cfg_option;
char	*cfg_arg;
{
char	*id = "setcfgopt";
char	*p1;
struct	pqueues *p2;
long long tmpnum;

	if(!strncmp(cfg_option,"COMPLEX_",8)) {
		create_complex(cfg_option,cfg_arg);
		create_qlist(cfg_arg);

	} else if(!strcmp(cfg_option,"ROUTER")) {
		create_router(cfg_arg);

	} else if(!strcmp(cfg_option,"HIGH_SWAPRATE")) {
		HIGH_SWAPRATE = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"MAX_LOADAVG")) {
		MAX_LOADAVG = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"MIN_LOADAVG")) {
		MIN_LOADAVG = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"NOM_LOADAVG")) {
		NOM_LOADAVG = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"HIGH_SYSTIME")) {
		HIGH_SYSTIME = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"MAX_JOBS")) {
		MAX_JOBS = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"MIN_JOBS")) {
		MIN_JOBS = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_MEM")) {
		LARGE_MEM = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_MEM_PRIME")) {
		LARGE_MEM_PRIME = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_MEM_NONPRIME")) {
		LARGE_MEM_NONPRIME = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_CNT")) {
		LARGE_CNT = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_CNT_PRIME")) {
		LARGE_CNT_PRIME = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_CNT_NONPRIME")) {
		LARGE_CNT_NONPRIME = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_ALLOC")) {
		LARGE_ALLOC = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_ALLOC_PRIME")) {
		LARGE_ALLOC_PRIME = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"LARGE_ALLOC_NONPRIME")) {
		LARGE_ALLOC_NONPRIME = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"LONG_TIME")) {
		LONG_TIME = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"LONG_CNT")) {
		LONG_CNT = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"BATCH_MEM")) {
		BATCH_MEM = val2byte(cfg_arg);

	} else if(!strcmp(cfg_option,"STARVATION_TIMEOUT")) {
		STARVATION_TIMEOUT = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"MAX_DED_JOBS")) {
		MAX_DED_JOBS = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"ENFORCE_ALLOC")) {
		ENFORCE_ALLOC = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"ENFORCE_DEDTIME")) {
		ENFORCE_DEDTIME = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"DED_TIME_COMMAND")) {
		DED_TIME_COMMAND = strdup(cfg_arg);

	} else if(!strcmp(cfg_option,"SCHED_ACCT_DIR")) {
		SCHED_ACCT_DIR = strdup(cfg_arg);

	} else if(!strcmp(cfg_option,"SYSTEM_NAME")) {
		SYSTEM_NAME = strdup(cfg_arg);

	} else if(!strcmp(cfg_option,"SYSTEM_NAME_CAPS")) {
		SYSTEM_NAME_CAPS = strdup(cfg_arg);

	} else if(!strcmp(cfg_option,"SERVER_HOST")) {
		SERVER_HOST = strdup(cfg_arg);

	} else if(!strcmp(cfg_option,"SCHED_HOST")) {
		SCHED_HOST = strdup(cfg_arg);

	} else if(!strcmp(cfg_option,"SCHED_ITERATION")) {
		SCHED_ITERATION = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"LINEAR_USAGE")) {
		LINEAR_USAGE = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"PNP")) {
		PNP = atoi(cfg_arg);

	} else if(!strcmp(cfg_option,"STARVATION_ORDER")) {
		create_starvlist(cfg_arg);

	} else if(!strcmp(cfg_option,"SCHEDULE_QUEUES")) {
		p1 = strtok(cfg_arg,",");
		Phead = (struct pqueues *)malloc(sizeof(struct pqueues));
		p2 = Phead;
		memset(p2,0,sizeof(struct pqueues));
		p2->queue_name = strdup(p1);
		while((p1=strtok(NULL,",")) != NULL) {
			p2->next = (struct pqueues *)malloc(sizeof(struct pqueues));
			p2 = p2->next;
			memset(p2,0,sizeof(struct pqueues));
			p2->queue_name = strdup(p1);
		}

	} else {
		/*
		 *  An unrecognized option was encountered, but
		 *  before assuming it was an error, it might be
		 *  complex information, so check that out first.
		 */
		p1 = strstr(cfg_option,"_");

		if(p1 == NULL)
			return(0);

		/*
		 * Got potential
		 */
		*p1 = '\0';
		p1++;
		if(!strcmp(p1,"JOBS") || 
		   !strcmp(p1,"MEMORY") || 
		   !strcmp(p1,"CPU")) 
			qcset(cfg_option,p1,cfg_arg);
		
		return(0);
	}

	if(cfg_arg[0] == '\0') {
		/*
		 *  A recognized option was found that did not
		 *  have that arguement that it should be set to
		 */
		return(-1);
	}

	return(0);
}

void
prtconfig()
{
char	*id = "prtconfig";
struct	routeq *rq;
struct	 qcomplex *qc;
struct	qthresh *qptr;

	(void)sprintf(log_buffer,"LARGE_MEM            = %lld",LARGE_MEM);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	(void)sprintf(log_buffer,"LARGE_MEM_PRIME      = %lld",LARGE_MEM_PRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	(void)sprintf(log_buffer,"LARGE_MEM_NONPRIME   = %lld",
		LARGE_MEM_NONPRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"LARGE_CNT            = %d",LARGE_CNT);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	(void)sprintf(log_buffer,"LARGE_CNT_PRIME      = %d",LARGE_CNT_PRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	(void)sprintf(log_buffer,"LARGE_CNT_NONPRIME   = %d",
		LARGE_CNT_NONPRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"LARGE_ALLOC          = %lld",LARGE_ALLOC_PRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	(void)sprintf(log_buffer,"LARGE_ALLOC_PRIME    = %lld",LARGE_ALLOC_PRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	(void)sprintf(log_buffer,"LARGE_ALLOC_NONPRIME = %lld",
		LARGE_ALLOC_NONPRIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"PNP                  = %d",PNP);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"HIGH_SWAPRATE        = %d",HIGH_SWAPRATE);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"MAX_LOADAVG          = %d",MAX_LOADAVG);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"MIN_LOADAVG          = %d",MIN_LOADAVG);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"NOM_LOADAVG          = %d",NOM_LOADAVG);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"HIGH_SYSTIME         = %d",HIGH_SYSTIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"MAX_JOBS             = %d",MAX_JOBS);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"MIN_JOBS             = %d",MIN_JOBS);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"LONG_TIME            = %d",LONG_TIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"LONG_CNT             = %d",LONG_CNT);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"BATCH_MEM            = %lld",BATCH_MEM);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"STARVATION_TIMEOUT   = %d",
		STARVATION_TIMEOUT);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"MAX_DED_JOBS         = %d",MAX_DED_JOBS);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"LINEAR_USAGE         = %d",LINEAR_USAGE);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"ENFORCE_ALLOC        = %d",ENFORCE_ALLOC);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"ENFORCE_DEDTIME      = %d",ENFORCE_DEDTIME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"SCHED_ACCT_DIR       = %s",SCHED_ACCT_DIR);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"DED_TIME_COMMAND     = %s",DED_TIME_COMMAND);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"SYSTEM_NAME          = %s",SYSTEM_NAME);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"SYSTEM_NAME_CAPS     = %s",SYSTEM_NAME_CAPS);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"SERVER_HOST          = %s",SERVER_HOST);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"SCHED_HOST           = %s",SCHED_HOST);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"SCHED_ITERATION      = %d",SCHED_ITERATION);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	(void)sprintf(log_buffer,"      Queue Complexes and Routers"); 
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	rq = Rhead;
	while(rq != (struct routeq *)0L) {
		(void)sprintf(log_buffer,"Router %s: %s",rq->name,rq->qlist); 
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		rq = rq->next;
	}

	qc = Chead;
	while(qc != (struct qcomplex *)0L) {
		(void)sprintf(log_buffer,
	"Complex %s: %d jobs, %lld bytes, %d cpu seconds, %s members",
		qc->name,qc->jobs,qc->mem,qc->cpu,qc->members);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		qc = qc->next;
	}

	(void)sprintf(log_buffer,"Queue Starvation Thresholds"); 
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	qptr = Thresholds;

	while(qptr != (struct qthresh *)0L) {
		(void)sprintf(log_buffer,"%s: %d",qptr->qname,qptr->starve); 
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		qptr = qptr->next;
	}
}
