static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: libsched.a
       File: queue_limits.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    There are two routines here.  The first, queue_limits, evaluates
    the limits of a queue to see if the queue can support running
    another job.  The is user_limits which evaluates a job against any
    imposed user limit for that queue.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <string.h>
/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
/*
 *  Scheduler header files
 */
#include "toolkit.h"

extern	int connector;

static	int user_run_limit;
static	long long queue_mem_limit;
static	long long queue_mem_assn;
static	long long default_mem_size;
static  int queue_ncpu_limit;
static  int queue_ncpu_assn;

/*
 *  Check queue limits to see if a job should be
 *  run out of this job.
 */
int queue_limits(queue_name)
char	*queue_name;
{
char	*id = "queue_limits";
struct	batch_status *bs;
struct	attrl *aptr;
int	running,queued,started,maxrun;

static	struct attrl alist[] = {
	{&alist[1],ATTR_start,     "",""},
	{&alist[2],ATTR_count,     "",""},
	{&alist[3],ATTR_maxuserrun,"",""},
	{&alist[4],ATTR_rescavail, "",""},
	{&alist[5],ATTR_rescassn,  "",""},
	{&alist[6],ATTR_rescdflt,  "",""},
	{&alist[7],ATTR_rescmax,   "",""},
	{NULL,     ATTR_maxrun,    "",""}
};

	/*
	 *  Initialize to UNSPECIFIED
	 */
	running=queued=started=maxrun = UNSPECIFIED;
	user_run_limit=queue_mem_limit=queue_mem_assn = UNSPECIFIED;
	default_mem_size = 0L;

	/*
	 *  Ask the server for queue information
	 */
	if((bs=pbs_statque(connector,queue_name,alist,NULL)) == 
	   (struct batch_status *)0L) {
		sprintf(log_buffer,"pbs_statque failed, \"%s\"%d",
			queue_name,pbs_errno);
		log_record(PBSEVENT_ERROR,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(-1);
	} 

	/*
	 *  Process the results
	 */
	aptr = bs->attribs;
	while(aptr != (struct attrl *)0L) {
		if(!strcmp(aptr->name,ATTR_start)) {
			if(!strcmp(aptr->value,"True"))
				started = 1;
			else if(!strcmp(aptr->value,"False"))
				started = 0;

		} else if(!strcmp(aptr->name,ATTR_count)) {
			running = how_many(aptr->value,SC_RUNNING);
			queued  = how_many(aptr->value,SC_QUEUED);

		} else if(!strcmp(aptr->name,ATTR_maxrun)) {
			maxrun = atoi(aptr->value);

		} else if(!strcmp(aptr->name,ATTR_maxuserrun)) {
			user_run_limit = atoi(aptr->value);

/*
		} else if(!strcmp(aptr->name,ATTR_rescavail)) {
*/
		} else if(!strcmp(aptr->name,ATTR_rescmax)) {
			if(!strcmp("mem",aptr->resource))
				queue_mem_limit = val2byte(aptr->value);

			if(!strcmp("ncpus",aptr->resource))
				queue_ncpu_limit = atoi(aptr->value);

		} else if(!strcmp(aptr->name,ATTR_rescassn)) {
			if(!strcmp("mem",aptr->resource)) 
				queue_mem_assn = val2byte(aptr->value);

			if(!strcmp("ncpus",aptr->resource)) 
				queue_ncpu_assn = atoi(aptr->value);

		} else if(!strcmp(aptr->name,ATTR_rescdflt)) {
			if(!strcmp("mem",aptr->resource))
				default_mem_size = val2byte(aptr->value);
		}

		aptr = aptr->next;
	}

	/*
	 *  Is the queue started
	 */
	if((started == UNSPECIFIED) || (started == 0)) {
		sprintf(log_buffer,"Queue %s not started",queue_name);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_QUEUE,id,log_buffer);
		report_skip_job(bs, log_buffer);
		pbs_statfree(bs);
		return(-1);
	}

	/*
	 *  Is the queue below the run limit
	 */
	if((maxrun != UNSPECIFIED) && (maxrun <= running)) {
		sprintf(log_buffer,"Queue %s run limit reached",queue_name);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_QUEUE,id,log_buffer);
		report_skip_job(bs, log_buffer);
		pbs_statfree(bs);
		return(-1);
	}

	/*
	 *  Are there any jobs to run
	 */
	if(!queued) {
		sprintf(log_buffer,"no jobs queued in %s",queue_name);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_QUEUE,id,log_buffer);
		return(-1);
	}

	/*
	 *  There are queued jobs and no queue limits
	 *  have been reached.  Set the user run limit and 
	 *  return okay to continue;
	 */
	return(0);
}

/*
 *  Given a user batch request and a queue name, evaluate the
 *  the job against limits retrieved from the queue_limits call
 */
int user_limits(bs,qname)
struct	batch_status *bs;
char	*qname;
{
char	*id = "user_limits";
static	struct	attrl *a;
struct	batch_status *b1, *b2;
int	jobs_running;
long long	xx;
int 	yy;
static  struct attropl alist[] = {
        {&alist[1],ATTR_state,"","R",EQ},
        {&alist[2],ATTR_q,    "","", EQ},
	{&alist[3],ATTR_euser,"","", EQ},
	{NULL,     ATTR_qtype,"","E",EQ}
};


	/*
	 *  If the queue has a user run limit then
	 *  check to make sure the user is under this limit
	 */
	if((user_run_limit != UNSPECIFIED) && user_run_limit) {

		/*
		 *  Store the queue name in the selstat request
		 */
		alist[1].value = qname;

		/*
		 *  Get the username from the batch request supplied
		 */
		a = bs->attribs;
		while(a != (struct attrl *)0L) {
			if(!strcmp(a->name,ATTR_euser))
				break;
			a = a->next;
		}

		/*
		 *  Store the username in the selstat request
		 */
		alist[2].value = a->value;

		/*
		 *  Get a list of all the jobs that fit the 
		 *  specified criteria
		 */
		b1 = pbs_selstat(connector,alist,NULL);
		if(pbs_errno) {
			pbs_statfree(b1);
			sprintf(log_buffer,"pbs_selstat failed, %d",pbs_errno);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,
				id,log_buffer);
		} else if(b1 != (struct batch_status *)0L) {
			/*
			 *  Count the jobs
			 */
			b2 = b1;
			jobs_running = 0;
			while(b2 != (struct batch_status *)0L) {
				jobs_running++;
				b2 = b2->next;
			}

			pbs_statfree(b1);

			/*
			 *  See if run limit will be exceeded
			 */
			if(user_run_limit <= jobs_running) {
				sprintf(log_buffer,"%s: user run limit",
					a->value);
				log_record(PBSEVENT_SYSTEM,
					PBS_EVENTCLASS_SERVER,id,log_buffer);
				return(-1);
			}
		}
	}

    if (!strncmp(qname, "bm", 2)) {
	/*
	 *  If the queue has a maximum ncpu limit then enforce it
	 */
	if((queue_ncpu_limit != UNSPECIFIED) && queue_ncpu_limit) {

		/*
		 *  How many CPUs is this job requesting
		 */
		a = bs->attribs;
		yy = 1;
		while(a != (struct attrl *)0L) {
			if(!strcmp(a->name,ATTR_l) && 
			   !strcmp(a->resource,"ncpus")) {
				yy = atoi(a->value);
				break;
			}
			a = a->next;
		}

		/*
		 *  Will this job exceed the queue ncpu limit
		 */
		if((yy+queue_ncpu_assn) > queue_ncpu_limit) {
			sprintf(log_buffer,"queue %s ncpu limit",qname);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,
				id,log_buffer);
			return(-1);
		}
	}

	/*
	 *  If the queue has a maximum memory limit then enforce it
	 */
	if((queue_mem_limit != UNSPECIFIED) && queue_mem_limit) {

		/*
		 *  How much memory is this job requesting
		 */
		a = bs->attribs;
		xx = default_mem_size;
		while(a != (struct attrl *)0L) {
			if(!strcmp(a->name,ATTR_l) && 
			   !strcmp(a->resource,"mem")) {
				xx = val2byte(a->value);
				break;
			}
			a = a->next;
		}

		/*
		 *  Will this job exceed the queue memory limit
		 */
		if((xx+queue_mem_assn) > queue_mem_limit) {
			sprintf(log_buffer,"queue %s memory limit",qname);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,
				id,log_buffer);
			return(-1);
		}
	}
    }
	return(0);
}
