static char USMID[] = "%Z%%M%	%I%	%G% %U%";

/*

    Program: pbs_sched.a
       File: seljob.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    This module provides the job selection algorithm.  Given
    a list of jobs, try to run a job from it.  Try each job in the 
    list before giving up.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include	<stdio.h>
/*
 *  PBS header files
 */
#include        "pbs_error.h"
#include        "pbs_ifl.h"
#include        "log.h"

#include "gblxvars.h"
#include "msgs.h"
#include "schdefs.h"
#include "toolkit.h"

extern	char		schedhost[255];
extern	int		connector;
extern	long long	LARGE_MEM_LIM;
extern	long long	LARGE_ALLOC_LIM;
extern	int		LARGE_CNT_LIM;
static	int 		run_job();
static	void 		skip_job();
extern	struct resutl 	*rsrcs;

static  char    *nohigh = "Insufficient high priority allocation";

int seljob(bs,qname,flag)
struct	batch_status *bs;
char	*qname;
int	flag;
{
char	*id = "seljob";
struct	attrl *aptr;
long long	jobmem;
long long	bm;
char	*usr;
char	*accnt;
int	cput;
int	ncpus_req;

	/*
	 *  Are there any jobs?
	 */
	if(bs == (struct batch_status *)NULL) {
		(void) sprintf(log_buffer,"no jobs in %s",qname);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(0);
	}

	/*
	 *  Go find a job to run
	 */
	while(bs != (struct batch_status *)0L) {
		/*
		 *  Check user limits, if okay go through algorithm
		 */
		jobmem = 0L;
		if(user_limits(bs,qname)) {
			bs = bs->next;
			continue;
		}

		/*
		 *  If this is a high priority queue
		 *  then recheck the user
		 */
		if(!strcmp(qname,"highbatch")) {
			cput = val2sec(getat(ATTR_l,bs,"cput"));
			usr = getat(ATTR_euser,bs,NULL);
			if(!usrok(usr,cput)) {
				exit(0);
				pbs_deljob(connector,bs->name,nohigh);
				bs = bs->next;
				continue;
			}
		}

		/*
		 *  If this is a LARGE_MEM job then 
		 *  make sure we should run it.
		 */
		jobmem = val2byte(getat(ATTR_l,bs,"mem"));

		if((LARGE_MEM_LIM) && 
		   (jobmem >= LARGE_MEM_LIM) && 
		   (flag == F_SCHEDULE)){
			/*
			 *  If not mt queue then only allow
			 *  LARGE_CNT - 1 LARGE_MEM jobs.  MT
			 *  enforce by queue limits
			 */
			if(strncmp(qname,"mt",2)) {
				if(rsrcs->lrgcnt >= (LARGE_CNT_LIM - 1)) {
					report_skip_job(bs,job_messages[NONMTLRGMEMLIM]);
					bs = bs->next;
					continue;
				}
			} else if((rsrcs->lrgcnt+rsrcs->lrgmtcnt+1) >
			    LARGE_CNT_LIM){
				report_skip_job(bs,job_messages[LRGMEMLIM]);
				bs = bs->next;
				continue;
			}

			/*
			 *  Is there enough LARGE_ALLOC 
			 *  memory for this job
			 */
			if((rsrcs->lrgmem + jobmem) > LARGE_ALLOC_LIM) {
				report_skip_job(bs,job_messages[LRGALLOCLIM]);
				bs = bs->next;
				continue;
			}
		}

		/*
		 *  Is there enough BATCH_MEM free?
		 */
		bm = rsrcs->batchmem + jobmem;
/*
(void) sprintf(log_buffer, "DEBUG: %s memory used(%lld)+job(%lld)=%lld max=%lld",
	bs->name, rsrcs->batchmem, jobmem, (rsrcs->batchmem+jobmem), BATCH_MEM);
log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
*/

		if((bm > BATCH_MEM) && (flag == F_SCHEDULE)) {
			report_skip_job(bs,job_messages[NOBATCHMEM]);
			bs = bs->next;
			continue;
		}

		/*
		 * If this is a mt job, could it push the load too high?
		 */
		ncpus_req = atoi(getat(ATTR_l,bs,"ncpus"));
		if ( rsrcs->loadave + ncpus_req > MAX_LOADAVG ) {
			report_skip_job(bs,job_messages[HILOAD]);
			bs = bs->next;
			continue;
		}

		/*
		 * ...or could it over-allocate CPUs?
		 */
		if ( rsrcs->ncpus_alloc + ncpus_req > rsrcs->ncpus_total ) {
			report_skip_job(bs,job_messages[NOCPU]);
			bs = bs->next;
			continue;
		}

                /*
		 *  Would this job cause the user to exceed his/her
		 *  group's current allocation? 
		 */
                if ( ENFORCE_ALLOC && over_allocation(bs, connector, 0, 0))
                        continue;

                /*
		 * 'twill interfere with Ded time?
		 */
                if ( ENFORCE_DEDTIME && have_dedicated_time )
                        if ( !dedicated_can_run(connector,0,bs) )
                                /* yes - can't run */
                              continue;


		/*
		 *  Is there enough SRFS space for this job
		 */
#if SRFS == 1
		if(srfs_check(bs)) {
		} else {
			report_skip_job(bs,job_messages[NOSRFS]);
		}
#endif
		/*
		 *  Run the job and stop looking for a job.
		 *  If the run fails, keep looking for a 
		 *  job to run.
		 */
		/*
		if(!run_job(bs))
			return(1);
		*/
		/* run the job, then find another to run...
		 */
		if(!run_job(bs)) {
			if (!strncmp(qname, "bm", 2))
			    return(1);

			rsrcs->ncpus_alloc += ncpus_req;
			rsrcs->loadave += ncpus_req;
			rsrcs->batchmem += jobmem;
		}
		bs = bs->next;
	}
	return(0);
}

static int run_job(bs)
struct	batch_status *bs;
{
char	*id = "run_job";

	/* 
	 * first clear the why-the-job-is-not-running comment
	 */
	report_skip_job(bs,NULL);

	/*
	 * then run that puppy!
	 */
	if(pbs_runjob(connector,bs->name,schedhost,NULL)) {
		(void) sprintf(log_buffer, "failed to start job %s, %d",
			bs->name,pbs_errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(-1);
	}

	(void) sprintf(log_buffer, "job %s started on %s",bs->name,schedhost);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	return(0);
}

static void skip_job(jname,reason)
char	*jname;
int	reason;
{
char	*id = "skip_job";

	(void) sprintf(log_buffer,"%s - %s",jname,job_messages[reason]);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
}

