/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 *
 * This file contains routines pulled over from the
 * parallel systems' PBS toolkit.
 *
 * Authors include: James Patton Jones, NAS/NASA Ames
 *                  Dr. Ed Hook, NASA LaRC
 *                  Chris Batten, NASA LaRC
 *
 * 
 * eventually I would like to have these two toolkits 
 * merged into one... -JJPJ
 *
 *
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <errno.h>

#include "gblxvars.h"
#include "toolkit.h"
/*
#include "msgs.h"
*/
extern char job_messages[];
extern int connector;

#include <pbs_error.h>
#include <pbs_ifl.h>
#include "log.h"
#include <sys/stat.h>

/*
 * following from: sched_io.c,v 1.7 1996/08/14 21:17:42 batten 
 */
#include <sys/stat.h>
#define WHY strerror(errno)

static int cmp(const void *a,const void *b);

/*========================================================*/
/* read allocation and usage data for current fiscal year */
/*========================================================*/

void sched_alloc_info(void)
{
	char *id = "sched_alloc_info";
	int i;
	int linenum, jobs, ret, alloc;
	int count = 0;
	char fname[MAXTXT+1];
	char buffer[MAXTXT*2];
	char gname[MAX_USER_NAME_SIZE];
	char uid[MAXTXT+1];
	char gid[MAXTXT+1];
	char used[MAXTXT+1];
	char alloc_line[MAXTXT+1];
	FILE *fp;
	char *p;
	float cpu_used, pct;
	int table_modified = 0;
	struct stat stat_buffer;

	if ( !need_to_get_alloc_info )
		goto get_current;

	/*=========================*/
	/* initialize the table(s) */
	/*=========================*/
	for ( i=0 ; i < MAX_GROUP ; ++i ) {
		group_table[i].allocation  = 0.0;
		strcpy(group_table[i].gid,"");
	}

	/*============================*/
	/*  fetch the allocation data */
	/*============================*/
	sprintf(fname,"%s/allocations",SCHED_ACCT_DIR);
	if ((fp=fopen(fname,"r")) == NULL ) {
                (void) sprintf(log_buffer,
		    "Warning: group allocation data is missing. Skipping...");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	} else {

		/* this is done here so that I already know that the */
		/* file exists. etc., etc. -- error-checking omitted */
		stat(fname,&stat_buffer);
		got_alloc_info = (long )stat_buffer.st_mtime;

		while ( fgets(buffer,sizeof buffer,fp) ) {
			p = strtok(buffer,"=");
			p = strtok(NULL,"=");
			p = strtok(NULL,"=");

			++p;
			if ( isdigit(*p) )
				sprintf(gname,"g%s",p);
			else
				sprintf(gname,"%s",p);
			gname[strlen(gname)-1] = '\0';	/* lose trailing ' ' */

			p = strtok(NULL,"=");
			if ( !strcmp(SYSTEM_NAME,"davinci") )
				;
			if ( !strcmp(SYSTEM_NAME,"babbage") )
				p = strtok(NULL,"=");
			if ( !strcmp(SYSTEM_NAME,"poseidon") )
				p = strtok(NULL,"=");

			alloc = atoi(p);
			if ( alloc ) {
				strcpy(group_table[count].gid,gname);
				group_table[count].allocation = alloc;
				++count;
			}
		}
		fclose(fp);
	}
	n_Allocation = count;
	qsort(group_table,count,sizeof(struct alloc_group),cmp);
	need_to_get_alloc_info = 0;
	need_to_get_YTD_info = 1;	/* necessary */
	table_modified = 1;

get_current:
	if ( !need_to_get_YTD_info )
		goto exeunt;

	/*=========================*/
	/* initialize the table(s) */
	/*=========================*/
	for ( i=0 ; i < n_Allocation ; ++i )
		group_table[i].total_usage = 0.0;

	/*===========================*/
	/*  fetch the YTD-usage data */
	/*===========================*/
	sprintf(fname,"%s/current",SCHED_ACCT_DIR);
	if ((fp=fopen(fname,"r")) == NULL ) {
		/* honest-to-G*d error */
                (void) sprintf(log_buffer,
		    "Error: %s: unable to open %s for read.", id, fname);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	} else {

		/* this is done here so that I already know that the */
		/* file exists. etc., etc. -- error-checking omitted */
		stat(fname,&stat_buffer);
		got_YTD_info = (long )stat_buffer.st_mtime;

		linenum = 0;
		while ( fgets(buffer, sizeof buffer,fp) ) {
			++linenum;
			ret = sscanf(buffer,
					"%s %s %d %f",
					uid,		/* user */
					gid,		/* group */
					&jobs,		/* nbr of jobs */
					&cpu_used	/* cput-hours */
			);
			if ( ret != 4 ) {	/* oops */
                		(void) sprintf(log_buffer,
					"Warning: line %d of %s data is corrupt. Skipping...",
					linenum, fname);
				log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
				continue;
			}
			/* find data for this group in our table */
			for ( i=0 ; i < n_Allocation ; ++i )
				if ( !strcmp(group_table[i].gid,gid) )
					break;
			if ( i == n_Allocation ) {	/* not there */
                		(void) sprintf(log_buffer,
					"Warning: no allocation for %s", gid);
				log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
				continue;
			}
			group_table[i].total_usage += cpu_used;
		}
		fclose(fp);
	}
	need_to_get_YTD_info = 0;
	table_modified = 1;

exeunt:
	if ( table_modified ) {
		for ( i=0 ; i < n_Allocation ; ++i ) {
			alloc=group_table[i].allocation;
			if (alloc > 0 ) {
				pct = group_table[i].total_usage;
				pct /= alloc;
				pct *= 100.0;
				sprintf(used,"%3.2f %%",pct);
				sprintf(alloc_line,"%d",alloc);
			} else {
				used[0] = '\0';
				strcpy(alloc_line,"N/A");
			}
                	(void) sprintf(log_buffer,
			    "Debug: %-8s Used: %16.4f Allocation: %10s  %s\n",
			    group_table[i].gid, group_table[i].total_usage, alloc_line, used);
			log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		}
	}
	return;
}

static int cmp(const void *a,const void *b)
{
	struct alloc_group *actual_a = (struct alloc_group *)a;
	struct alloc_group *actual_b = (struct alloc_group *)b;

	return strcmp(actual_a->gid,actual_b->gid);
}

int is_new_allocations_file(void)
{
	char *id = "is_new_allocations_file";
	char fname[MAXTXT+1];
	struct stat stat_buffer;

	sprintf(fname,"%s/allocations",SCHED_ACCT_DIR);
	if ( stat(fname,&stat_buffer) == -1 ) {
               	(void) sprintf(log_buffer, "%s: stat(%s) failed: %s\n",id,fname,WHY);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return 0;
	}
	return (long )stat_buffer.st_mtime != got_alloc_info;
}

int is_new_current_file(void)
{
	char *id = "is_new_current_file";
	char fname[MAXTXT+1];
	struct stat stat_buffer;

	sprintf(fname,"%s/current",SCHED_ACCT_DIR);
	if ( stat(fname,&stat_buffer) == -1 ) {
               	(void) sprintf(log_buffer, "%s: stat(%s) failed: %s\n",id,fname,WHY);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return 0;
	}
	return (long )stat_buffer.st_mtime != got_YTD_info;
}



/*
 * following from: sched_time.c,v 1.6 1996/08/14 21:17:42 batten 
 */

int MMDDYYYY_HHMM(char *MMDDYYYY,char *HHMM)
{
	struct tm junk;
	char copy[MAXTXT+1];

	strcpy(copy,MMDDYYYY);
	copy[2] = copy[5] = '\0';
	junk.tm_mon   = atoi(copy);
	junk.tm_mon  -= 1;
	junk.tm_mday  = atoi(copy+3);
	junk.tm_year  = atoi(copy+6);
	junk.tm_year -= 1900;

	strcpy(copy,HHMM);
	copy[2] = '\0';
	junk.tm_hour = atoi(copy);
	junk.tm_min  = atoi(copy+3);
	junk.tm_sec  = 0;

	junk.tm_isdst = -1;

	return (int )mktime(&junk);
}

int HHMMSS(char *spec,int *hours,int *mins,int *secs)
{
	char *id = "HHMMSS";
	char copy[MAXTXT+1];
	char *p;

	/*===================================================*/
	/* N.B. this function does [basically] NO validation */
	/*===================================================*/

	strcpy(copy,spec);
	p = strrchr(copy,':');
	if ( !p )
		return -1;
	*p++ = '\0';
	*secs = atoi(p);

	p = strrchr(copy,':');
	if ( !p )
		return -1;
	*p++ = '\0';
	*mins = atoi(p);

	*hours = atoi(copy);

	return (*hours * 3600 + *mins * 60 + *secs) ;
}

int hhmmss(char *spec)
{
	int hours;
	int mins;
	int secs;

	return HHMMSS(spec,&hours,&mins,&secs);
}

char *timestr(int seconds_since_1969,char *p)
{
	time_t junk;
	struct tm tm_junk;

	junk    = (time_t )seconds_since_1969;
	tm_junk = *localtime(&junk);

	strftime(p,MAXTXT+1,"%H:%M:%S",&tm_junk);

	return p;
}


/*===========================================================================*/
/* determine whether we're currently in dedicated time and, if not, when the */
/* next scheduled period of dedicated time will begin -- there's some bogus  */
/* generality here, since it's NOT made obvious that this code is making use */
/*    of the 'schedule' command [and, so, it can't be invoked at LaRC ...].  */
/*===========================================================================*/
void sched_dedicated_time(void)
{
	char *id = "sched_dedicated_time";
	char buffer[MAXTXT+1];
	char *p, *dedp;
	char Start_Date[MAXTXT+1];
	char Start_Time[MAXTXT+1];
	char End_Date[MAXTXT+1];
	char End_Time[MAXTXT+1];
        char ded_ck_cmd[200];
        char start_date[11],end_date[11],wks_host[30];
        char time_buffer[BUFSIZ],tbuf[250],savbuffer[250];
        int  i,c;
	int  sockfd;
	FILE *schedule_pipe;
	short err=0;

	have_dedicated_time = 0;

	sprintf(ded_ck_cmd, "%s %s", DED_TIME_COMMAND, SYSTEM_NAME);
        if ((schedule_pipe=popen(ded_ck_cmd,"r")) == NULL ) {

        	(void) sprintf(log_buffer, "popen(\"%s\",\"r\")",ded_ck_cmd);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

                return;
        }

        while ( fgets(buffer,sizeof buffer,schedule_pipe) != NULL ) {


                if ( strstr(buffer,"No ") )
                        break;                  /* None scheduled */

                if ( strstr(buffer,"CANCELLED") )
                        continue;               /* this was, but no more */


    		/*====================================================*/
    		/* format: "SYSTEM MM/DD/YYYY HH:MM-HH:MM MM/DD/YYYY" */
    		/*====================================================*/
    		p = strtok(buffer," \t\n");
    		if ( strcmp(p,SYSTEM_NAME_CAPS) )
    			continue;
    		p = strtok(NULL," \t\n");
    		strcpy(Start_Date,p);
    		p = strtok(NULL,"-");
    		strcpy(Start_Time,p);
    		p = strtok(NULL," \t\n");
    		strcpy(End_Time,p);
    		p = strtok(NULL," \t\n");
    		strcpy(End_Date,p);
    
    		/* convert to "absolute" times */
    		Start_of_Dedicated_Time = MMDDYYYY_HHMM(Start_Date,Start_Time);
    		End_of_Dedicated_Time   = MMDDYYYY_HHMM(End_Date,End_Time);
    
    		/* will it end at some _future_ time ?? */
    		if ( End_of_Dedicated_Time > time_now ) {
    
    			have_dedicated_time = 1;
    
    			strcpy(Ded_Start_Date,Start_Date);
    			strcpy(Ded_Start_Time,Start_Time);
    			strcpy(Ded_End_Date,End_Date);
    			strcpy(Ded_End_Time,End_Time);
    
    			break;
    	        }
	}
	if (have_dedicated_time) {
                (void) sprintf(log_buffer, "Ded/PM Time: "
			"Start_date=%s _time=%s", Ded_Start_Date, Ded_Start_Time);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
                (void) sprintf(log_buffer, "Ded/PM Time: "
			"End_date=%s _time=%s", Ded_End_Date, Ded_End_Time);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	} else {
                (void) sprintf(log_buffer, "Ded/PM Time: None Scheduled");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	}

	pclose(schedule_pipe);
	return;
}

/*
 * following from: sched_run.c,v 1.14 1996/10/01 00:59:05 jjones 
 */

/*==============================================================*/
/* Quick check on percentage of group allocation used           */
/*==============================================================*/
int is_over_allocation(char *group)
{
        int  percent_used;
        float alloc, used;

        alloc = get_allocation(group);
        if (alloc < 1.0)
            return(0);

        used = get_FY_used(group);
        percent_used = used / alloc * 100;

        return( percent_used < 100 ? 0 : percent_used );
}

/*==============================================================*/
/* User is over allocation, so enforce new limits specific to   */
/* the amount over; return val:  over allocation?  1=Yes  0=No  */
/*==============================================================*/
int over_allocation(struct  batch_status *bs, int sv_conn, int R_conn,
                                                        int report_only)
{
        char *id="over_allocation";
        char buffer[4*MAXTXT+1];
        int  rc;
        float alloc, used;
        char *p;
	char *group;
        char *nastygram =
                "PBS job %s would exceed group/project allocation.\n"
                "\n"
                "This limit has been imposed on you because your group (%s)\n"
                "has used %3.2f node-hours of its allocation of %3.1f\n"
                "node-hours. Contact your group Principal Investigator (PI)\n"
                "for additional information. The PI should contact his\n"
                "HPCCP Resource Monitor to apply for an allocation increase.\n"
                "\n"
        ;

	group = getat(ATTR_egroup,bs,NULL);
        alloc = get_allocation(group);
        used = get_FY_used(group);

        if (alloc >= 1.0 && used < alloc)
            return(0);

        /*
         * User has used up allocation, so we get to
         * delete this job. But we need to log the
         * delete, and notify the user via email why
         * the job was toasted.
         */
        else {
            /*
             * Don't mess with this delete stuff
             * if this is a remote job
             */
            if (R_conn == sv_conn) {
              /* reject local job */
              sprintf(buffer,nastygram,bs->name,group,used,alloc);
              p = strchr(buffer,'\n');
              *p = '\0';
              (void) sprintf(log_buffer, "%s", buffer);
	      log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
              *p = '\n';

              /*
               * ...and delete the job, including the above message.
               */
#ifdef THE_REAL_THING
               /* since job cannot run, delete it */
               rc = pbs_deljob(sv_conn,bs->name,buffer);
               if ( rc ) {
        	    (void) sprintf(log_buffer, "pbs_deljob failed: %d", rc);
		    log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	       }
#else
               /* report that we _would_ have trashed it */
#endif
            }
        }
        return 1;       /* yep, this one's over allocation */
}

float get_FY_used(char *gid)
{
        int i;

        for (i=0; i<n_Allocation; i++) {
            if (!strcmp(group_table[i].gid, gid))
                break;
        }
        return ( i==n_Allocation ? 0.0 : group_table[i].total_usage);
}

float get_allocation(char *gid)
{
        int i = 0;

        for (i=0; i<n_Allocation; i++) {
            if (!strcmp(group_table[i].gid, gid))
                break;
        }
        return ( i==n_Allocation ? 0.0 : group_table[i].allocation);

}

/*========================================================================*/
/* there's dedicated time scheduled sometime soon -- would starting 'job' */
/*                        now interfere with that ??                      */
/*========================================================================*/
int dedicated_can_run(int sv_conn, int R_conn, struct  batch_status *bs)
{
	char buffer[MAXTXT+1];

	/*  are we in the midst of dedicated time ? */
	if ( Start_of_Dedicated_Time <= time_now ) {
		if ( time_now < End_of_Dedicated_Time ) {
			sprintf(buffer,
				"Waiting for end of dedicated time (%s %s)",
				Ded_End_Date,
				Ded_End_Time
			);
			report_skip_job(bs, buffer);
			return 0;
		}
	} else		/* time_now < Start_of_Dedicated_Time */

	/* if the dedicated time is still in the future, */
	/*       would the job spill over into it ?      */
	if ( time_now + atoi(getat(ATTR_l,bs,"cput")) > Start_of_Dedicated_Time ) {
		sprintf(buffer,
			"Can't complete before next dedicated time (%s %s)",
			Ded_Start_Date,
			Ded_Start_Time
		);
		report_skip_job(bs, buffer);
		return 0;
	}

	return 1;	/* no problems -- job could run, AFAIAC */
}


/*==========================================================================*/
/* Record the reason why the current candidate job can't run at the moment; */
/* When it is decided that the job will remain queued, log the reason in    */
/* the comment string of the job itself.                                    */
/* --- was sched_report_status() ---                                        */
/*==========================================================================*/
void report_skip_job(bs, reason) 
struct batch_status *bs;
char  *reason;
{
	char *id = "report_skip_job";
	char status_msg[MAXTXT+1+1];
	char *msg_ptr;
	char *old_msg;

	/* 
	 * first make sure not to pass a NULL string to strcmp
	 * since Solaris really, really doesn't like it.
	 */

	if (reason == NULL)
		msg_ptr = "";
	else
		msg_ptr = strdup(reason);

	old_msg = getat(ATTR_comment,bs,NULL);

	if (old_msg == NULL) 
		sched_alterjob(connector, bs, ATTR_comment, msg_ptr, NULL);
	else {
	    if (strcmp(msg_ptr, old_msg))   /* if reasons are different... */
		sched_alterjob(connector, bs, ATTR_comment, msg_ptr, NULL);
	}
	return;
}

/*====================================================================*/
/* Alter a job's actual attributes                                    */
/*====================================================================*/
int sched_alterjob(int sv_conn, struct batch_status *bs, char* name, 
                                                char* value, char* resource)
{
        int err;
        struct attrl *atp;
        char *id = "sched_alterjob";

        /* Create an attribute structure for the pbs_alterjob() call */
        atp = (struct attrl *)malloc(sizeof(struct attrl));
	if ( !atp ) {
        	(void) sprintf(log_buffer, "malloc(atp) failed: %d", errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return errno;
	}
	
        /* Fill the attribute structure with function parameters */ 
        atp->name = name;
        atp->value = value;
	atp->resource = resource;
        atp->next = NULL;

        err = pbs_alterjob(sv_conn, bs->name, atp, NULL);

        if ( err ) {
        	(void) sprintf(log_buffer, "pbs_alterjob failed: %d", pbs_errno);
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
	}
        free(atp);
        return err;
}
