/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
static char USMID[] = "%Z%%M%   %I%     %G% %U%";

/*

    Program: pbs_sched.a
       File: schedule.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    This is the schedulers main loop.  The scheduler receives a 
    command and and calls the routine schedule.  Appropriate 
    actions are taken based on the command specified.  All actions
    taken by the scheduler must be initiated from here.

*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <time.h>
#include <string.h>
/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
#include "sched_cmds.h"
/*
 *  Scheduler header files
 */
#include "gblvars.h"
#include "schdefs.h"
#include "toolkit.h"

void clrcomp();

extern int connector;
struct	resutl *get_resutl();
struct	resutl *rsrcs;
struct alloc_group group_table[MAX_GROUP];

long long	LARGE_MEM_LIM;
int		LARGE_CNT_LIM;
long long	LARGE_ALLOC_LIM;

/*
 *  This routine gets the scheduling command and
 *  takes appropriate action. 
 *
 *  Valid commands are:
 *    SCH_ERROR	           ERROR
 *    SCH_SCHEDULE_NULL    This is only a place holder and is never sent to
 *                         the scheduler
 *    SCH_SCHEDULE_NEW     A new job was placed in an execution queue
 *    SCH_SCHEDULE_TERM    A running job terminated
 *    SCH_SCHEDULE_TIME    The schedulers time interval expired
 *    SCH_SCHEDULE_RECYC   Only 1 job was started in the prior scheduling
 *                         cycle
 *    SCH_SCHEDULE_CMD     The server attribute "scheduling" was set to true
 *    SCH_CONFIGURE        Perform any scheduler configuration stuff
 *    SCH_QUIT             QUIT
 *    SCH_RULESET          Reread the scheduler rules
 *    SCH_SCHEDULE_FIRST   First schedule after server starts
 */
int schedule(cmd)
int	cmd;
{
char	*id = "schedule";

	switch(cmd) {
		case SCH_ERROR:		/* error */
		case SCH_SCHEDULE_NULL:
		case SCH_RULESET:	/* reread scheduler rules */
			break;

		case SCH_SCHEDULE_FIRST: /*1st schedule after server starts */
		case SCH_SCHEDULE_CMD:	 /* schedule on command */
		case SCH_SCHEDULE_TERM:  /* running job terminated */
		case SCH_SCHEDULE_RECYC: /*recycle scheduler after 1 run */
		case SCH_SCHEDULE_NEW: 	 /* new job */
		case SCH_SCHEDULE_TIME:	 /* scheduler interval reached */
			if (!schedreq(cmd))
			    schedreq(cmd);
			break;

		case SCH_CONFIGURE:	/* redo any initialization */
			clrcomp();
			getconfig();
			prtconfig();
			read_holidays();
			break;

		case SCH_QUIT:		/* exit */
			exit(0);
	}
	return(0);
}

/*
 *  Schedule requests
 */
int schedreq(cmd)
int	cmd;
{
char	*id = "schedreq";
struct	pqueues *p2;

	if(pnp()) {
		LARGE_MEM_LIM = 
			(LARGE_MEM_NONPRIME)?LARGE_MEM_NONPRIME:LARGE_MEM;
		LARGE_CNT_LIM = 
			(LARGE_CNT_NONPRIME)?LARGE_CNT_NONPRIME:LARGE_CNT;
		LARGE_ALLOC_LIM = 
			(LARGE_ALLOC_NONPRIME)?LARGE_ALLOC_NONPRIME:LARGE_ALLOC;
	} else {
		LARGE_MEM_LIM = (LARGE_MEM_PRIME)?LARGE_MEM_PRIME:LARGE_MEM;
		LARGE_CNT_LIM = (LARGE_CNT_PRIME)?LARGE_CNT_PRIME:LARGE_CNT;
		LARGE_ALLOC_LIM = 
			(LARGE_ALLOC_PRIME)?LARGE_ALLOC_PRIME:LARGE_ALLOC;
	}

	(void) sprintf(log_buffer,">>>  Begin Scheduling Cycle (%d)  <<<",cmd);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);


        /* allocation and usage information are updated at [roughly] 2am */
        /* (Eastern time) -- so we need to fetch them anew in the middle */
        /*      of the night: that's the purpose of the next bit ...     */

        if ( ENFORCE_ALLOC ) {
        	if ( !need_to_get_alloc_info )  /* already fetched it */
                	need_to_get_alloc_info = is_new_allocations_file();

        	if ( !need_to_get_YTD_info )
                	need_to_get_YTD_info = is_new_current_file();

		if ( need_to_get_alloc_info || need_to_get_YTD_info )
			sched_alloc_info();     /* get allocation & FY usage */
	}

        /* since our policy is sensitive to the current time, there are  */
        /* various time-related parameters that need to be recalculated  */
        /* for each run through the algorithm; 				 */

        time(&time_now);
        tm_now = *localtime(&time_now);

        /* we don't want to interfere with dedicated time on the system  */
        /* if we can help it -- find out what we can about scheduled     */
        /* unavailability                                                */

        have_dedicated_time = 0;
        if ( ENFORCE_DEDTIME ) 		/* requires 'schedule' or clone  */
                sched_dedicated_time();


	/*
	 *  Get the current resource utilization
	 */
	if((rsrcs=get_resutl()) == (struct resutl *)0L) {
		(void) sprintf(log_buffer,">>>  Cancel Scheduling Cycle <<<");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(0);
	}

	/*
	 *  Route pending jobs
	 *  route("pending");
	 */

	/*
	 *  Always run the debug queue to the limit
	 * while(run_queue("debug",F_BYPASS));
	 */

	/*
	 *  Evaluate the system resources
	 */
	if(evaluate_system(rsrcs)) {
		/*
		 *  Run the production queues based on resources
		 */
		p2 = Phead;
		while(p2 != NULL) {
			if(run_queue(p2->queue_name,F_SCHEDULE)) {
				break;
			}
			p2 = p2->next;
		}
	}

	(void) sprintf(log_buffer,">>>  End Scheduling Cycle  <<<");
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	return(1);
}
