/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

static char USMID[] = "%Z%%M%	%I%	%G% %U%";

#include <pbs_config.h>   /* the master config generated by configure */

#if SRFS == 1

/*

    Program: pbs_sched.a
       File: srfs.c

        Author : Nicholas P. Cardo
                 Sterling Software
                 NAS Facility
                 NASA Ames Research Center

  Description:
    Well SRFS requires some special handling.  These routines help
    to manage the SRFS space by providing checking and assisting 
    routines for jobs.

*/

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <sys/srfs.h>
#include <sys/quota.h>
/*
 *  PBS header files
 */
#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"
/*
 *  Scheduler header files
 */
#include "gblxvars.h"
#include "toolkit.h"

extern	int connector;

#define	NOSRFS  0
#define SRFSOK  1

char	*getreq();

/*
 *  Check the srfs space for the specified job
 */
int srfs_check(bs)
struct	batch_status *bs;
{
char	*id = "srfs_check";
int	rm;
char	*ptr1;
struct	attrl *aptr;
int	fastavail,bigavail;
int	fastreq,bigreq;
int	rtc;
struct	fsres_s sr;

	fastreq = bigreq = 0;
	rtc = SRFSOK;

	/*
	 *  See if the user was requesting fast and/or big
	 */
	fastreq = val2byte(getat(ATTR_l,bs,"srfs_fast"));
	bigreq  = val2byte(getat(ATTR_l,bs,"srfs_big" ));

	/*
	 *  If no srfs space requested then the check is
	 *  okay since I want no space.
	 */
	if(!fastreq && !bigreq) 
		return(SRFSOK);

	/*
	 *  Get the current usage for big and fast
	 */
        quotactl("/fast",SRFS_INFO,(char *)&sr);
	fastavail = (sr.ares_avail + sr.delta + sr.reserve) * BSIZE;

        quotactl("/big" ,SRFS_INFO,(char *)&sr);
	bigavail  = (sr.ares_avail + sr.delta + sr.reserve) * BSIZE;

	(void) sprintf(log_buffer,"%s big=%d/%d, fast = %d/%d",
		bs->name, bigreq,bigavail, fastreq,fastavail);
	log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);

	/*
	 *  If fast or big was requested, see if there is enough
	 *  space available.
	 */
	if((fastreq) && (fastreq > fastavail))
		rtc = NOSRFS;

	if((bigreq) && (bigreq > bigavail))
		rtc = NOSRFS;

	return(rtc);
}

/*
 *  Remove the srfs assist
 */
int srfs_clear()
{
char	*id = "srfs_clear";
int	rm;
char	srfs_rsv[80];
struct	attrl alist[] = {
	{NULL,ATTR_l,"srfs_assist","false"}
};
char	*ptr1;

	/*
	 *  Establish a connection to the resource monitor
	 */
	if((rm=openrm(SERVER_HOST,0)) == -1) {
		(void) sprintf(log_buffer,"Unable to contact resource monitor");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(1);
	}

	/*
	 *  Clear the reservation
	 */
	sprintf(srfs_rsv,"srfs_reserve[FASTDIR=0][BIGDIR=0]");

	addreq(rm,srfs_rsv);

	/*
	 *  Disconnect from the resource monitor
	 */
	closerm(rm);
	
	/*
	 *  Now mark the job as not being assisted, if a job was being
	 *  assisted.  srfs_clear can be called as a sanity check.
	 */
	if(Assist.srfsfast || Assist.srfsbig)
		pbs_alterjob(connector,Assist.jobname,alist,NULL);

	return(0);
}

/*
 *  Set the srfs reserve
 */
int srfs_assist()
{
char	*id = "srfs_assist";
int	rm;
char	srfs_rsv[80];
struct	attrl alist[] = {
	{NULL,ATTR_l,"srfs_assist","true"}
};
char	*ptr1;

	/*
	 *  See if a srfs reservation is necessary
	 */
	if(!Assist.srfsfast && !Assist.srfsbig)
		return(0);

	/*
	 *  Establish a connection to the resource monitor
	 */
	if((rm=openrm(SERVER_HOST,0)) == -1) {
		(void) sprintf(log_buffer,"Unable to contact resource monitor");
		log_record(PBSEVENT_SYSTEM,PBS_EVENTCLASS_SERVER,id,log_buffer);
		return(1);
	}

	/*
	 *  Make the reservation
	 */
	sprintf(srfs_rsv,"srfs_reserve[FASTDIR=%d][BIGDIR=%d]",
		Assist.srfsfast,Assist.srfsbig);

	addreq(rm,srfs_rsv);

	ptr1=getreq(rm);

	/*
	 *  Disconnect from the resource monitor
	 */
	closerm(rm);

	/*
	 *  Now mark the job as being assisted so that mom
	 *  knows to do some special handling
	 */
	pbs_alterjob(connector,Assist.jobname,alist,NULL);

	return(0);
}
#endif
