/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* $Id: movejob.c,v 1.1 1999/04/16 19:33:16 hender Exp $ */

#include	<stdio.h>

/* PBS header files */
#include        "pbs_error.h"
#include        "pbs_ifl.h"
#include        "log.h"

/* Toolkit header files */
#include "toolkit.h"
#include "gblxvars.h"

/*
 * Move the internal representation of the given job from the list on its
 * queue to the tail of the destination queue's list.
 *
 * If the destination queue is NULL, this is equivalent to deleting the job
 * from the per-queue lists.
 */

int
schd_move_job_to(Job *thisjob, Queue *destq)
{
    Job    *prev, *tail;
    Queue  *srcq;

    srcq = thisjob->queue;

    if (srcq == NULL) {
	DBPRT(("move_job_to(Job %s, Queue %s) before job->queue init'd\n",
	    thisjob->jobid, destq ? destq->qname : "[dead]"));
	return (-1);
    }

    if (srcq->jobs == NULL) {
	DBPRT(("job %s says queue %s is owner, but joblist is NULL.\n",
	    thisjob->jobid, srcq->qname));
	return (-1);
    }

    /*
     * If the head of the source queue's job list is not the job in question,
     * walk down the list until we find the element before the job (i.e.
     * until the element's next pointer is equal to the job's pointer).
     */
    prev = NULL;

    if (srcq->jobs != thisjob) {
	for (prev = srcq->jobs; prev != NULL; prev = prev->next) {
	    if (prev->next == thisjob)
		break;
	}

	if (prev == NULL) {
	    DBPRT(("job %s says queue %s is owner, but not on queue joblist.\n",
		thisjob->jobid, srcq->qname));
	    return (-1);
	}
    }

    /* 
     * Remove the job from the source queue's list.  The previous pointer may
     * be NULL -- this indicates that the job is the head of the source list.
     * In that case, simply move the source queue's pointer forward and we're
     * done.  Otherwise, point the previous job's next pointer to skip over
     * this one.  Either way, the job is no longer a list, so set its next 
     * pointer to NULL.
     */
    if (prev == NULL)
	srcq->jobs = srcq->jobs->next;
    else
	prev->next = thisjob->next;
    thisjob->next = NULL;

    if (destq) {
	/*
	 * Append the job to the destination queue job list.  Like the source
	 * queue, a NULL pointer in queue->jobs indicates that the list is
	 * empty.  In this case, the detached job becomes the head of the
	 * list.  Otherwise, find the tail of the list and hook the new job
	 * onto the end of the list.
	 */
	if (destq->jobs == NULL) {
	    destq->jobs = thisjob;
	} else {
	    for (tail = destq->jobs; tail->next != NULL; tail = tail->next)
		/* do nothing -- just walk the list */ 
		;
	    tail->next = thisjob;
	}

	/* Make the destination the owner of the job. */
	thisjob->queue = destq;

    } else {
	/*
	 * Moving to a NULL queue is a job deletion.  This job is no longer
	 * referenced on the source, so will be lost.  Free it's resources.
	 */

	schd_free_jobs(thisjob);
    }

    /*
     * Account for the moved job.  Decrement the appropriate counters on the
     * source queue, and increment them on the destination queue (if present)
     */

    switch (thisjob->state) {
    case 'R':
	srcq->running --;
	if (destq)
	    destq->running ++;
	break;
    
    case 'Q':
	srcq->queued --;
	if (destq)
	    destq->queued ++;
	break;

    default:	/* Do nothing.  Present for completeness. */
	break;
    }

    return (0);
}
