/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/* $Id: user_limits.c,v 1.1 1999/04/16 19:34:07 hender Exp $ */

/*
 * There are two routines here.  The first, queue_limits(), evaluates the 
 * limits of a queue to see if the queue can support running another job.  
 * The other, user_limits(), evaluates a job against any imposed user 
 * limit for the specified queue.
 */

#include <stdio.h>
#include <stdlib.h>
#include <errno.h>
#include <sys/types.h>
#include <sys/param.h>
#include <sys/sysmacros.h>
#include <string.h>

/* PBS header files */

#include "pbs_error.h"
#include "pbs_ifl.h"
#include "log.h"

/* Scheduler header files */

#include "toolkit.h"
#include "gblxvars.h"

extern int connector;

static int user_running (Queue *queue, char *user);

/*
 * Given a user batch request and a queue name, evaluate the job against 
 * limits retrieved from the queue_limits() routine.
 */
int 
schd_user_limits(Job *job, Queue *queue, char *reason)
{
    /* char   *id = "schd_user_limits"; */
    int     jobs_running;

    /*
     * If there is a user run limit, make sure the user is under their limit.
     */

    if ((queue->userrun != UNSPECIFIED) && queue->userrun) {
	jobs_running = user_running (queue, job->owner);

	/* See if run limit will be exceeded */
	if ((jobs_running != UNSPECIFIED) && 
	    (jobs_running >= queue->userrun)) 
	{
	    if (reason)
		sprintf(reason, "Would exceed queue %s userrun limit (%d)",
		    queue->qname, queue->userrun);
	    return (-1);
	}
    }

    /* If the queue has a maximum node limit, enforce it. */
    if ((queue->nodes_max != UNSPECIFIED) && queue->nodes_max) {

	/* Will this job exceed the queue node limit? */
	if ((job->nodes + queue->nodes_assn) > queue->nodes_max) {
	    if (reason)
		sprintf(reason, "Would exceed queue %s node limit (%d)", 
		    queue->qname, queue->nodes_max);
	    return (-1);
	}
    }

    /* If the job meets all the criteria above, it is runnable. */
    return (0);
}

static int
user_running(Queue *queue, char *user)
{
    char   *id = "user_running";
    Job    *job;
    int     jobs_running;

    DBPRT(("%s(%s, %s)\n", id, user, queue->qname));

    jobs_running = 0;

    /* 
     * Count the jobs in the queue's job list that are running and owned
     * by this user.
     */

    for (job = queue->jobs; job != NULL; job = job->next) {
	if ((job->state != 'R') && (!strcmp(job->owner, user)))
	    jobs_running++;
    }

    DBPRT(("    user %s has %d job%s running in queue '%s'\n", 
	user, jobs_running, jobs_running == 1 ? "" : "s", queue->qname));

    return (jobs_running);
}
