/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * save_attr.c - This file contains the functions to perform a buffered
 *	save of an object (structure) and an attribute array to a file.
 *	It also has the function to recover (reload) an attribute array.
 *
 * Included public functions are:
 *
 *	save_setup	called to initialize the buffer
 *	save_struct	copy a struct into the save i/o buffer
 *	save_flush	flush out the current save operation
 *	save_attr	buffer and write attributes to disk file
 *	recov_attr	read attributes from disk file
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include "pbs_ifl.h"
#include <assert.h>
#include <errno.h>
#include <memory.h>
#include <stdlib.h>
#include <stdio.h>
#include "list_link.h"
#include "attribute.h"
#include "log.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: attr_recov.c,v $ $Revision: 2.2 $";

/* Global Variables */

extern int resc_access_perm;

/* data items global to functions in this file */

#define PKBUFSIZE 2048
#define ENDATTRIBUTES -711

char   pk_buffer[PKBUFSIZE];	/* used to do buffered output */
static int     pkbfds = -2;	/* descriptor to use for saves */
static size_t  spaceavail;	/* space in pk_buffer available */
static size_t  spaceused;	/* amount of space used  in pkbuffer */


/*
 * save_setup - set up the save i/o buffer.
 *	The "buffer control information" is left updated to reflect
 *	the file descriptor, and the space in the buffer.
 */

void save_setup(fds)
	int	 fds;		/* file descriptor to use for save */
{

	if (pkbfds != -2) {	/* somebody forgot to flush the buffer */
		log_err(-1, "save_setup", "someone forgot to flush");
	}

	/* initialize buffer control */

	pkbfds = fds;
	spaceavail = PKBUFSIZE;
	spaceused = 0;
}

/*
 * save_struct - Copy a structure (as a block)  into the save i/o buffer
 *	This is useful to save fixed sized structure without pointers
 *	that point outside of the structure itself.
 *
 *	Write out buffer as required. Leave spaceavail and spaceused updated
 *
 *	Returns: 0 on success
 *		-1 on error
 */

int save_struct(pobj, objsize)
	char		*pobj;
	unsigned int	 objsize;
{  
	int	 amt;
	size_t	 copysize;
	int      i;
	char	*pbufin;
	char	*pbufout;


	assert(pkbfds >= 0);

	while (objsize > 0) {
		pbufin = pk_buffer + spaceused;
		if (objsize > spaceavail) {
			if ((copysize = spaceavail) != 0) {
				(void)memcpy(pbufin, pobj, copysize);
			}
			amt = PKBUFSIZE;
			pbufout = pk_buffer;
			while ((i=write(pkbfds, pbufout, amt)) != amt) {
				if (i == -1) {
					if (errno != EINTR) {
						return (-1);
					}
				} else {
					amt -= i;
					pbufout += i;
				}
			}
			pobj += copysize;
			spaceavail = PKBUFSIZE;
			spaceused  = 0;
		} else {
			copysize = (size_t)objsize;
			(void)memcpy(pbufin, pobj, copysize);
			spaceavail -= copysize;
			spaceused  += copysize;
		}
		objsize   -= copysize;
	}
	return (0);
}
		

/*
 * save_flush - flush out the current save operation
 *	Flush buffer if needed, reset spaceavail, spaceused,
 *	clear out file descriptor
 *
 *	Returns: 0 on success
 *		-1 on failure (flush failed)
 */

int save_flush()
{
	int i;
	char *pbuf;

	assert(pkbfds >= 0);

	pbuf = pk_buffer;
	if (spaceused > 0) {
		while ((i=write(pkbfds, pbuf, spaceused)) != spaceused) {
			if (i == -1) {
				if (errno != EINTR)  {
					log_err(errno,"save_flush","bad write");
					return (-1);
				}
			} else {
				pbuf += i;
				spaceused -= i;
			}
		}
	}
	pkbfds  = -2;	/* flushed flag */
	return (0);
}


/*
 * save_attr() - write set of attributes to disk file
 *
 *	Each of the attributes is encoded  into the attrlist form.
 *	They are packed and written using save_struct().
 *
 *	The final real attribute is followed by a dummy attribute with a 
 *	al_size of ENDATTRIB.  This cannot be mistaken for the size of a
 *	real attribute.
 *
 *	Note: attributes of type ATR_TYPE_ACL are not saved with the other
 *	attribute of the parent (queue or server).  They are kept in their
 *	own file.
 */

int save_attr(padef, pattr, numattr) 
	struct attribute_def *padef;	/* attribute definition array */
	struct attribute *pattr;	/* ptr to attribute value array */
	int	numattr;		/* number of attributes in array */
{
	svrattrl	 dummy;
	int		 errct = 0;
	list_head 	 lhead;
	int		 i;
	svrattrl	*pal;
	int		 rc;
	
	/* encode each attribute which has a value (not non-set) */

	CLEAR_HEAD(lhead);

	for (i = 0; i < numattr; i++) {

		if ( (padef+i)->at_type != ATR_TYPE_ACL) {

			/* note access lists are not saved this way */

			rc = (padef+i)->at_encode(pattr+i, &lhead,
			                 (padef+i)->at_name,
					 (char *)0, ATR_ENCODE_SAVE);
		
			if (rc < 0)
				errct++;

			(pattr+i)->at_flags &= ~ATR_VFLAG_MODIFY; 
	
			/* now that it has been encoded, block and save it */
	
			while ((pal = (svrattrl *)GET_NEXT(lhead)) !=
				(svrattrl *)0)  {
	
				if (save_struct((char *)pal, pal->al_tsize) < 0)
					errct++;
				delete_link(&pal->al_link);
				(void)free(pal);
			}
		}
	}

	/* indicate last of attributes by writting dummy entry */

#ifdef DEBUG
	(void)memset(&dummy, 0, sizeof(dummy));
#endif
	dummy.al_tsize = ENDATTRIBUTES;
	if (save_struct((char *)&dummy, sizeof (dummy)) < 0)
		errct++;


	if (errct)
		return (-1);
	else
		return (0);
}


/*
 * recov_attr() - read attributes from disk file
 *
 *	Recover (reload) attribute from file written by save_attr().
 *	Since this is not often done (only on server initialization),
 *	Buffering the reads isn't done.  
 */

int recov_attr(fd, parent, padef, pattr, limit, unknown) 
	int	fd;
	void   *parent;
	struct attribute_def *padef;
	struct attribute     *pattr;
	int	limit;
	int	unknown;
{
	static	  char	id[] = "recov_attr";
	int	  amt;
	int	  i;
	int	  index;
	svrattrl *pal;
	int	  palsize = 0;
	svrattrl  tempal;

	/* set all privileges (read and write) for decoding resources	*/
	/* This is a special (kludge) flag for the recovery case, see	*/
	/* decode_resc() in lib/Libattr/attr_fn_resc.c			*/

	resc_access_perm = ATR_DFLAG_ACCESS;

	/* For each attribute, read in the attr_extern header */

	while (1) {
		i  = read(fd, (char *)&tempal, sizeof (tempal));
		if (i != sizeof (tempal)) {
			log_err(errno, id, "read1");
			return (-1);
		}
		if (tempal.al_tsize == ENDATTRIBUTES)
			break;		/* hit dummy attribute that is eof */

		/* read in the attribute chunck (name and encoded value) */

		if (palsize == 0)
			pal = (svrattrl *)malloc(tempal.al_tsize);
		else if (palsize < tempal.al_tsize)
			pal = (svrattrl *)realloc(pal, tempal.al_tsize);
		palsize = tempal.al_tsize;
		if (pal == (svrattrl *)0)
			return (-1);
		*pal = tempal;
		CLEAR_LINK(pal->al_link);
		
		/* read in the actual attribute data */

		amt = pal->al_tsize - sizeof (svrattrl);

		i = read(fd, (char *)pal + sizeof (svrattrl), amt);
		if (i != amt) {
			log_err(errno, id, "read2");
			return (-1);
		}

		/* the pointer into the data are of course bad, so reset them */

		pal->al_name = (char *)pal + sizeof (svrattrl);
		if (pal->al_rescln) 
			pal->al_resc = pal->al_name + pal->al_nameln;
		else
			pal->al_resc = (char *)0;
		if (pal->al_valln)
			pal->al_value = pal->al_name + pal->al_nameln +
				        pal->al_rescln;
		else
			pal->al_value = (char *)0;


		/* find the attribute definition based on the name */

		index = find_attr(padef, pal->al_name, limit);
		if (index < 0)	{

			/*
			 * There are two ways this could happen:
			 * 1. if the (job) attribute is in the "unknown" list -
			 *    keep it there;
			 * 2. if the server was rebuilt and an attribute was
			 *    deleted, -  the fact is logged and the attribute
			 *    is discarded (system,queue) or kept (job)
			 */
			if (unknown > 0) {
				index = unknown;
			} else {
			    log_err(-1, id, "unknown attribute discarded");
			    continue;
			}
		}

		(void)(padef+index)->at_decode(pattr+index, pal->al_name,
					       pal->al_resc, pal->al_value);
		if ((padef+index)->at_action != (int (*)())0)
			(padef+index)->at_action(pattr+index,parent,ATR_ACTION_RECOV);
		(pattr+index)->at_flags = pal->al_flags & ~ATR_VFLAG_MODIFY; 
	}

	if (palsize)
		(void)free(pal);
	return (0);
}
