/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * dis_read.c - contains function to read and decode the DIS
 *	encoded requests and replies.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <stdlib.h>
#include <errno.h>
#include <string.h>
#include <stdio.h>
#include "dis.h"
#include "libpbs.h"
#include "list_link.h"
#include "server_limits.h"
#include "attribute.h"
#include "log.h"
#include "pbs_error.h"
#include "credential.h"
#include "batch_request.h"
#include "net_connect.h"

static char ident[] = "@(#) $RCSfile: dis_read.c,v $ $Revision: 2.1 $";


/* External Global Data */

extern char	*msg_nosupport;

/*
 * dis_request_read - read in an DIS encoded request from the network 
 * and decodes it:
 *	Read and decode the request into the request structures 
 *
 *	returns: 0 if request read ok, batch_request pointed to by request is
 *		   updated.
 *		-1 if EOF (no request but no error)
 *		>0 if errors ( a PBSE_ number)
 */

int dis_request_read (sfds, request)
	int	  sfds;			/* socket to read */
	struct batch_request *request;	/* server internal structure */
{
	int	 proto_type;
	int	 proto_ver;
	int	 rc; 	/* return code */

	DIS_tcp_setup(sfds);	/* setup for DIS over tcp */

	/* Decode the Request Header, that will tell the request type */

	if (rc = decode_DIS_ReqHdr(sfds, request, &proto_type, &proto_ver)) {
		if (rc == DIS_EOF)
			return EOF;
		(void)sprintf(log_buffer, "Req Header bad, dis error %d", rc);
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST, "?",
			  log_buffer);
		return PBSE_DISPROTO;
	}

	if (proto_ver != PBS_BATCH_PROT_VER) {
		return PBSE_DISPROTO;
	}

	/* Decode the Request Body based on the type */

	switch (request->rq_type) {
	    case PBS_BATCH_QueueJob:
		CLEAR_HEAD(request->rq_ind.rq_queuejob.rq_attr);
		rc = decode_DIS_QueueJob(sfds, request);
		break;

	    case PBS_BATCH_JobCred:
		rc = decode_DIS_JobCred(sfds, request);
		break;

	    case PBS_BATCH_jobscript:
	    case PBS_BATCH_MvJobFile:
		rc = decode_DIS_JobFile(sfds, request);
		break;

	    case PBS_BATCH_RdytoCommit:
	    case PBS_BATCH_Commit:
	    case PBS_BATCH_Rerun:
		rc = decode_DIS_JobId(sfds, request->rq_ind.rq_commit);
		break;

	    case PBS_BATCH_DeleteJob:
	    case PBS_BATCH_HoldJob:
	    case PBS_BATCH_ModifyJob:
		rc = decode_DIS_Manage(sfds, request);
		break;

	    case PBS_BATCH_MessJob:
		rc = decode_DIS_MessageJob(sfds, request);
		break;

	    case PBS_BATCH_Shutdown:
		rc = decode_DIS_ShutDown(sfds, request);
		break;

	    case PBS_BATCH_SignalJob:
		rc = decode_DIS_SignalJob(sfds, request);
		break;

	    case PBS_BATCH_StatusJob:
		rc = decode_DIS_Status(sfds, request);
		break;

#ifndef PBS_MOM
	    case PBS_BATCH_LocateJob:
		rc = decode_DIS_JobId(sfds, request->rq_ind.rq_locate);
		break;

	    case PBS_BATCH_Manager:
	    case PBS_BATCH_ReleaseJob:
		rc = decode_DIS_Manage(sfds, request);
		break;

	    case PBS_BATCH_MoveJob:
	    case PBS_BATCH_OrderJob:
		rc = decode_DIS_MoveJob(sfds, request);
		break;

	    case PBS_BATCH_RunJob:
	    case PBS_BATCH_AsyrunJob:
	    case PBS_BATCH_StageIn:
		rc = decode_DIS_RunJob(sfds, request);
		break;

	    case PBS_BATCH_SelectJobs:
	    case PBS_BATCH_SelStat:
		CLEAR_HEAD(request->rq_ind.rq_select);
		rc = decode_DIS_svrattrl(sfds, &request->rq_ind.rq_select);
		break;

	    case PBS_BATCH_StatusNode:
	    case PBS_BATCH_StatusQue:
	    case PBS_BATCH_StatusSvr:
		rc = decode_DIS_Status(sfds, request);
		break;

	    case PBS_BATCH_TrackJob:
		rc = decode_DIS_TrackJob(sfds, request);
		break;

	    case PBS_BATCH_Rescq:
	    case PBS_BATCH_ReserveResc:
	    case PBS_BATCH_ReleaseResc:
		rc = decode_DIS_Rescl(sfds, request);
		break;

	    case PBS_BATCH_RegistDep:
		rc = decode_DIS_Register(sfds, request);
		break;

	    case PBS_BATCH_AuthenUser:
		rc = decode_DIS_Authen(sfds, request);
		break;

	    case PBS_BATCH_JobObit:
		rc = decode_DIS_JobObit(sfds, request);
		break;

#else	/* yes PBS_MOM */
		
	    case PBS_BATCH_CopyFiles:
	    case PBS_BATCH_DelFiles:
		rc = decode_DIS_CopyFiles(sfds, request);
		break;

#endif	/* PBS_MOM */

	    default:
		sprintf(log_buffer, "%s: %d from %s", msg_nosupport, request->rq_type, request->rq_user);
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST, "?",
			  log_buffer);
		rc = PBSE_UNKREQ;
		break;
	}
		
	if (rc == 0) {

		/* Decode the Request Extension, if present */

		if (rc = decode_DIS_ReqExtend(sfds, request)) {
			(void)sprintf(log_buffer, "Req Extension bad, dis error %d", rc);
			LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST, "?",
				  log_buffer);
				rc = PBSE_DISPROTO;
		}
	} else if (rc != PBSE_UNKREQ) {
		(void)sprintf(log_buffer, "Req Body bad, dis error %d, type %d",
			      rc, request->rq_type);
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_REQUEST, "?",
			  log_buffer);
		rc = PBSE_DISPROTO;
	}

	return (rc);
}


/*
 * DIS_reply_read - read and decode DIS based batch reply
 *
 *	Returns  0 - reply read and decoded ok
 *		!0 - otherwise
 */


int DIS_reply_read(sock, preply)
	int 	sock;
	struct batch_reply *preply;
{
	DIS_tcp_setup(sock);

	return  (decode_DIS_replySvr(sock, preply));
}
