/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * queue_func.c - various functions dealing with queues
 *
 * Included functions are:
 *	que_alloc()	- allocacte and initialize space for queue structure
 *	que_free()	- free queue structure
 *	que_purge()	- remove queue from server
 *	find_queuebyname() - find a queue with a given name
 *	get_dfltque()	- get default queue
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/param.h>
#include <memory.h>
#include "pbs_ifl.h"
#include <errno.h>
#include <string.h>
#include "list_link.h"
#include "log.h"
#include "attribute.h"
#include "server_limits.h"
#include "server.h"
#include "queue.h"
#include "pbs_error.h"
#if __STDC__ != 1
#include <memory.h>
#endif

static char ident[] = "@(#) $RCSfile: queue_func.c,v $ $Revision: 2.1 $";

/* Global Data */

extern char     *msg_err_unlink;
extern char	*path_queues;
extern struct    server server;
extern list_head svr_queues;

/*
 * que_alloc - allocate space for a queue structure and initialize 
 *	attributes to "unset"
 *
 *	Returns: pointer to structure or null is space not available.
 */

pbs_queue *que_alloc(name)
	char *name;
{
	int        i;
	pbs_queue *pq;


	pq = (pbs_queue *)malloc(sizeof (pbs_queue));
	if (pq == (pbs_queue *)0) {
		log_err(errno, "que_alloc", "no memory");
		return ((pbs_queue *)0);
	}
	(void)memset((char *)pq, (int)0, (size_t)sizeof(pbs_queue));
	pq->qu_qs.qu_type = QTYPE_Unset;
	CLEAR_HEAD(pq->qu_jobs);
	CLEAR_LINK(pq->qu_link);

	strncpy(pq->qu_qs.qu_name, name, PBS_MAXQUEUENAME);
	append_link(&svr_queues, &pq->qu_link, pq);
	server.sv_qs.sv_numque++;

	/* set the working attributes to "unspecified" */

	for (i=0; i<(int)QA_ATR_LAST; i++) {
		clear_attr(&pq->qu_attr[i], &que_attr_def[i]);
	}

	return (pq);
}


/*
 * que_free - free queue structure and its various sub-structures
 */

void que_free (pq)
	pbs_queue	*pq;
{
	int		 i;
	attribute	*pattr;
	attribute_def	*pdef;
	
	/* remove any malloc working attribute space */

	for (i=0; i < (int)QA_ATR_LAST; i++) {
		pdef  = &que_attr_def[i];
		pattr = &pq->qu_attr[i];

		pdef->at_free(pattr);

		/* remove any acl lists associated with the queue */

		if (pdef->at_type == ATR_TYPE_ACL)  {
			pattr->at_flags |= ATR_VFLAG_MODIFY;
			save_acl(pattr, pdef, pdef->at_name, pq->qu_qs.qu_name);
		}
	}

	/* now free the main structure */

	server.sv_qs.sv_numque--;
	delete_link(&pq->qu_link);
	(void)free((char *)pq);
}



/*
 * que_purge - purge queue from system
 *
 * The queue is dequeued, the queue file is unlinked.
 * If the queue contains any jobs, the purge is not allowed.
 */

int que_purge(pque)
	pbs_queue *pque;
{
	char     namebuf[MAXPATHLEN];

	if (pque->qu_numjobs != 0)
		return (PBSE_QUEBUSY);

	
	(void)strcpy(namebuf, path_queues);	/* delete queue file */
	(void)strcat(namebuf, pque->qu_qs.qu_name);
	if (unlink(namebuf) < 0) {
		(void)sprintf(log_buffer, msg_err_unlink, "Queue", namebuf);
		log_err(errno, "queue_purge", log_buffer);
	}

	que_free(pque);

	return (0);
}

/*
 * find_queuebyname() - find a queue by its name
 */

pbs_queue *find_queuebyname(quename)
	char *quename;
{
	char  *pc;
	pbs_queue *pque;
	char   qname[PBS_MAXDEST + 1];

	(void)strncpy(qname, quename, PBS_MAXDEST);
	qname[PBS_MAXDEST] ='\0';
	pc = strchr(qname, (int)'@');	/* strip off server (fragment) */
	if (pc)
		*pc = '\0';
	pque = (pbs_queue *)GET_NEXT(svr_queues);
	while (pque != (pbs_queue *)0) {
		if (strcmp(qname, pque->qu_qs.qu_name) == 0)
			break;
		pque = (pbs_queue *)GET_NEXT(pque->qu_link);
	}
	if (pc)
		*pc = '@';	/* restore '@' server portion */
	return (pque);
}

/*
 * get_dftque - get the default queue (if declared)
 */

pbs_queue *get_dfltque()
{
	pbs_queue *pq = (pbs_queue *)0;

	if (server.sv_attr[SRV_ATR_dflt_que].at_flags & ATR_VFLAG_SET)
		pq = find_queuebyname(server.sv_attr[SRV_ATR_dflt_que].at_val.at_str);
	return (pq);
}
