/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * queue_recov.c - This file contains the functions to record a queue
 *	data struture to disk and to recover it from disk.
 *
 *	The data is recorded in a file whose name is the queue name
 *
 *	The following public functions are provided:
 *		que_save()  - save the disk image 
 *		que_recov()  - recover (read) queue from disk
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include <sys/param.h>
#include "pbs_ifl.h"
#include <errno.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "queue.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: queue_recov.c,v $ $Revision: 2.1 $";

/*
 * the following funny business is due to the fact that O_SYNC 
 * is not currently POSIX
 */
#ifdef O_SYNC
#define O_Sync O_SYNC
#elif _FSYNC
#define O_Sync _FSYNC
#else
#define O_Sync 0
#endif

/* data global to this file */

extern char *path_queues;

/*
 * que_save() - Saves a queue structure image on disk
 *
 *
 *	For a save, to insure no data is ever lost due to system crash:
 *	1. write (with O_SYNC) new image to a new file using a temp name
 *	2. unlink the old (image) file
 *	3. link the correct name to the new file
 *	4. unlink the temp name
 *	
 *	Then, if the queue has any access control lists, they are saved
 *	to their own files.
 */

int que_save (pque)
	pbs_queue	*pque;		/* pointer to queue structure */
{
	int	fds;
	int	i;
	char   *myid = "que_save";
	char	namebuf1[MAXPATHLEN];
	char	namebuf2[MAXPATHLEN];

	(void)strcpy(namebuf1, path_queues);
	(void)strcat(namebuf1, pque->qu_qs.qu_name);
	(void)strcpy(namebuf2, namebuf1);
	(void)strcat(namebuf2, ".new");
	fds = open(namebuf2, O_CREAT | O_WRONLY | O_Sync, 0600);
	if (fds < 0) {
		log_err(errno, myid, "open error");
		return (-1);
	}

	/* set up save buffering system */

	save_setup(fds);

	/* save basic queue structure (fixed length stuff) */

	if (save_struct((char *)&pque->qu_qs, sizeof (struct queuefix)) != 0) {
		log_err(-1, myid, "save_struct failed");
		(void)close(fds);
		return (-1);
	}

	/* save queue attributes  */

	if (save_attr(que_attr_def,pque->qu_attr,(int)QA_ATR_LAST) != 0) {
		log_err(-1, myid, "save_attr failed");
		(void)close(fds);
		return (-1);
	}

	if (save_flush() != 0) {	/* flush the save buffer */
		log_err(-1, myid,"save_flush failed");
		(void)close(fds);
		return (-1);
	}

	(void)close(fds);

	(void)unlink(namebuf1);	
	if (link(namebuf2, namebuf1) < 0) {
		log_err(errno, myid, "unable to link queue name");
	} else {
		(void)unlink(namebuf2);
	}
	
	/* 
	 * now search queue's attributes for access control lists,
	 * they are saved separately in their own file:
	 * ../priv/(attr name)/(queue name)
	 */

	for (i=0; i < (int)QA_ATR_LAST; i++) {
		if (pque->qu_attr[i].at_type == ATR_TYPE_ACL) {
			save_acl(&pque->qu_attr[i], &que_attr_def[i],
				 que_attr_def[i].at_name, pque->qu_qs.qu_name);
		}
	}
	
	return (0);
}

/*
 * que_recov() - load (recover) a queue from its save file
 *
 *	This function is only needed upon server start up.
 *
 *	The queue structure is recovered from the disk.
 *	Space to hold the above is malloc-ed as needed.
 *
 *	Returns: pointer to new queue structure if successful
 *		 null if error
 */

pbs_queue *que_recov(filename)
	char   *filename;	/* pathname to queue save file */
{
	int	   fds;
	int	   i;
	pbs_queue *pq;
	char	   namebuf[MAXPATHLEN];


	pq = que_alloc(filename);  /* allocate & init queue structure space */
	if (pq == (pbs_queue *)0) {
		log_err(-1, "que_recov", "que_alloc failed");
		return ((pbs_queue *)0);
	}

	(void)strcpy(namebuf, path_queues);
	(void)strcat(namebuf, filename);
	fds = open(namebuf, O_RDONLY, 0);
	if (fds < 0) {
		log_err(errno, "que_recov", "open error");
		free((char *)pq);
		return ((pbs_queue *)0);
	}

	/* read in queue save sub-structure */

	if (read(fds, (char *)&pq->qu_qs, sizeof(struct queuefix)) !=
	    sizeof(struct queuefix)) {
		log_err(errno, "que_recov", "read error");
		free((char *)pq);
		(void)close(fds);
		return ((pbs_queue *)0);
	}

	/* read in queue attributes */

	if (recov_attr(fds, pq, que_attr_def, pq->qu_attr,
	    (int)QA_ATR_LAST, 0) != 0) {
		log_err(-1, "que_recov", "recov_attr[common] failed");
		que_free(pq);
		(void)close(fds);
		return ((pbs_queue *)0);
	}

	/*
	 * now reload the access control lists, these attributes were
	 * saved separately
	 */

	for (i=0; i < (int)QA_ATR_LAST; i++) {
		if (pq->qu_attr[i].at_type == ATR_TYPE_ACL) {
			recov_acl(&pq->qu_attr[i], &que_attr_def[i],
			         que_attr_def[i].at_name, pq->qu_qs.qu_name);
		}
	}
	
	/* all done recovering the queue */

	(void)close(fds);
	return (pq);
}
