/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_holdjob.c 
 *
 * Functions relating to the Hold and Release Job Batch Requests.
 *
 * Included funtions are:
 *	req_holdjob()
 *	req_releasejob()
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include "libpbs.h"
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "work_task.h"
#include "pbs_error.h"
#include "log.h"
#include "acct.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: req_holdjob.c,v $ $Revision: 2.1 $";

/* Private Functions Local to this file */

static int get_hold A_((list_head *, char **));
static void post_hold A_((struct work_task *));

/* Global Data Items: */

extern struct server server;
extern char	*msg_jobholdset;
extern char	*msg_jobholdrel;
extern char	*msg_mombadhold;
extern char	*msg_postmomnojob;
extern time_t	 time_now;

int chk_hold_priv A_((long val, int perm));

/* Private Data */

static attribute temphold;

/*
 * chk_hold_priv - check that client has privilege to set/clear hold
 */

int chk_hold_priv(val, perm)
	long	val;	/* hold bits being changed */
	int	perm;	/* client privilege */
{
	if ( (val & HOLD_s) && ((perm & ATR_DFLAG_MGWR) == 0) )
		return (PBSE_PERM);
	if ( (val & HOLD_o) && ((perm & (ATR_DFLAG_MGWR|ATR_DFLAG_OPWR)) == 0) )
		return (PBSE_PERM);
	return (PBSE_NONE);
}


/*
 * req_holdjob - service the Hold Job Request
 *
 *	This request sets one or more holds on a job.
 *	The state of the job may change as a result.
 */

void req_holdjob(preq)
	struct batch_request *preq;
{
	long		*hold_val;
	int		 newstate;
	int		 newsub;
	long		 old_hold;
	job		*pjob;
	char		*pset;
	int		 rc;

	pjob = chk_job_request(preq->rq_ind.rq_hold.rq_orig.rq_objname, preq);
	if (pjob == (job *)0)
		return;

	/* cannot do anything until we decode the holds to be set */

	if ( (rc=get_hold(&preq->rq_ind.rq_hold.rq_orig.rq_attr, &pset)) != 0) {
		req_reject(rc, 0, preq);
		return;
	}
			
	/* if other than HOLD_u is being set, must have privil */

	if ((rc = chk_hold_priv(temphold.at_val.at_long, preq->rq_perm)) != 0) {
			req_reject(rc, 0, preq);
			return;
	}

	hold_val = &pjob->ji_wattr[(int)JOB_ATR_hold].at_val.at_long;
	old_hold = *hold_val;
	*hold_val |= temphold.at_val.at_long;
	pjob->ji_wattr[(int)JOB_ATR_hold].at_flags |= ATR_VFLAG_SET;
	(void)sprintf(log_buffer, msg_jobholdset, pset, preq->rq_user,
		      preq->rq_host);

	if ( (pjob->ji_qs.ji_state == JOB_STATE_RUNNING) &&
	     (pjob->ji_wattr[(int)JOB_ATR_chkpnt].at_val.at_str) &&
	     (*pjob->ji_wattr[(int)JOB_ATR_chkpnt].at_val.at_str != 'n') ) {
	     
		/* have MOM attempt checkpointing */

		if ((rc = relay_to_mom(pjob->ji_qs.ji_un.ji_exect.ji_momaddr,
				       preq, post_hold)) != 0) {
			*hold_val = old_hold;	/* reset to the old value */
			req_reject(rc, 0, preq);
		} else {
			pjob->ji_qs.ji_substate = JOB_SUBSTATE_RERUN;
			pjob->ji_qs.ji_svrflags |=
					JOB_SVFLG_HASRUN | JOB_SVFLG_CHKPT;
			(void)job_save(pjob, SAVEJOB_QUICK);
			LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
				  pjob->ji_qs.ji_jobid, log_buffer);
		}
	} else {

		/* every thing went well, may need to update the job state */

		LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
			  pjob->ji_qs.ji_jobid, log_buffer);
		if (old_hold != *hold_val) {
			/* indicate attributes changed     */
			pjob->ji_modified = 1;	 
			svr_evaljobstate(pjob, &newstate, &newsub, 0);
			(void)svr_setjobstate(pjob, newstate, newsub);
		}
		reply_ack(preq);
	}
}


/*
 * req_releasejob - service the Release Job Request 
 *
 *	This request clears one or more holds on a job.
 *	As a result, the job might change state.
 */

void req_releasejob(preq)
	struct batch_request *preq;	/* ptr to the decoded request   */
{
	int		 newstate;
	int		 newsub;
	long		 old_hold;
	job		*pjob;
	char		*pset;
	int		 rc;


	pjob = chk_job_request(preq->rq_ind.rq_release.rq_objname, preq);
	if (pjob == (job *)0)
		return;

	/* cannot do anything until we decode the holds to be set */

	if ( (rc=get_hold(&preq->rq_ind.rq_hold.rq_orig.rq_attr, &pset)) != 0) {
		req_reject(rc, 0, preq);
		return;
	}
			
	/* if other than HOLD_u is being released, must have privil */

	if ((rc = chk_hold_priv(temphold.at_val.at_long, preq->rq_perm)) != 0) {
			req_reject(rc, 0, preq);
			return;
	}

	/* all ok so far, unset the hold */

	old_hold = pjob->ji_wattr[(int)JOB_ATR_hold].at_val.at_long;
	if (rc = job_attr_def[(int)JOB_ATR_hold].at_set(&pjob->ji_wattr[(int)JOB_ATR_hold], &temphold, DECR)) {
		req_reject(rc, 0, preq);
		return;
	}

	/* every thing went well, if holds changed, update the job state */

	if (old_hold != pjob->ji_wattr[(int)JOB_ATR_hold].at_val.at_long) {
		pjob->ji_modified = 1;	/* indicates attributes changed    */
		svr_evaljobstate(pjob, &newstate, &newsub, 0);
		(void)svr_setjobstate(pjob, newstate, newsub); /* saves job */
	}
	(void)sprintf(log_buffer, msg_jobholdrel, pset, preq->rq_user,
		      preq->rq_host);
	LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, pjob->ji_qs.ji_jobid,
		  log_buffer);
	reply_ack(preq);
}

/*
 * get_hold - search a list of attributes (svrattrl) for the hold-types
 * 	attribute.  This is used by the Hold Job and Release Job request,
 *	therefore it is an error if the hold-types attribute is not present,
 *	or there is more than one.
 *
 *	Decode the hold attribute into temphold.
 */

static int get_hold(phead, pset)
	list_head *phead;
	char	 **pset;	/* RETURN - ptr to hold value */
{
	int		 have_one = 0;
	struct svrattrl *holdattr;
	struct svrattrl *pal;

	pal = (struct svrattrl *)GET_NEXT((*phead));
	while (pal) {
		if (!strcmp(pal->al_name,job_attr_def[(int)JOB_ATR_hold].at_name)) {
			holdattr = pal;
			*pset    = pal->al_value;
			have_one++;
		} else {
			return (PBSE_IVALREQ);
		}
		pal = (struct svrattrl *)GET_NEXT(pal->al_link);
	}
	if (have_one != 1)
		return (PBSE_IVALREQ);

	/* decode into temporary attribute structure */

	clear_attr(&temphold, &job_attr_def[(int)JOB_ATR_hold]);
	return ( job_attr_def[(int)JOB_ATR_hold].at_decode(
					&temphold,
					holdattr->al_name,
					(char *)0,
					holdattr->al_value) );
}


/*
 * post_hold - A round hold in the ground in which a post is placed :-)
 *	called when a hold request was sent to MOM and the answer
 *	is received.  Completes the hold request for running jobs.
 */

static void post_hold(pwt)
	struct work_task *pwt;
{
	int		      code;
	svrattrl	     *pal;
	job		     *pjob;
	struct batch_request *preq;
	int		      rc;
	attribute	      temp;

	svr_disconnect(pwt->wt_event);	/* close connection to MOM */
	preq = pwt->wt_parm1;
	code = preq->rq_reply.brp_code;
	preq->rq_conn = preq->rq_orgconn;	/* restore client socket */

	if ((pjob = find_job(preq->rq_ind.rq_hold.rq_orig.rq_objname)) == (job *)0) {
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_JOB,
			  preq->rq_ind.rq_hold.rq_orig.rq_objname,
			  msg_postmomnojob);
		req_reject(PBSE_UNKJOBID, 0, preq);
		return;
	}
	if (code != 0) {
	    pjob->ji_qs.ji_substate = JOB_SUBSTATE_RUNNING;	/* reset it */
	    if (code != PBSE_NOSUP) {
		(void)sprintf(log_buffer, msg_mombadhold, code);
		LOG_EVENT(PBSEVENT_DEBUG, PBS_EVENTCLASS_JOB,
			  pjob->ji_qs.ji_jobid, log_buffer);
		req_reject(code, 0, preq);
		return;
	    }
	} else if (code == 0) {

		/* record that MOM has a checkpoint file */

		if (preq->rq_reply.brp_auxcode)	/* chkpt can be moved */
		    pjob->ji_qs.ji_svrflags =
				(pjob->ji_qs.ji_svrflags & ~JOB_SVFLG_CHKPT) |
				JOB_SVFLG_HASRUN | JOB_SVFLG_ChkptMig;

		pjob->ji_modified = 1;	  /* indicate attributes changed     */
		(void)job_save(pjob, SAVEJOB_QUICK);

		/* note in accounting file */

		account_record(PBS_ACCT_CHKPNT, pjob, (char *)0);
	}
	reply_ack(preq);
}
