/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_modify.c 
 *
 * Functions relating to the Modify Job Batch Requests.
 *
 * Included funtions are:
 *	
 *
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include "libpbs.h"
#include <signal.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "resource.h"
#include "server.h"
#include "queue.h"
#include "credential.h"
#include "batch_request.h"
#include "job.h"
#include "work_task.h"
#include "pbs_error.h"
#include "log.h"
#include "svrfunc.h"

static char ident[] = "@(#) $RCSfile: req_modify.c,v $ $Revision: 2.2 $";

/* Global Data Items: */

extern attribute_def	    job_attr_def[];
extern char *msg_jobmod;
extern char *msg_manager;
extern char *msg_mombadmodify;
extern int   comp_resc_gt;
extern int   comp_resc_lt;

/*
 * post_modify_req - clean up after sending modify request to MOM
 */
static void post_modify_req(pwt)
	struct work_task *pwt;
{
	struct batch_request *preq;

	svr_disconnect(pwt->wt_event);  /* close connection to MOM */
	preq = pwt->wt_parm1;
	preq->rq_conn = preq->rq_orgconn;  /* restore socket to client */

	if (preq->rq_reply.brp_code) {
	    (void)sprintf(log_buffer, msg_mombadmodify,preq->rq_reply.brp_code);
	    log_event(PBSEVENT_JOB, PBS_EVENTCLASS_JOB,
		      preq->rq_ind.rq_modify.rq_objname, log_buffer);
	    req_reject(preq->rq_reply.brp_code, 0, preq);
	} else
	    reply_ack(preq);
}

	


/*
 * req_modifyjob - service the Modify Job Request
 *
 *	This request modifys a job's attributes.
 */

void req_modifyjob(preq)
	struct batch_request *preq;
{
	int		 bad = 0;
	int		 i;
	int		 newstate;
	int		 newsubstate;
	job		*pjob;
	svrattrl	*plist;
	resource_def	*prsd;
	int		 rc;
	int		 sendmom = 0;

	pjob = chk_job_request(preq->rq_ind.rq_modify.rq_objname, preq);
	if (pjob == (job *)0)
		return;

	/* cannot be in exiting or transit, exiting has already be checked */

	if (pjob->ji_qs.ji_state == JOB_STATE_TRANSIT) {
		req_reject(PBSE_BADSTATE, 0, preq);
		return;
	}

	plist = (svrattrl *)GET_NEXT(preq->rq_ind.rq_modify.rq_attr);
	if (plist == (svrattrl *)0) {	/* nothing to do */
		reply_ack(preq);
		return;
	}

	/* if job is running, special checks must be made */

	if (pjob->ji_qs.ji_state == JOB_STATE_RUNNING) {
	    while (plist) {

		/* are the attribute modifiable in RUN state ? */

		i = find_attr(job_attr_def,plist->al_name,JOB_ATR_LAST);
		if ((i < 0) ||
		    ((job_attr_def[i].at_flags & ATR_DFLAG_ALTRUN) == 0))  {

			reply_badattr(PBSE_MODATRRUN, 1, plist, preq);
			return;
		}
		if (i == (int)JOB_ATR_resource) {

		    /* is the specified resource modifiable while */
		    /* the job is running                         */

		    prsd = find_resc_def(svr_resc_def, plist->al_resc,
					 svr_resc_size);
		    if (prsd == 0) {
			reply_badattr(PBSE_UNKRESC, 1, plist, preq);
			return;
		    }
		    if ((prsd->rs_flags & ATR_DFLAG_ALTRUN) == 0) {
			reply_badattr(PBSE_MODATRRUN, 1, plist, preq);
			return;
		    }

		    sendmom = 1;
		}
		plist = (svrattrl *)GET_NEXT(plist->al_link);
	    }
	}

	/* modify the jobs attributes */

	bad = 0;
	plist = (svrattrl *)GET_NEXT(preq->rq_ind.rq_modify.rq_attr);
	rc = modify_job_attr(pjob, plist, preq->rq_perm, &bad);
	if (rc) {
		reply_badattr(rc, bad, plist, preq);
		return;
	}

	/* Reset any defaults resource limit which might have been unset */

	set_resc_deflt(pjob);

	/* if job is not running, may need to change its state */

	if (pjob->ji_qs.ji_state != JOB_STATE_RUNNING) {
		svr_evaljobstate(pjob, &newstate, &newsubstate, 0);
		(void)svr_setjobstate(pjob, newstate, newsubstate);
	} else {
		(void)job_save(pjob, SAVEJOB_FULL);
	}
	(void)sprintf(log_buffer, msg_manager, msg_jobmod,
		      preq->rq_user, preq->rq_host);
	LOG_EVENT(PBSEVENT_JOB, PBS_EVENTCLASS_JOB, pjob->ji_qs.ji_jobid,
		  log_buffer);

	/* if a resource limit changed for a running job, send to MOM */

	if (sendmom) {
		if (rc = relay_to_mom(pjob->ji_qs.ji_un.ji_exect.ji_momaddr,
				      preq, post_modify_req))
			req_reject(rc, 0, preq);    /* unable to get to MOM */
		return;
	}

	reply_ack(preq);
}


/*
 * modify_job_attr - modify the attributes of a job attomically
 *	Used by req_modifyjob() to alter the job attributes and by
 *	stat_update() [see req_stat.c] to update with latest from MOM
 */
int modify_job_attr(pjob, plist, perm, bad)
	job	  *pjob;
	svrattrl  *plist;
	int	   perm;
	int	  *bad;
{
	int	   allow_unkn;
	long	   i;
	attribute  newattr[(int)JOB_ATR_LAST];
	attribute *pattr;
	int	   rc;

	if (pjob->ji_qhdr->qu_qs.qu_type == QTYPE_Execution)
		allow_unkn = -1;
	else
		allow_unkn = (int)JOB_ATR_UNKN;

	pattr = pjob->ji_wattr;

	/* call attr_atomic_set to decode and set a copy of the attributes */

	rc = attr_atomic_set(plist, pattr, newattr, job_attr_def, JOB_ATR_LAST,
			     allow_unkn, perm, bad);

	/* If resource limits are being changing ... */

	if ( (rc == 0) &&
	     (newattr[(int)JOB_ATR_resource].at_flags & ATR_VFLAG_SET) &&
	     ((perm & (ATR_DFLAG_MGWR | ATR_DFLAG_OPWR)) == 0) ) {

		/* If job is running, only manager/operator can raise limits */

		if (pjob->ji_qs.ji_state == JOB_STATE_RUNNING) {

		    if ( (comp_resc(&pjob->ji_wattr[(int)JOB_ATR_resource],
				    &newattr[(int)JOB_ATR_resource]) == -1) ||
			 comp_resc_lt )
				rc = PBSE_PERM;
				 
		}
		/* Also check against queue and system limits */

		rc =  chk_resc_limits(&newattr[(int)JOB_ATR_resource],
				      pjob->ji_qhdr);
	}

	/* special check on permissions for hold */

	if (newattr[(int)JOB_ATR_hold].at_flags & ATR_VFLAG_MODIFY) {

		i = newattr[(int)JOB_ATR_hold].at_val.at_long ^ 
		    (pattr+(int)JOB_ATR_hold)->at_val.at_long;
		rc = chk_hold_priv(i, perm);
	}

	if (rc == 0) {
	    for (i=0; i<JOB_ATR_LAST; i++) {
		if (newattr[i].at_flags & ATR_VFLAG_MODIFY) {
			if (job_attr_def[i].at_action) {
				rc = job_attr_def[i].at_action(&newattr[i],
					        pjob, ATR_ACTION_ALTER);
				if (rc)
					break;
			}
		}
	    }
	    if ((newattr[(int)JOB_ATR_userlst].at_flags & ATR_VFLAG_MODIFY) ||
                (newattr[(int)JOB_ATR_grouplst].at_flags & ATR_VFLAG_MODIFY)) {
		/* Need to reset execution uid and gid */
		rc = set_jobexid(pjob, newattr);
	    }
			
	}
	if (rc) {
		for (i=0; i<JOB_ATR_LAST; i++)
			job_attr_def[i].at_free(newattr+i);
		return (rc);
	}


	/* OK, now copy the new values into the job attribute array */

	for (i=0; i<JOB_ATR_LAST; i++) {
		if (newattr[i].at_flags & ATR_VFLAG_MODIFY) {
			job_attr_def[i].at_free(pattr+i);
			if ((newattr[i].at_type == ATR_TYPE_LIST) ||
			    (newattr[i].at_type == ATR_TYPE_RESC)) {
				list_move(&newattr[i].at_val.at_list, 
					  &(pattr+i)->at_val.at_list);
			} else {
				*(pattr+i) = newattr[i];
			}
			(pattr+i)->at_flags = newattr[i].at_flags;
		}
	}
	/* note, the newattr[] attributes are on the stack, they goaway auto */

	pjob->ji_modified = 1;
	return (0);
}
