/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_rescq.c 
 *
 * Functions relating to the Resource Query Batch Request.
 *
 * Included funtions are:
 *	
 *
 */
#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include "libpbs.h"
#include <ctype.h>
#include <errno.h>
#include <stdlib.h>
#include <string.h>
#include "server_limits.h"
#include "list_link.h"
#include "attribute.h"
#include "server.h"
#include "credential.h"
#include "batch_request.h"
#include "pbs_error.h"
#include "svrfunc.h"
#include "log.h"

static char ident[] = "@(#) $RCSfile: req_rescq.c,v $ $Revision: 2.1 $";

/* Global Data Items: */

resource_t next_resource_tag;

/*
 * req_rescq - Resource Query
 *
 *	This responds to a Resource Query request with the
 *	amount of the resource(s) available/allocated/reserved/down.
 *
 *	Currently only recognizes "nodes"
 */

void req_rescq(preq)
	struct batch_request *preq;
{
	int i;
	char *pn;
	char *pv;
	int rc;
	struct brp_rescq *preply;
	struct rq_rescq *prescq = &preq->rq_ind.rq_rescq;

	if (prescq->rq_num < 1) {
		req_reject(RM_ERR_BADPARAM, 0, preq);
		return;
	}
	preply = &preq->rq_reply.brp_un.brp_rescq;
	preq->rq_reply.brp_choice = BATCH_REPLY_CHOICE_RescQuery;
	preply->brq_number = prescq->rq_num;

	preply->brq_avail = (int *)malloc( prescq->rq_num * sizeof (int) );
	preply->brq_alloc = (int *)malloc( prescq->rq_num * sizeof (int) );
	preply->brq_resvd = (int *)malloc( prescq->rq_num * sizeof (int) );
	preply->brq_down  = (int *)malloc( prescq->rq_num * sizeof (int) );
	if ( (preply->brq_avail == 0) || (preply->brq_alloc == 0) ||
	     (preply->brq_resvd == 0) || (preply->brq_down  == 0) ) {
		req_reject(PBSE_SYSTEM, errno, preq);
		return;
	}


	for (i=0; i<prescq->rq_num; ++i) {

		*(preply->brq_resvd + i) = 0;
		*(preply->brq_down + i) = 0;
		
		pn = *(prescq->rq_list + i);
		while (isspace((int)*pn))
			++pn;

		pv = pn;
		
		while (isalnum((int)*pv))
			++pv;
		if (*pv != '\0')
			*pv++ = '\0';		/* skip delimiter */

		if (strcmp(pn, "nodes") == 0) {
			/* check on availability of nodes */

			if ((rc = node_avail(pv, preply->brq_avail+i,
					         preply->brq_alloc+i,
					         preply->brq_resvd+i,
					         preply->brq_down+i)) != 0) {
				req_reject(rc, 0, preq);
				return;
			}
		} else {
			req_reject(RM_ERR_BADPARAM, 0, preq);
			return;
		}
	
	}
	(void)reply_send(preq);
}

/*
 * req_rescreserve - Resource reserve
 *
 *	Reserve a set of resources (only "nodes" for now)
 */

void req_rescreserve(preq)
	struct batch_request *preq;
{
	int	i;
	int	freeold = 0;
	struct rq_rescq *prescq = &preq->rq_ind.rq_rescq;
	int     part = 0;
	char   *pn;
	char   *pv;
	int     rc;
	resource_t tag;

	if ((preq->rq_perm & ( ATR_DFLAG_MGWR | ATR_DFLAG_OPWR )) == 0) {
		req_reject(PBSE_PERM, 0, preq);
		return;
	}

	if (prescq->rq_num < 1) {
		req_reject(RM_ERR_BADPARAM, 0, preq);
		return;
	}

	tag = prescq->rq_rhandle;

	if (tag > 0) {
		freeold = 1;
	} else {
		tag = next_resource_tag++;
	}

	for (i=0; i<prescq->rq_num; ++i) {
		
		pn = *(prescq->rq_list + i);
		while (isspace((int)*pn))
			++pn;
		pv = pn;
		while (isalnum((int)*pv))
			++pv;
		if (*pv != '\0')
			*pv++ = '\0';           /* skip delimiter */

		if (strcmp(pn, "nodes") == 0) {

			/* If existing (partial) reservation, free it */

			if (freeold == 1)
				node_unreserve(tag);

			/* attempt to reserve some nodes */

			rc = node_reserve(pv, tag);
			if (rc < 0) {
				req_reject(rc, 0, preq);
				return;

			} else if (rc == 0)
				part = 1;

			continue;
		} else {
			req_reject(PBSE_RMUNKNOWN, 0, preq);
			return;
		}
	}

	 if (part)
		preq->rq_reply.brp_code = PBSE_RMPART;
	else
		preq->rq_reply.brp_code = 0;
	preq->rq_reply.brp_auxcode = tag;
	reply_send(preq);
}

/*
 * req_rescfree - Free a reserved set of resources
 */

void req_rescfree(preq)
	struct batch_request *preq;
{
	if ((preq->rq_perm & ( ATR_DFLAG_MGWR | ATR_DFLAG_OPWR )) == 0) {
		req_reject(PBSE_PERM, 0, preq);
		return;
	}

	node_unreserve(preq->rq_ind.rq_rescq.rq_rhandle);

	reply_ack(preq);
}
