/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/
/*
 * svr_resc_def is the array of resource definitions for the server.
 * Each legal server resource is defined here.
 */

#include <pbs_config.h>   /* the master config generated by configure */

#include <ctype.h>
#include "pbs_ifl.h" 
#include <string.h>
#include "list_link.h" 
#include "attribute.h" 
#include "resource.h" 
#include "pbs_error.h"

static char ident[] = "@(#) $RCSfile: resc_def_all.c,v $ $Revision: 2.4 $";

/*
 * The entries for each attribute are (see attribute.h):
 *	name,
 *	decode function,
 *	encode function,
 *	set function,
 *	compare function,
 *	free value space function,
 *	access permission flags,
 *	value type
 */

static int decode_nodes A_((struct attribute *, char *, char *, char *));
static int set_node_ct A_((resource *, attribute *, int actmode));


resource_def svr_resc_def[] = {

    {	"arch",				/* system architecture type */
	decode_str,
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	READ_WRITE,
	ATR_TYPE_STR
    },
    {	"cpupercent",
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	NO_USER_SET,
	ATR_TYPE_LONG
    },
    {	"cput",
	decode_time,
	encode_time,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
	ATR_TYPE_LONG
    },
    {	"file",
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM,
        ATR_TYPE_SIZE
    },
    {	"mem",
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN | ATR_DFLAG_RASSN,
        ATR_TYPE_SIZE
    },
    {	"pmem",
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN | ATR_DFLAG_RMOMIG,
        ATR_TYPE_SIZE
    },
    {	"ncpus",                        /* number of processors for job */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_RMOMIG | ATR_DFLAG_RASSN,
	ATR_TYPE_LONG
    },
    {	"vmem",
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN | ATR_DFLAG_RASSN,
        ATR_TYPE_SIZE
    },
    {	"pvmem",
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_RMOMIG,
        ATR_TYPE_SIZE
    },
    {	"nice",				/* job nice value */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"pcput",
	decode_time,
	encode_time,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"walltime",			/* wall clock time limit for a job */
	decode_time,
	encode_time,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
	ATR_TYPE_LONG
    },
#if NODEMASK != 0
    {	"nodemask",
	decode_ll,
	encode_ll,
	set_ll,
	comp_ll,
	free_null,
	NULL_FUNC,
	NO_USER_SET | ATR_DFLAG_MOM | ATR_DFLAG_RMOMIG,
	ATR_TYPE_LL
    },
#endif /*NODEMASK*/
    {	"host",				/* host to execute on */
	decode_str,
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	READ_WRITE,
	ATR_TYPE_STR
    },
    {	"nodes",			/* user specification of nodes */
	decode_nodes,
	encode_str,
	set_str,
	comp_str,
	free_str,
	set_node_ct,
	READ_WRITE | ATR_DFLAG_RMOMIG,
	ATR_TYPE_STR
    },
    {	"neednodes",			/* scheduler modified specification */
	decode_str,			/* of nodes */
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	ATR_DFLAG_MGRD | ATR_DFLAG_MGWR | ATR_DFLAG_SvWR | ATR_DFLAG_RMOMIG | ATR_DFLAG_MOM,
	ATR_TYPE_STR
    },
    {	"nodect",			/* count of number of nodes requested */
	decode_l,			/* read-only, set by server whenever  */
	encode_l,			/* "nodes" is set, for use by sched   */
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_ONLY | ATR_DFLAG_MGWR | ATR_DFLAG_RASSN | ATR_DFLAG_RMOMIG,
	ATR_TYPE_LONG
    },
    {
	"mpplabel",			/* PE label required for execution */
	decode_str,
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	READ_WRITE,
	ATR_TYPE_STR
    },
    {
	"other",
	decode_str,
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	READ_WRITE,
	ATR_TYPE_STR
    },
    {	"software",			/* software required for execution */
	decode_str,
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	READ_WRITE,
	ATR_TYPE_STR
    },
    {	"taskspn",			/* number of tasks per node  	   */
	decode_l,			/* set by admin on queue or system */
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	NO_USER_SET | ATR_DFLAG_MOM | ATR_DFLAG_RMOMIG,
	ATR_TYPE_LONG
    },

	/* the following are found only on Cray systems */

    {	"mta",				/* mag tape, a class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mtb",				/* mag tape, b class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mtc",				/* mag tape, c class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mtd",				/* mag tape, d class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mte",				/* mag tape, e class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mtf",				/* mag tape, f class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mtg",				/* mag tape, g class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mth",				/* mag tape, h class */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"pf",				/* max file space for job */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
        ATR_TYPE_SIZE
    },
    {	"ppf",				/* max file space per process */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
        ATR_TYPE_SIZE
    },
    {	"sds",				/* max SDS for job */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
        ATR_TYPE_SIZE
    },
    {	"psds",				/* max SDS per process */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
        ATR_TYPE_SIZE
    },
    {	"procs",			/* number of processes per job */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mppe",				/* number of mpp nodes */
	decode_l,
	encode_l,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
    {	"mppt",				/* total mpp time for job */
	decode_time,
	encode_time,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
	ATR_TYPE_LONG
    },
    {	"pmppt",			/* max mpp time for any process */ 
	decode_time,
	encode_time,
	set_l,
	comp_l,
	free_null,
	NULL_FUNC,
	READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
	ATR_TYPE_LONG
    },
    {	"mppmem",			/* max mppmem memory for job */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM | ATR_DFLAG_ALTRUN,
        ATR_TYPE_SIZE
    },
#if	SRFS
    {	"srfs_tmp",			/* SRFS TMPDIR allocation */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM,
        ATR_TYPE_SIZE
    },
    {	"srfs_big",			/* SRFS BIGDIR allocation */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM,
        ATR_TYPE_SIZE
    },
    {	"srfs_fast",			/* SRFS FASTDIR allocation */
	decode_size,
        encode_size,
        set_size,
        comp_size,
        free_null,
	NULL_FUNC,
        READ_WRITE | ATR_DFLAG_MOM,
        ATR_TYPE_SIZE
    },
    {	"srfs_assist",			/* SRFS assist flag */
	decode_b,
	encode_b,
	set_b,
	comp_b,
	free_null,
	NULL_FUNC,
	ATR_DFLAG_OPRD | ATR_DFLAG_MGRD | ATR_DFLAG_SvRD | ATR_DFLAG_SvWR | ATR_DFLAG_MOM,
	ATR_TYPE_LONG
    },
#endif	/* SRFS */

#if PE_MASK != 0      /* PE mask on Cray T3e (similar to nodemask on SGI O2K */
    {
	"pe_mask",
	decode_str,
	encode_str,
	set_str,
	comp_str,
	free_str,
	NULL_FUNC,
	NO_USER_SET | ATR_DFLAG_MOM | ATR_DFLAG_RMOMIG,
	ATR_TYPE_STR
    },
#endif	/* PE_MASK */


	/* the definition for the "unknown" resource MUST be last */

    {	"|unknown|",
	decode_unkn,
	encode_unkn,
	set_unkn,
	comp_unkn,
	free_unkn,
	NULL_FUNC,
	READ_WRITE,
	ATR_TYPE_LIST
    }
	/* DO NOT ADD DEFINITIONS AFTER "unknown", ONLY BEFORE */
};

int svr_resc_size = sizeof (svr_resc_def) / sizeof (resource_def);


/*
 * decode_nodes - decode a node requirement specification, 
 *	Check if node requirement specification is syntactically ok,
 *	then call decode_str()
 *
 *	val if of the form:	node_spec[+node_spec...]
 *	where node_spec is:	number | properity | number:properity
 */

static int decode_nodes(patr, name, rescn, val)
	struct attribute *patr;
	char *name;             /* attribute name */
	char *rescn;            /* resource name - unused here */
	char *val;              /* attribute value */
{
	char *pc;

	pc = val;
	while (1) {
		while (isspace((int)*pc))
			++pc;

		if ( ! isalnum((int)*pc) )
			return (PBSE_BADATVAL);
		if (isdigit((int)*pc)) {
			while (isdigit((int)*++pc)) ;
			if (*pc == '\0')
				break;
			else if ((*pc != '+') && (*pc != ':') && (*pc != '#'))
				return (PBSE_BADATVAL);
		} else if (isalpha((int)*pc)) {
			while (isalnum((int)*++pc) || *pc == '-' || *pc == '.' || *pc == '=');
			if (*pc  == '\0')
				break;
			else if ((*pc != '+') && (*pc != ':') && (*pc != '#'))
				return (PBSE_BADATVAL);
		}
		++pc;
	}
	return (decode_str(patr, name, rescn, val));
}

/*
 * ctnodes = count nodes, turn node spec (see above) into a
 *	plain number of nodes.
 */

int ctnodes(spec)
	char *spec;
{
	int   ct = 0;
	char *pc;

	while (1) {

		while (isspace((int)*spec))
			++spec;

		if (isdigit((int)*spec))
			ct += atoi(spec);
		else
			++ct;
		if ((pc = strchr(spec, '+')) == (char *)0)
			break;
		spec = pc+1;
	}
	return (ct);
}

/*
 * set_node_ct = set node count
 *
 *	This is the "at_action" routine for the resource "nodes".
 *	When the resource_list attribute changes, then set/update
 *	the value of the resource "nodect" for use by the scheduler.
 */

static int set_node_ct(pnodesp, pattr, actmode)
	resource  *pnodesp;
	attribute *pattr;
	int	   actmode;
{
	resource	*pnct;
	resource_def	*pndef;

	if (actmode == ATR_ACTION_RECOV)
		return (0);

	/* Set "nodect" to count of nodes in "nodes" */

	pndef = find_resc_def(svr_resc_def, "nodect", svr_resc_size);
	if (pndef == (resource_def *)0)
		return (PBSE_SYSTEM);

	if ((pnct = find_resc_entry(pattr, pndef)) == (resource *)0) {
		if ((pnct = add_resource_entry(pattr, pndef)) == 0)
			return (PBSE_SYSTEM);
	}

	pnct->rs_value.at_val.at_long =
				ctnodes(pnodesp->rs_value.at_val.at_str);
	pnct->rs_value.at_flags |= ATR_VFLAG_SET;

	/* Set "neednodes" to "nodes", may be altered by Scheduler */

	pndef = find_resc_def(svr_resc_def, "neednodes", svr_resc_size);
	if (pndef == (resource_def *)0)
		return (PBSE_SYSTEM);

	if ((pnct = find_resc_entry(pattr, pndef)) == (resource *)0) {
		if ((pnct = add_resource_entry(pattr, pndef)) == 0)
			return (PBSE_SYSTEM);
	} else {
		pndef->rs_free(&pnct->rs_value);
	}
	pndef->rs_decode(&pnct->rs_value, (char *)0, (char *)0,
			 pnodesp->rs_value.at_val.at_str);
	pnct->rs_value.at_flags |= ATR_VFLAG_SET;
	
	return (0);
}
