/*
*         Portable Batch System (PBS) Software License
* 
* Copyright (c) 1999, MRJ Technology Solutions.
* All rights reserved.
* 
* Acknowledgment: The Portable Batch System Software was originally developed
* as a joint project between the Numerical Aerospace Simulation (NAS) Systems
* Division of NASA Ames Research Center and the National Energy Research
* Supercomputer Center (NERSC) of Lawrence Livermore National Laboratory.
* 
* Redistribution of the Portable Batch System Software and use in source
* and binary forms, with or without modification, are permitted provided
* that the following conditions are met:
* 
* - Redistributions of source code must retain the above copyright and
*   acknowledgment notices, this list of conditions and the following
*   disclaimer.
* 
* - Redistributions in binary form must reproduce the above copyright and 
*   acknowledgment notices, this list of conditions and the following
*   disclaimer in the documentation and/or other materials provided with the
*   distribution.
* 
* - All advertising materials mentioning features or use of this software must
*   display the following acknowledgment:
* 
*   This product includes software developed by NASA Ames Research Center,
*   Lawrence Livermore National Laboratory, and MRJ Technology Solutions.
* 
*         DISCLAIMER OF WARRANTY
* 
* THIS SOFTWARE IS PROVIDED BY MRJ TECHNOLOGY SOLUTIONS ("MRJ") "AS IS" WITHOUT 
* WARRANTY OF ANY KIND, AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING,
* BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY, FITNESS
* FOR A PARTICULAR PURPOSE, AND NON-INFRINGEMENT ARE EXPRESSLY DISCLAIMED.
* 
* IN NO EVENT, UNLESS REQUIRED BY APPLICABLE LAW, SHALL MRJ, NASA, NOR
* THE U.S. GOVERNMENT BE LIABLE FOR ANY DIRECT DAMAGES WHATSOEVER,
* NOR ANY INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
* DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
* OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER
* CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
* OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE
* USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
* 
* This license will be governed by the laws of the Commonwealth of Virginia,
* without reference to its choice of law rules.
*/

#include <pbs_config.h>   /* the master config generated by configure */

#include <sys/types.h>
#include "portability.h"
#include <errno.h>
#include <netinet/in.h>
#include "log.h"
#include "pbs_ifl.h"
#include "net_connect.h"
#include "svrfunc.h"
#include "sched_cmds.h"

static char ident[] = "@(#) $RCSfile: run_sched.c,v $ $Revision: 2.1 $";

/* Global Data */

extern pbs_net_t pbs_scheduler_addr;
extern unsigned int pbs_scheduler_port;
extern char      server_name[];
extern struct connection svr_conn[];
extern int	 svr_do_schedule;
extern char     *msg_sched_called;
extern char     *msg_sched_nocall;

int scheduler_sock = -1;
int scheduler_jobct = 0;


/* Functions private to this file */

static int  put_4byte A_((int sock, unsigned int val));
static void scheduler_close A_((int));

/*
 * contact_sched - open connection to the scheduler and send it a command
 */

static int contact_sched(cmd)
	int cmd;
{
	int sock;
	char *myid = "contact_sched";

	/* connect to the Scheduler */

	sock = client_to_svr(pbs_scheduler_addr, pbs_scheduler_port, 1);

	if (sock < 0) {
		log_err(errno, myid, msg_sched_nocall);
		return (-1);
	}
	add_conn(sock, FromClientDIS, pbs_scheduler_addr, pbs_scheduler_port, 
		 process_request);
	svr_conn[sock].cn_authen = PBS_NET_CONN_FROM_PRIVIL;
	net_add_close_func(sock, scheduler_close);

	/* send command to Scheduler */

	if (put_4byte(sock, cmd) < 0) {
		log_err(errno, myid, msg_sched_nocall);
		close_conn(sock);
		return (-1);
	}
	(void)sprintf(log_buffer, msg_sched_called, cmd);
	log_event(PBSEVENT_SCHED, PBS_EVENTCLASS_SERVER,server_name,log_buffer);
	return (sock);
}

/*
 * schedule_jobs - contact scheduler and direct it to run a scheduling cycle
 *	If a request is already outstanding, skip this one.
 *
 *	Returns: -1 = error
 *		  0 = scheduler notified
 *		 +1 = scheduler busy
 */

int schedule_jobs()
{
	int cmd;
	static int first_time = 1;

	if (first_time)
		cmd = SCH_SCHEDULE_FIRST;
	else
		cmd = svr_do_schedule;
	svr_do_schedule = SCH_SCHEDULE_NULL;

	if (scheduler_sock == -1) {
		scheduler_jobct = 0;
		if ((scheduler_sock = contact_sched(cmd)) < 0)
			return (-1);
		first_time = 0;
		return (0);
	} else
		return (1);	
}

/*
 * scheduler_close - connection to scheduler has closed, clear scheduler_called
 */

static void scheduler_close(sock)
	int sock;
{
	scheduler_sock = -1;

/*
 *	This bit of code is intended to support the scheduler - server - mom
 *	sequence.  A scheduler script may bes written to run only one job per
 *	cycle to  ensure its newly taken resources are considered by the
 *	scheduler before selecting another job.  In that case, rather than
 *	wait a full cycle before scheduling the next job, we check for
 *	one (and only one) job was run by the scheduler.  If true, then
 *	recycle the scheduler.
 */

	if (scheduler_jobct == 1) {
		/* recycle the scheduler */
		svr_do_schedule = SCH_SCHEDULE_RECYC;
	}
}

/*
 * put_4byte() - write a 4 byte integer in network order
 *
 *	Returns:  0 for sucess, -1 if error.
 */

static int put_4byte(sock, val)
	int		sock;	/* socket to read from */
	unsigned int	val;	/* 4 byte interger to write */
{
	
	int amt;
	union {
		int unl;
		char unc[sizeof (unsigned int)];
	} un;

	un.unl = htonl(val);
	amt = write(sock, (char *)(un.unc+sizeof(unsigned int)-4), 4);
	if (amt == 4)
		return (0);
	else 
		return (-1);
}
