/*-
 *  Copyright (c) 2001  Peter Pentchev
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 */

#include <sys/types.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "penv.h"
#include "pe_compat.h"

#define PE_BUFSIZ	BUFSIZ

__RINGID("$Ringlet: c/misc/penv/pe_compat.c,v 1.5 2004/01/06 16:38:10 roam Exp $");

#ifdef HAVE_FGETLN

pe_err_t
pe_getline(FILE *fp, char **pbuf, size_t *psz) {
	char *line, *buf;
	size_t all;
	
	if (line = fgetln(fp, &all), line == NULL) {
		/* EOF or error */
		*pbuf = NULL;
		return (PE_ERR_NONE);
	}
	if (buf = calloc(all, 1), buf == NULL)
		return (PE_ERR_NOMEM);
	memcpy(buf, line, all);
	*pbuf = buf;
	*psz = all;
	return (PE_ERR_NONE);
}

#else  /* HAVE_FGETLN */

pe_err_t
pe_getline(FILE *fp, char **pbuf, size_t *psz) {
	char *line, *buf;
	size_t ofs, all;
	buf = line = NULL;
	all = ofs = 0;
	for (;;) {
		/* Allocate the new chunk */
		if (line = realloc(buf, all + PE_BUFSIZ), line == NULL) {
			free(buf);
			return (PE_ERR_NOMEM);
		}
		buf = line;
		all += PE_BUFSIZ;

		/* Read.. */
		if (fgets(buf + ofs, all - ofs, fp) == NULL) {
			free(buf);
			buf = NULL;
			if (feof(fp))
				break;
			return (PE_ERR_CFG_READ);
		}
		/* Actually read something */
		ofs += strlen(buf + ofs);
		if ((ofs > 0) && (buf[ofs - 1] == '\n'))
			break;
	}

	*pbuf = buf;
	*psz = ofs;
	return (PE_ERR_NONE);
}

#endif /* HAVE_FGETLN */

#ifndef HAVE_STRLCPY
pe_err_t
pe_strlcpy(char *dst, const char *src, size_t sz) {

	strncpy(dst, src, sz);
	if (sz > 1)
		dst[sz - 1] = '\0';
	return (PE_ERR_NONE);
}
#endif
