/*-
 *  Copyright (c) 2001  Peter Pentchev
 *  All rights reserved.
 * 
 *  Redistribution and use in source and binary forms, with or without
 *  modification, are permitted provided that the following conditions
 *  are met:
 *  1. Redistributions of source code must retain the above copyright
 *     notice, this list of conditions and the following disclaimer.
 *  2. Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 * 
 *  THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
 *  ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 *  IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 *  ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
 *  FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 *  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 *  OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 *  HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 *  LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 *  OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 *  SUCH DAMAGE.
 */

#include <sys/types.h>

#include <errno.h>
#include <stdio.h>
#include <string.h>

#include "penv.h"
#include "pe_err.h"

__RINGID("$Ringlet: c/misc/penv/pe_err.c,v 1.10 2004/01/06 16:38:10 roam Exp $");

static const char *errs[PE_ERR_LAST] = {
	"no error",
	"out of memory",
	"bad command-line arguments",
	"getcwd(3) error",
	"basename(3) error",
	"dirname(3) error",
	"fstat(2) error",
	"exec(2) error",
	"opening config file",
	"reading config file",
	"syntax error in config file",
	"nonexistent variable name in config file",
	"invalid value for variable type",
	"string variable too long",
	"invalid command specified",
	"conflicting command actions specified",
	"creating directory",
	"changing to directory",
	"deleting file",
	"reading directory",
	"bad format in variable assignment",
	"creating file",
	"setting file mode",
	"INTERNAL ERROR"
};

static int errsn[PE_ERR_LAST] = {
	0, 0, 0,	/* NONE, NOMEM, CMDLINE */
	1, 1, 1,	/* GETCWD, BASENAME, DIRNAME */
	1, 1,		/* FSTAT, EXEC */
	1, 1,		/* CFG_OPEN, CFG_READ */
	0, 0, 0, 0,	/* CFG_SYNTAX, CFG_NXVAR, CFG_BADVAL, CFG_STRLONG */
	0, 0,		/* NOCMD, CONFCMD */
	1, 1, 1, 1,	/* MKDIR, CHDIR, UNLINK, OPENDIR */
	0,		/* VAR_FMT */
	1, 1,		/* FOPEN, FCHMOD */
	0		/* INT */
};

static char	errbuf[512];

/*
 * Function:
 * 	pe_strerror			- build an error message string
 * Inputs:
 * 	code				- error code
 * Returns:
 * 	pointer to static error message string
 * Modifies:
 * 	errbuf
 */

char *
pe_strerror(pe_err_t code) {
	
	if (code > PE_ERR_INT)
		code = PE_ERR_INT;

	if (errsn[code])
		snprintf(errbuf, sizeof(errbuf), "%s: %s", errs[code],
				strerror(errno));
	else
		snprintf(errbuf, sizeof(errbuf), "%s", errs[code]);
	
	return (errbuf);
}

/*
 * Function:
 * 	pe_prerror			- print error message and context
 * Inputs:
 * 	s				- context string to be printed
 * 	code				- error code
 * Returns:
 * 	code
 * Modifies:
 * 	nothing; writes to stderr
 */

pe_err_t
pe_prerror(const char *s, pe_err_t code) {
	
	if (code > PE_ERR_INT)
		code = PE_ERR_INT;
	
	fprintf(stderr, "penv error: %s: %s\n", s, pe_strerror(code));
	return (code);
}
