/* PIPServer - A deamon for finger protocol v2
 *
 * Copyright (C) 1999-2001 Michael Baumer <baumi@vis.ethz.ch>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <unistd.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <errno.h>

#include "in.fingerd.h"
#include "log.h"
#include "parser.h"

node_info* n_info[256];
int max_node = 1;

int getNodeType(char *id);
void parse_handler(Node *node);
void trim(char *str);

/* when a subnode is finished its callback func is called if it exists */

/* this is no longer done recursively */

/* Note on aborting the parsing:
//
// If a memory allocation fails or the input stream closes unexpectedly, 
// we immediately return from the function.
// 
// That behaviour simply aborts the current parsing.
// However, the already parsed tree stays intact.
// 
// We return 0 if the parsing failed
// and 1 for success
*/

int ParseNode(int fd, Node *node)
{
  char *scan;
  char *scan_tmp;
  char *inp;
  char *ptr;
  int offset;
  int nread;
  int len;

  /* if called recursively this should be 'static' */
  char *inpbuf = NULL;

  if (!inpbuf) {
    /* 2K should be enough to begin with */
    if ((inpbuf = (char *)malloc(BUFSIZE)) == NULL) {
      log(LOG_ERR, "Not enough memory");
      return 0;
    }
  }

  inp = inpbuf;
  scan = inpbuf;
  len = BUFSIZE - 1;
  offset = 0;

  while (1) {
    if ((nread = read(fd, inp, len)) < 0 ) {
      if (errno == EINTR)
	continue;
      else
	/* Simulate EOF */
	nread = 0;
    }

    if (!nread) {
      /* EOF but not correct end of file
       * We return node to its topmost parent,
       * which may or may not be correct
       */
      while (node->parent)
	node = node->parent;
      break;
    }

    inp += nread;
    len -= nread;
    
    *inp='\0';

    /* precondition: node->val_len - offset >= inp - inpbuf */

    while (scan != inp) {

      /* character is either value or '<' */

      if (*scan != '<') {
	node->value[offset++] = *scan;
	if ((*scan == ';') && (offset>3)) {
	  if (!strncmp(&node->value[offset-4], "&lt;", 4)) {
	    node->value[offset-4] = '<';
	    offset -= 3;
	  }
	  else if (!strncmp(&node->value[offset-4], "&rt;", 4)) {
	    node->value[offset-4] = '>';
	    offset -= 3;
	  }
	}
	scan++;
      }
      else {
	/* found new key element */

	/* did we read the complete key? */
	if (!(scan_tmp = strchr(scan, '>'))) 
	  break;

	/* If this is a comment slice it out and
	   continue adding value
	*/
	if ((scan[1] == '!') && (scan[2] == '-') && (scan[3] == '-')) {
	  /* Have we already read the end of the comment ? */
	  if (scan_tmp = strstr(scan, "-->")) {
	    scan = (char *)(scan_tmp + 3);
	    continue;
	  } 
	  else
	    break;
	}
	  
	/* finish value */
	node->value[offset] = '\0';
	trim(node->value);
	
	/* discard spaces */
	scan++;
	while (isspace(*scan))
	  scan++;

	/* new key or end of key ? */
	if (*scan == '/') {
	  /* end of key */
	  
	  /* todo: check integrity */
	  
	  scan = (char *)(scan_tmp + 1);

	  parse_handler(node);

	  /* return to papa */
	  if (node->parent) {
	    node = node->parent;
	    if (node->value) 
	      offset = strlen(node->value);
	    else
	      offset = 0; /* this should never happen */
	  }
	  else {
	    free(inpbuf);
	    return 1;
	  }
	}
	else {
	  /* new node */

	  /* Allocate a new node
	  // If memory alloc fails, abort parsing
	  */
	  if (node->child) {
	    /* have already some friends */
	    node->last->next = (Node *)malloc(sizeof(Node));
	    if (!node->last->next) {
	      free(inpbuf);
	      return 0;
	    }
	    node->last = node->last->next;
	  }
	  else {
	    node->child = (Node *)malloc(sizeof(Node));
	    if (!node->child) {
	      free(inpbuf);
	      return 0;
	    }
	    node->last = node->child;
	  }
	  node->last->parent = node;
	  node = node->last;
	  
	  /* init the node */
	  node->next = NULL;
	  node->child = NULL;
	  node->last = NULL;

	  /* nodename */
	  offset = 0;
	  while((offset < MAX_ID_LEN - 1) && 
		(*scan != '>') && (*scan != '/') && !isspace(*scan)) {
	    node->id[offset++] = *scan;
	    scan++;
	  }
	  node->id[offset] = '\0';

	  /* discard surplus */
	  scan = (char *)(strchr(scan, '>') + 1);
	  offset = 0;

	  /* look up typenumber */
	  node->type = getNodeType(node->id);

	  /* EMPTY node ? (i.e. <key />) */
	  if (*(char *)(scan_tmp - 1) == '/') {

	    /* we have no value */
	    node->value = NULL;
	    node->val_len = 0;

	    /* immediately answer */
	    parse_handler(node);

	    /* return to papa */
	    if (node->parent) {
	      node = node->parent;
	      if (node->value) 
		offset = strlen(node->value);
	      else
		offset = 0; /* this should never happen */
	    }
	    else {
	      free(inpbuf);
	      return 1;
	    }
	  }
	  else {
	    
	    /* not an empty node -> initialize node->value */
	    
	    node->value = (char *)malloc(BUFSIZE);
	    if (!node->value) {
	      free(inpbuf);
	      return 0;
	    }
	    else
	      node->val_len = BUFSIZE;
	  }
	    
	}
      }
    }
    node->value[offset] = '\0';

    /* Is buffer still large enough ? */
    if (len < 512) {
      /*
      // If (scan == inpubuf) is still true, we are reading a key
      // Further if len is zero, we alread read 2K(!)
      // 
      // We define the max length of a key to 1536 Bytes and discard
      // the rest
      */
      if ((len == 0) && (scan == inpbuf)) {
	log(LOG_ERR, "Too large key received, discarding 512 Bytes");
	/*	printf("Limits exeeded\nClosing\n"); */
	len = 512;
	inp -= 512;
	continue;
      }
      
      /* move buffer to begin */
      memmove(inpbuf, scan, BUFSIZE-(scan-inpbuf));
      inp = inpbuf;
      scan = inpbuf;
      len = BUFSIZE;
 
      /* check if node->val_len is large enough */
      if (node->val_len - offset < len) {
	ptr = (char *)realloc(node->value, node->val_len + BUFSIZE);
	if (!ptr) {
	  log(LOG_ERR, "Call to realloc failed");
	  printf("Limits exeeded\nClosing\n");
	  free(inpbuf);
	  return 0;
	}
	node->value = ptr;
	node->val_len += BUFSIZE;
      }
    }
  }
  /*
  // If we get here, there was an error
  */
  free(inpbuf);
  return 0;
}

/* this has to be redone:
 * - the parser should not know the semantics 
 * - we should use some sort of table 
 *   then we could just *nodehandler[node->type](node)
 */

void parse_handler(Node *node)
{
  if (node->type && (node->type < max_node))
    n_info[node->type]->func(node);
}

/* Note: XML specifies, that keys are case sensitive */
int getNodeType(char *id)
{
  int i;

  /* this is done very inefficient */
  if (!strcmp(id, "pip"))
    return PIP;

  for(i=1 ; i < max_node ; i++)
    if (!strcmp(id, n_info[i]->id))
      return i;

  /* not found */
  return max_node;
}

void setHandler(char *id, handlefunc hfunc)
{
  node_info *new_ninfo;

  new_ninfo = (node_info *)malloc(sizeof(node_info));

  if (new_ninfo && (max_node < 256)) {
    n_info[max_node] = new_ninfo;
    strcpy(n_info[max_node]->id, id);
    n_info[max_node]->func = hfunc;
    max_node++;
  }
}

/* Free the subnodes of node
 *
 * the node itself will also be freed 
 */
void FreeParseTree(Node *node)
{
  Node *last = node;

  while(node) {
    if (node->value)
      free(node->value);

    if (node->child)
      FreeParseTree(node->child);

    last = node;
    node = node->next;
    free(last);
  }
}

/* trim a string */
void trim(char *str)
{
  int to, from;

  to=0;

  from=strlen(str); 

  /* trim end */
  while ((from>0) && isspace(str[from-1]))
    from--;
  str[from]='\0';

  from = 0;

  /* move to start */
  while(str[from] && isspace(str[from])) 
    from++;

  while(str[from]) {
    str[to] = str[from];
    to++;
    from++;
  }
  str[to] = '\0';
}

Node * newNode(int val_len)
{
  Node *node;

  node = (Node *)malloc(sizeof(Node));
  if (node) {
    memset(node, 0, sizeof(Node));
    node->type = max_node;
    if ((val_len < BUFSIZE))
      val_len = BUFSIZE;
    node->value = (char *)malloc(val_len);
    node->val_len = val_len;
  }
  else {
    /* not enough memory is a problem... */
    log(LOG_ERR, "Not enough memory");
    exit(1);
  }

  return(node);
}
