/* PIPServer - A deamon for finger protocol v2
 *
 * Copyright (C) 1999 Michael Baumer <baumi@vis.ethz.ch>
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#include <config.h>

#include <stdio.h>
#include <unistd.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <errno.h>
#include <string.h>
#include <stdlib.h>
#include <arpa/inet.h>

#include "in.fingerd.h"
#include "finger.h"
#include "parser.h"
#include "configfile.h"
#include "statusdb.h"
#include "init.h"
#include "compat.h"
#include "log.h"
#include "acl.h"

#ifndef HAVE_GETOPT_DECL
int getopt(int argc, char **argv, char *optstring);
extern char *optarg;
extern int optind;
#endif 

#define NOBODY_UID 65534
#define NOBODY_GID 65534

fingerconf config;
connection conn;
char hostname[64];

extern void finger_handler(Node *node);
extern void online_handler(Node *node);
extern void online_forwarder(Node *node);
extern void login_handler(Node *node);
extern void finger1(char *input);
extern void init_finger1();

void usage(char *progname)
{
  printf("Usage: %s [-c file] [-h] [-v] [-V] [-d]\n", progname);
  printf("\t-c file\tuse \"file\" as configuration file\n"
	 "\t-h\tprint this screen and exit\n"
	 "\t-v\tverbose output\n"
	 "\t-V\tprint version number and exit\n"
	 "\t-d\tdebug modus\n"
	 "\t-p\t\n\t-L\n\t-l\n\t-w\tno option - accepted for compatibility\n");
}

void StartConnection()
{
  char input[1024];  /* the input buffer */
  Node pipnode;
  int s, nread;

  if (!fgets(input, sizeof(input)-1, stdin)) 
    exit(0);

  input[1024] = '\0';

  /* parse the input */
  if (strncmp(input, "<pip>", 5)) {
	
    /* finger protocol version 1 */
    /* the first line is the only one ... */
	
    /* Forward to the sitehost if necessary */
    if (config.clusterType == 1) {
      s = make_conn(config.sitehost, NULL);
      if (s >= 0) {
	write(s, input, strlen(input));
	while ((nread = read(s, input, sizeof(input)-1)) >0) {
	  if (nread < sizeof(input))
	    input[nread] = '\0';
	  printf("%s", input);
	}
	exit(0);
      }
      /* else fall through */
      printf("Could not connect to site master. Displaying local info instead\r\n\r\n");
     }
    init_finger1();
    finger1(input);
    exit(0);
  }

  /* finger protocol version > 1 */

  memset(&conn, 0, sizeof(conn));
  conn.version = 2;

  /* init the parser */
  setHandler("finger", (handlefunc)finger_handler);
  if (config.clusterType != 1)
    setHandler("online", (handlefunc)online_handler);
  else 
    setHandler("online", (handlefunc)online_forwarder);
  setHandler("localonline", (handlefunc)online_handler);
  setHandler("login", (handlefunc)login_handler);

  /* initialize the parse tree */

  pipnode.next = NULL;
  pipnode.child = NULL;
  pipnode.last = NULL;
  pipnode.parent = NULL;
  strcpy(pipnode.id, "pip");
  pipnode.type = PIP;
  pipnode.value = (char *)malloc(BUFSIZE);
  pipnode.val_len = BUFSIZE;

  printf("<pip>\n");
  ParseNode(fileno(stdin), &pipnode);
  printf("</pip>\n");
}

void printConnInfo(int s)
{
  struct sockaddr_in remaddr;
  struct hostent *h_ent;
  int len;

  if (!isatty(fileno(stdin))) {
    
    /*    fprintf(stderr, "Setting buffering mode\n"); */

    if (setvbuf(stdin, NULL, _IOLBF, 0) < 0) {
      log(LOG_ERR, "setvbuf(stdin)");
      exit(1);
    }
    if (setvbuf(stdout, NULL, _IOLBF, 0) < 0) {
      log(LOG_ERR, "setvbuf(stdout)");
      exit(1);
    }
  }
   
  len = sizeof(remaddr);
  if (getpeername(s, (struct sockaddr *)&remaddr, &len) < 0) {
    if (errno != ENOTSOCK) {
      log(LOG_ERR, "while printing connection info: %m");
      exit(-1);
    }
    else {
      /* this is done this way so we can start in.fingerd in a shell */
      log(LOG_INFO, "Started from shell");
      conn.hostname = "localhost";
      acl_init(conn.hostname, "127.0.0.1");
      return;
    }
  }

  /* If we get here we have got the peername */

  /* This information may be wrong, but until we implement authuser
   * (identd) lookup it's all we have.
   * Note that identd lookups are _not_ a 100% secure way either
   */

  /* get the human readable name */
  h_ent = gethostbyaddr((char *)&remaddr.sin_addr, sizeof(remaddr.sin_addr), AF_INET);
  if (h_ent) 
    conn.hostname = strdup(h_ent->h_name);
  else
    conn.hostname = "(remote)";

  log(LOG_INFO, "Connection from %s (%s)", 
      conn.hostname,
      inet_ntoa(remaddr.sin_addr));

  /* Initialize ACL */
  acl_init(conn.hostname, inet_ntoa(remaddr.sin_addr));
} 

void get_hostname()
{
  char domainname[64];
  char *idx1, *idx2;

  /* get the qualified hostname. It is an error if that is not possible */

  if (gethostname(hostname, 63) < 0)
    hostname[0] = '\0';

  /* Are we the sitehost ?
   * Note: We check only the hostname, which could result in problems
   *       if the site host is in another domain 
   */
  if (config.sitehost) {
    if (idx1 = strchr(hostname, '.'))
      *idx1 = 0;
    if (idx2 = strchr(config.sitehost, '.'))
      *idx2 = 0;

    if (!strcmp(hostname, config.sitehost)) 
      /* site server gets info from DB */
      config.clusterType = 2;
    else
      config.clusterType = 1;

    if (idx1)
      *idx1 = '.';
    if (idx2)
      *idx2 = '.';
  }

#if HAVE_GETDOMAINNAME
  /* Note: This is _not_ necessarily the
   * fully qualified domainname (fqdn), especially if you are
   * using yp/NIS.
   * To get the fqdn we would have to make a gethostbyname(hostname).
   * But, since it is not that necessary...
   */
  if (!strchr(hostname, '.')) {
    if (getdomainname(domainname, 63) < 0) 
      domainname[0] = '\0';
    else {
      strcat(hostname, ".");
      strcat(hostname, domainname);
    }
  }
#endif /* HAVE_GETDOMAINNAME */
  hostname[63] = '\0';
}
  
int main(int argc, char *argv[])
{
  int flag;
  int verbose = FALSE;
  int debug = FALSE;
  int userlist = TRUE;
  int detach = TRUE;
  char *progname;

  /* At the very first we loose root rights */
  if (!getgid()) 
    if (setgid(NOBODY_GID)<0) {
      perror("Could not set uid");
      exit(1);
    }

  if (!getuid()) 
    if (setuid(NOBODY_UID)<0) {
      perror("Could not set uid");
      exit(1);
    }

  /*  
  fprintf(stderr, "starting fingerd %d\n", getpid()); 
  fflush(stderr);
  sleep(5);
  */

  /* extraxt base name */
  progname = strrchr(argv[0], '/');
  if (!progname)
    progname = argv[0];
  else
    progname++;

  /* parse the command line arguments */

  /* "p:L:wul" are there for compatibility reasons */

  config.gnufinger = 0;

  while ((flag = getopt(argc, argv,
			"c:dghvaVp:L:wul")) != EOF)

      switch (flag) {
      case 'a':
	detach = FALSE;
	break;
      case 'c':
	configfile = strdup(optarg);
	break;
      case 'd':
	debug = TRUE;
	detach = FALSE;
	break;
      case 'g':
	config.gnufinger = 1;
	break;
      case 'v':
	verbose = TRUE;
	break;
      case 'V':
	printf("%s: %s Version %s\n", progname, PACKAGE, VERSION);
	exit(0);
      case 'p':
      case 'L':
      case 'w':
      case 'l':
	/* compatibility options */
	break;
      case 'u':
	userlist = FALSE;
	break;
      case 'h':
      default:
	usage(progname);
	exit(0);
	
      }

  /* initialization */
  memset(&conn, 0, sizeof(conn));
  Init(progname, debug);

  if (!userlist)
    config.acl->options &= !ACL_USERLIST;

  /* print some info about the connection to syslog */
  printConnInfo(fileno(stdin));

  /* whats our name? */
  get_hostname();

  /* If we are started by inetd, we can simply hang around */

  StartConnection();

  /* The connection is closed.
   * Clean up
   */
  db_logout();

  exit(0);
}
