<?php
/*
Copyright 2001,2002,2003 Dashamir Hoxha, dashohoxha@users.sourceforge.net

This file is part of phpWebApp.

phpWebApp is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

phpWebApp is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with phpWebApp; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  
*/


/**
 * Renders an HTML page from the templates and structures of the $webPage.
 * @package parser 
 * @see WebPage
 */
class Render
{
  /** If $collect is true, then the page is not 
   * outputed directly to the browser... */
  var $collect;

  /** ...but is collected to the variable $html_page */
  var $html_page;

  /** Helps to render a well-indented page. */
  var $indent;


  function Render($collect =false)
    {
      $this->collect = $collect;
      $this->html_page = "";
      $this->indent = "";
    }

  /** 
   * Either sends the line to the browser or concats it
   * to the variable $this->html_page.
   */
  function print_line($line)
    {
      if ($this->collect)
        $this->html_page .= $this->indent.$line;
      else
        print $this->indent.$line;
    }

  /**
   * Renders the given template, replacing all variables
   * and including all subtemplates.
   */
  function render_tpl($tpl)
    {
      global $webPage, $tplVars;

      $tplVars->pushScope();
      $tplVars->addVars($tpl->getVars());

      $i = 0;
      while ($i < sizeof($tpl->content))
        {
          $line = $tpl->content[$i++];
          if (chop($line)<>"##")
            {
              $this->render_line($line);
            }
          else  //subtemplate indicator found
            {
              //get the subtemplate
              $line = $tpl->content[$i++];
              $subtpl_id = trim($line);
              $subtpl = $webPage->getTemplate($subtpl_id);

              //render the subtemplate
              $this->render_subtpl($subtpl);
            }
        }
                
      $tplVars->popScope();
    }

  /** 
   * Replace all {{tpl_vars}} in the given line 
   * and then print it out.
   */
  function render_line($line)
    {
      //replace all {{tpl_vars}}
      $line = WebApp::replaceVars($line);
      //output the line
      $this->print_line($line);
    }

  /**
   * Dispatcher function: according to the type of 
   * the given $tpl, calls the appropriate rendering function.
   */
  function render_subtpl($tpl)
    {
      switch ($tpl->type)
        {
        default:
          $this->render_Template($tpl);
          break;
        case "IfTpl":
          $this->render_IfTpl($tpl);
          break;
        case "FileTpl":
          $this->render_FileTpl($tpl);
          break;
        case "MainTpl":
          $this->render_MainTpl($tpl);
          break;
        case "WebObjectTpl":
          $this->render_WebObjectTpl($tpl);
          break;
        case "RepeatTpl":
          $this->render_RepeatTpl($tpl);
          break;
        }
    }

  /**
   * Renders the given Template (which may have $tpl->type
   * as 'Template', 'IfEmpty', 'Header', 'Footer' and 'Separator'.
   */
  function render_Template($tpl)
    {
      $this->render_tpl($tpl);
    }

  /**
   * Renders the given IfTpl, if its $condition evaluates to true.
   */
  function render_IfTpl($if_tpl)
    {
      $expr = $if_tpl->condition;
      $expr = WebApp::replaceVars($expr);
      $val = WebApp::evaluate($expr);
      if ($val) $this->render_tpl($if_tpl);
    }

  /**
   * Renders the given $file_tpl. Defines the variable {{./}} 
   * that contains the path of the file.
   */
  function render_FileTpl($file_tpl)
    {
      global $tplVars;

      $tplVars->pushScope();

      //add the variable {{./}} which contains the current path
      $tpl_path = dirname($file_tpl->filename) . "/";
      $tpl_path = WebApp::to_url($tpl_path);
      $tplVars->addVar("./", $tpl_path);
              
      //increase indentation
      $prev_indent = $this->indent;
      $this->indent .= $file_tpl->indentation;

      //render the FileTpl
      $this->render_tpl($file_tpl);

      //revert to previous indentation
      $this->indent = $prev_indent;

      $tplVars->popScope();
    }

  /**
   * Renders the given $main_tpl by rendering its head and its body.
   */
  function render_MainTpl($main_tpl)
    {
      array_unshift($main_tpl->head->content, "<head>\n");
      array_push($main_tpl->head->content, "</head>\n");
      array_push($main_tpl->body->content, "</body>\n");
      $this->render_FileTpl($main_tpl);
    }

  /**
   * Renders the given WebObjectTpl. Before rendering it, declares 
   * its state vars and params as template vars, and calls the function 
   * onRender() of the WebObject.
   */
  function render_WebObjectTpl($wobj_tpl)
    {
      global $tplVars;

      $tplVars->pushScope();
      //save the current scope of the var stack
      $wobj_tpl->scope = $tplVars->getCurrentScope();

      //add state vars to the current scope
      $tplVars->addVars($wobj_tpl->getSVars());

      //add some special variables which identify this object
      $tplVars->addVars($wobj_tpl->getObjVars());

      //add parameters to the current scope
      $tplVars->addVars($wobj_tpl->getParams());

      //add the class url variable {{./}}
      $class_path = $wobj_tpl->class->path;

      $class_url = WebApp::to_url($class_path);
      $tplVars->addVar("./", $class_url);

      //call the function onRender of this WebObject
      $wobj_tpl->onRender(); 

      $this->render_tpl($wobj_tpl); //render it to HTML

      $tplVars->popScope();
    }

  /**
   * Renders the given $tpl of type RepeatTpl.
   */
  function render_RepeatTpl($tpl)
    {
      global $webPage, $tplVars;
                
      //get the recordset id
      $rs_id = $tpl->rsId;
      $rs_id = WebApp::replaceVars($rs_id);
      //get a reference to the recordset of the RepeatTpl
      if (isset($webPage->rs_collection[$rs_id]))
        {
          //we cannot use "$webPage->getRecordset($rs_id);" here
          //because we need a reference to the recordset,  
          //and PHP functions cannot return a reference
          $rs = &$webPage->rs_collection[$rs_id];
        }
      else //such a recordset does not exist
        {
          $err_msg = WebApp::error_msg("Recordset '$rs_id' is undefined.");
          $this->print_line($err_msg);
          $this->render_Template($tpl);
          return;
        }

      $tplVars->pushScope();    //a new scope for the Repeat element

      //open the recordset
      if ($rs->type=="PagedRS")
        {
          $rs->Open();  
          //add page vars (PrevPage, NextPage, etc)
          $rs_page_vars = $rs->getPageVars();
          $tplVars->addVars($rs_page_vars);
        }
      else
        {
          $rs->Open();
        }
                
      if ($rs->EOF())
        {
          //recordset has no records
          $this->render_Template($tpl->if_empty);
          $tplVars->popScope();
          return;
        }

      //render the header
      $this->render_Template($tpl->header);

      //repeat the body for each record
      $row_nr = 1;
      while ( !$rs->EOF() )
        {
          $tplVars->pushScope();

          //add variables {{CurrentRecNr}} and {{CurrentRowNr}}
          $firstRecNr = ($rs->type=="PagedRS") ? WebApp::getVar("FirstRec") : 1;
          WebApp::addVar("CurrentRecNr", $firstRecNr + $row_nr - 1);
          WebApp::addVar("CurrentRowNr", $row_nr);

          //render the body of the repeat
          $tplVars->addVars($rs->Fields());
          $tplVars->addVars($tpl->getVars());
          $this->render_RepeatBody($tpl);

          if ($row_nr < $rs->count) 
            {
              //render the separator
              $this->render_Template($tpl->separator);
            }

          $tplVars->popScope();
          $rs->MoveNext();
          $row_nr++;
        }

      //render the footer
      $this->render_Template($tpl->footer);
                
      $tplVars->popScope();
    }

  function render_RepeatBody($tpl)
    {
      $this->render_tpl($tpl);
    }
}
?>