<?php
/*
Copyright 2001,2002,2003 Dashamir Hoxha, dashohoxha@users.sourceforge.net

This file is part of phpWebApp.

phpWebApp is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

phpWebApp is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with phpWebApp; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  
*/

/**
 * Contains some functions that are used frequently in the application.
 * These functions could have been defined in the global scope as well,
 * but instead they are defined inside a class just for the sake of
 * the namespace. They are called like this: WebApp::func_name() which
 * makes clear that they are functions of the web application framework.
 */
class WebApp
{
  /**
   * Returns true only the first time that the application
   * is opened, at the begining of the session.
   */
  function first_time()
    {
      global $request;
      return $request->firstTime;
    }

  /**
   * A free event has target 'none', doesn't have a target webobject,
   * so no webobject can handle it. Instead, it is handled by the
   * function on_eventName($event), that is in the file 'on.eventName.php',
   * which is placed in the same folder as the target page or the source
   * page of the event.
   */
  function callFreeEvent()
    {
      global $event;

      $file_name = "on.".$event->name.".php";
      $fun_name = "on_".$event->name;

      //look for the event handler in the path of targetPage
      $path = TPL_PATH.dirname($event->targetPage);
      $fname = $path."/".$file_name;
      if (file_exists($fname))
        {
          include_once $fname;
          $fun_name($event);
          return;
        }

      //look for the event handler in the path of sourcePage
      $path = TPL_PATH.dirname($event->sourcePage);
      $fname = $path."/".$file_name;
      if (file_exists($fname))
        {
          include_once $fname;
          $fun_name($event);
          return;
        }

      //no event handler found
      print WebApp::error_msg("event handler '$file_name' not found.");
    }
        
  /*----------- begin construct page functions --------------*/

  /** 
   * Constructs an HTML page which is sent directly to the browser.
   * If $collect is true, however, instead of sending the page to
   * the browser, it returns it as a big string. 
   */
  function constructHtmlPage($tpl_file, $collect =false)
    {
      global $webPage;

      //measure the time of page generation
      $comment = "Time that spends the application for constructing the page";
      $webPage->timer->Start("WebApp", $comment);

      //include 'before_page.php' if it exists           
      $fname = APP_PATH."before_page.php";
      if (file_exists($fname))  include_once $fname;

      //construct the structure of the web page 
      //by parsing the template file
      global $parser;
      $parser->parse_main($tpl_file);

      //append to <head> the JS code that initiates the JS timer
      $webPage->append_to_head($webPage->timer->JSTimer());

      //add also the stylesheet of the framework
      $webapp_styles = '  <link rel="stylesheet" type="text/css" '
        . 'href="'.WEBAPP_URL.'styles.css">'."\n";
      $webPage->append_to_head($webapp_styles);

      //parse and add the append.html after the MainTpl
      $tpl = $parser->parse_file(APPEND_PATH."append.html");
      $webPage->append_to_body("##\n");
      $webPage->append_to_body($tpl->id."\n");

      //render the html page
      global $render;
      $render->collect = $collect;
      //$render = new RenderAnnotated($collect);
      $render->render_MainTpl($webPage->rootTpl);

      //include 'after_page.php' if it exists            
      $fname = APP_PATH."after_page.php";
      if (file_exists($fname))  include_once $fname;

      if ($collect)  return $render->html_page;
    }

  /**
   * Constructs an HTML page and returns it as a string.
   * Then resumes constructing the previous page.
   * It is useful when you want to construct a page
   * for sending it by e-mail.
   * @see constructHtmlPage()
   */
  function getHtmlPage($tpl_file)
    {
      global $webPage, $tplVars;

      //save the current webPage and whatever 
      //is already constructed in it
      $old_webPage = $webPage;
      $old_tplVars = $tplVars;

      //create a new web page in the global variable $webPage
      $webPage = new WebPage;
      $tplVars = new VarStack;

      //construct it into the $html_page variable
      $collect = true;
      $html_page = WebApp::constructHtmlPage($tpl_file, $collect);  

      //return to the previous web page 
      //and continue with its construction
      $webPage = $old_webPage;
      $tplVars = $old_tplVars;

      //return the constructed page
      return $html_page;
    }

  /**
   * This function stops loading the current page
   * and starts loading and constructing the given page.
   * If no page is given, reloads the same page again.
   *
   * @todo It doesn't work yet, see how to fix it.
   */
  function reloadPage($tpl_file=UNDEFINED)
    {
      global $webPage, $tplVars;

      if ($tpl_file==UNDEFINED) //no parameter given
        {
          //reload again the same template
          $tpl_file = $webPage->tpl_file; 
        }
      unset($webPage);
      unset($varStack);
      $webPage = new WebPage;
      $tplVars = new VarStack;
      $webPage->load($tpl_file);
    }

  /*------ end construct functions ----------------*/
        
  /*------ begin variable functions ---------------*/

  /**
   * Add a template variable in the current scope 
   * (for the current webbox or template).
   * @see addVars(), getVar(), addGlobalVar(), addGlobalVars()
   */
  function addVar($var_name, $var_value)
    {
      global $tplVars;
      $tplVars->addVar($var_name, $var_value);
    }
        
  /**
   * Add a template variable in the global scope;
   * it will be available in all the application templates.
   * @see  addGlobalVars(), getVar(), addVar(), addVars()
   */
  function addGlobalVar($var_name, $var_value)
    {
      global $tplVars;
      $tplVars->addGlobalVar($var_name, $var_value);
    }
        
  /**
   * Add a list (associative array) of template variables 
   * at once. 
   * @see addVar(), getVar(), addGlobalVar(), addGlobalVars()
   */
  function addVars($arrVars)
    {
      global $tplVars;
      $tplVars->addVars($arrVars);
    }
        
  /**
   * Add a list of global template variables at once.
   * @see  addGlobalVar(), getVar(), addVar(), addVars()
   */
  function addGlobalVars($arrVars)
    {
      global $tplVars;
      $tplVars->addGlobalVars($arrVars);
    }
        
  /**
   * Returns the value of a template variable,
   * or UNDEFINED if such a variable does not exist. 
   * @see addVar(), addVars(), addGlobalVar(), addGlobalVars()
   */
  function getVar($var_name)
    {
      global $tplVars;
      return $tplVars->getVar($var_name);
    }

  /**
   * Add a session variable. 
   * @see getSVar(), setSVar()
   */
  function addSVar($var_name, $var_value, $db =false)
    {
      global $session;
      $session->addVar($var_name, $var_value, $db);
    }

  /**
   * Set a new value to a session variable.
   * @see addSVar(), getSVar()
   */
  function setSVar($var_name, $var_value)
    {
      //set it in the session
      global $session;
      $session->setVar($var_name, $var_value);

      //if it is the state var of a webobject
      //set it in the webobject's scope as well
      if (ereg("(.*)->(.*)", $var_name, $regs))  
        {
          $obj_id = $regs[1];
          $v_name  = $regs[2];
          $webobj = WebApp::getObject($obj_id);
          if ($webobj==UNDEFINED)   return;

          global $tplVars;
          $tplVars->addVar($v_name, $var_value, $webobj->scope);
        }
    }
    
  /**
   * Get the value of a session variable.
   * @see addSVar(), setSVar()
   */
  function getSVar($var_name)
    {
      global $session;
      $var_value = $session->getVar($var_name);
      return $var_value;
    }

  /**
   * Returns the web object with the given id,
   * or UNDEFINED.
   */
  function getObject($obj_id)
    {
      global $webPage;
      $obj = $webPage->getObject($obj_id);
      return $obj;
    }

  /*------ end variable functions ----------------*/

  /*------ begin evaluate ------------------------*/

  /** Returns the value of the given expression. */
  function evaluate($expr)
    {
      //declare as global any variables that are used in $expr
      $var_names = implode(",", WebApp::get_var_names($expr));
      if ($var_names <> '')
        {
          eval("global $var_names;");
        }
      eval("\$webApp_value = $expr;");
      return $webApp_value;
    }
  /**
   * Returns an array with all the variable names that are used
   * in the given $expr. Called by evaluate() (which will be declared 
   * these variables as global).
   * @see evaluate()
   */
  function get_var_names($expr)
    {
      $var_names = array();
      while ( ereg('\$[[:alnum:]_]+', $expr, $regs) )
        {
          $var_names[] = $regs[0];
          $expr = str_replace($regs[0], '', $expr);
        }
      return $var_names;
    }
  /*------ end evaluate --------------------------*/
        
  /**
   * Returns the value of a variable by looking first at $tplVars,
   * and then at $session vars, then at PHP constants, and finally
   * at PHP global variables. If this variable is not found, 
   * returns the constant VAR_NOT_FOUND.
   */
  function getVarValue($var_name)
    {
      //search first in the template variables ($tplVars)
      $var_value = WebApp::getVar($var_name);
      if (!($var_value===UNDEFINED))    //found
        {
          return $var_value;    
        }

      //if not found look at session (state) variables
      $var_value = WebApp::getSVar($var_name);
      if (!($var_value===UNDEFINED))
        {
          return $var_value;
        }

      //if not found look at PHP constants
      if ( defined($var_name) )
        {
          eval("\$tmp = $var_name;");
          $var_value = $tmp;
          return $var_value;
        }

      //if not found look at PHP global variables
      if ( isset($GLOBALS[$var_name]) )
        {
          $var_value = $GLOBALS[$var_name];
          return $var_value;
        }

      //if still not found return the constant VAR_NOT_FOUND
      $var_value = str_replace("var_name", $var_name, VAR_NOT_FOUND);
      return $var_value;
    }
        
        
  /**
   * Search in the given line for variable occourences
   * like this: {{var_name}} and replace them by their values.
   *
   * If single (double) quotes surround the var_name, like
   * this: {{'var_name'}} ({{"var_name"}}), then all occourences 
   * of the single (double) quote inside the value of the variable 
   * are escaped, like this: \' (\"). This is done because sometimes
   * the variable is inside a string, and if it contains quotes inside
   * the value, then it may break the string.
   */
  function replaceVars($line)
    {
      while ( ereg("\{\{([^#{][^{}]*[^#}])\}\}", $line, $regs) )
        {
          $str2replace = $regs[0];
          $var_name = $regs[1];
          $var_value = $regs[1];
          list($var_name, $quote) = WebApp::get_quote($var_name);
          $var_value = WebApp::getVarValue($var_name);
          $var_value = WebApp::escape_quotes($var_value, $quote);
          $line = str_replace($str2replace, $var_value, $line);
        }
      $line = str_replace('{{#', '{{', $line);
      $line = str_replace('#}}', '}}', $line);
      return $line;
    }
  /**
   * Get the quote which surrounds the $var_name:
   * single, double, or none.
   * @see replaceVars()
   */
  function get_quote($var_name)
    {
      if ( ereg("^'(.*)'$", $var_name, $regs) )
        {       //surrounded by single quotes
          $var_name = $regs[1];
          $quote = "'";
        }
      elseif ( ereg('^"(.*)"$', $var_name, $regs) )
        {       //surrounded by double quotes
          $var_name = $regs[1];
          $quote = '"';
        }
      else
        {       //not surrounded by quotes
          $quote = '';
        }
      return array($var_name, $quote);
    }
  /**
   * Escape all the quotes in the $var_value.
   * @see replaceVars(), get_quote()
   */
  function escape_quotes($var_value, $quote)
    {
      switch ($quote)
        {
        case '"':               //double quote
          $var_value = str_replace('"', '\"', $var_value);
          $var_value = '"'.$var_value.'"';
          break;
        case "'":               //single quote
          $var_value = str_replace("'", "\'", $var_value);
          $var_value = "'".$var_value."'";
          break;
        case '':                //no quotes
                                //do nothing
          break;
        default:
          print "warning: escape_quotes: the quote '$quote' is unknown.<br>\n";
          break;
        }
      return $var_value;
    }
  /*------ end replaceVars() --------------------*/
        
  /*------ begin DB functions -------------------*/

  /**
   * Execute the given query.
   * @return Recordset 
   * @see execDBCmd(), openRS()
   */
  function execQuery($query, $conn =UNDEFINED)
    {
      $rs = new EditableRS(UNDEFINED, $query, $conn);
      $rs->Open();
      return $rs;
    }

  /**
   * Open the given table and return the records that
   * satisfy the given $condition. 
   * @return TableRS
   */
  function openTable($tbl_name, $condition ="(1=1)", $conn =UNDEFINED)
    {
      $query = "SELECT * FROM $tbl_name WHERE $condition";
      $rs = new TableRS($tbl_name, $query, $conn);
      $rs->Open();
      return $rs;
    }
        
  /**
   * Execute the DB command with the given id.
   * $params is an array of {{vars}} that are used in the query.
   * Returns TRUE or FALSE indicating success or failure.
   * @see execQuery(), openRS()
   */
  function execDBCmd($cmd_id, $params =array())
    {
      global $webPage, $tplVars;

      $tplVars->pushScope();
      $tplVars->addVars($params);
      $rs = $webPage->getRecordset($cmd_id);
      if ($rs==UNDEFINED)
        {
          print WebApp::error_msg("WebApp::execDBCmd(): dbCommand '$cmd_id' is undefined.");
          return false;
        }
      $result = $rs->Open();
      $tplVars->popScope();

      return $result;
    }
        
  /**
   * Open and return the recordset with the given id.
   * $params is an array of {{vars}} that are used in the query.
   * @return Recordset
   * @see execDBCmd(), execQuery()
   */
  function openRS($rs_id, $params =array())
    {
      global $webPage, $tplVars;

      if (isset($webPage->rs_collection[$rs_id]))
        {
          $tplVars->pushScope();
          $tplVars->addVars($params);
          $rs = &$webPage->rs_collection[$rs_id];
          $rs->Open();
          $tplVars->popScope();
          return $rs;
        }
      else
        {
          print WebApp::error_msg("WebApp::openRS(): recordset '$rs_id' is undefined.");
          return UNDEFINED;
        }
    }
  /*------ end DB functions ---------------------*/
        
  /*------ mix functions ------------------------*/

  /** Convert the given path to url. */
  function to_url($path)
    {
      $DocumentRoot = ereg_replace(APP_URL."\$", '', APP_PATH);
      $url = str_replace($DocumentRoot, '', $path);
      return $url;
    }

  /**
   * Construct and return a link that can be used to access
   * from outside a certain page of the application, in a certain
   * state.
   */
  function get_external_link()
    {
      global $request;

      $extern_link = APP_URL."index.php";
      $extern_link .= '?sourcePage='.urlencode($request->sourcePage);
      $extern_link .= '&targetPage='.urlencode($request->targetPage);
      $extern_link .= '&sessionVars='.urlencode($request->sessionVars);
      $extern_link .= '&strEvent='.urlencode($request->strEvent);
      $extern_link .= '&phpVars='.urlencode($request->phpVars);

      return $extern_link;
    }

  /** Display a pop-up message box to the user. */
  function message($msg)
    {
      global $webPage;
      $webPage->messages[] = $msg;
    }

  /** Output a debug message. */
  function debug_msg($dbg_msg, $comment ="")
    {
      global $webPage;
      $webPage->addDbgMsg($dbg_msg, $comment);
    }
        
  /** Format and return an error message. */
  function error_msg($err_msg)
    {
      if (!SHOW_ERROR_MESSAGES)  return "";

      $err_msg = str_replace(UP_PATH, '', $err_msg);
      $err_msg = htmlentities($err_msg);
      return "\n<span class='webapp_error'><b>WebApp Error:</b> "
        . $err_msg . "</span>\n";
    }

  /** Format and return a warning message. */
  function warning_msg($warn_msg)
    {
      if (!SHOW_ERROR_MESSAGES)  return "";

      $err_msg = str_replace(UP_PATH, '', $err_msg);
      $warn_msg = htmlentities($warn_msg);
      return "\n<span class='webapp_warning'><b>WebApp Warning:</b> "
        . $warn_msg . "</span>\n";
    }
                
}
?>