<?php
/*
Copyright 2001,2002,2003 Dashamir Hoxha, dashohoxha@users.sourceforge.net

This file is part of phpWebApp.

phpWebApp is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

phpWebApp is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with phpWebApp; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  
*/


/** 
 * This class represents a recordset (the table returned by the
 * execution of a query). It executes the query using a given
 * connection with a database. If no connection is specified, then
 * it uses the global connection of the framework: $cnn.
 *
 * @package database
 */
class Recordset
{
  /** The ID of the Recordset. */
  var $ID;

  /** The type of the recordset: Recordset, PagedRS, 
   *  StaticRS, DynamicRS, EditableRS, TableRS */
  var $type;
        
  /** Query as string (may contain {{variables}}). */
  var $query;
  /** The connection that will execute the query. */
  var $cnn;
        
  /** The result after executing the query (2 dimensional array). */
  var $content;
  /** The current position in the array $content. */
  var $pos;
  /** The number of rows in the array $content, */
  var $count;
               

  function EOF()
    {
      return $this->pos >= $this->count;
    }

  function BOF()
    {
      return $this->pos < 0;
    }

  function MoveNext()
    {
      if ( !$this->EOF() )      $this->pos++;
    }

  function MovePrev()
    {
      if ( !$this->BOF() )      $this->pos--;
    }

  function MoveFirst()
    {
      $this->pos = 0;
    }

  function MoveLast()
    {
      $this->pos = $this->count - 1;
    }

  /** Returns the value of the field given as parameter. */
  function Field($fld_name)
    {
      if ( isset($this->content[$this->pos][$fld_name]) )
        return $this->content[$this->pos][$fld_name];
      else
        return UNDEFINED;
    }

  /** Returns all the fields at the current position as an associated array. */
  function Fields()
    {
      return $this->content[$this->pos];
    }

  /** Constructor */
  function Recordset($id =UNDEFINED, $query =UNDEFINED, $conn =UNDEFINED)
    {
      global $cnn;

      $this->ID = $id;
      $this->query = $query;
      $this->cnn = ($conn==UNDEFINED ? $cnn : $conn);
      $this->content = array();
      $this->count = 0;
      $this->pos = 0;
      $this->type = "Recordset";
    }

  /** Executes the query and puts the result into $this->content. */
  function Open($q =UNDEFINED)
    {
      if ($q<>UNDEFINED)        $this->query = $q;
      $query = $this->query;
      if ($query==UNDEFINED)  return;

      $this->Close();           //empty the content

      $query = WebApp::replaceVars($query);
      $result = $this->cnn->execQuery($query, $this->ID);
      if (is_array($result))
        {
          $this->content = $result;
          $this->count = count($result);
          $this->pos = 0;
        }
      else
        {
          return $result;
        }
    }
        
  function Close()
    {
      if (isset($this->content))   unset($this->content);
      $this->count = 0;
      $this->pos = 0;
    }

  /**
   * Returns a new recordset that contains a slice of this recordset.
   * @see array_slice
   */
  function slice($offset =0, $length =UNDEFINED)
    {
      $slice = array();
      if ($length===UNDEFINED)
        {
          $slice = array_slice($this->content, $offset);
        }
      else
        {
          $slice = array_slice($this->content, $offset, $length);
        }

      //create a new recordset of the same type as this
      $rs = new $this->type;

      $rs->content = $slice;
      $rs->count = sizeof($slice);
      $rs->pos = 0;

      return $rs;
    }

  /**
   * Returns true if the query at the given position
   * matches the $condition (usually used by the functions
   * find() and filter()). If no position is given
   * matches the current record.
   * Currently the $condition supports only one field.
   * @see find(), filter()
   */
  function match($condition, $pos=UNDEFINED)
    {
      //get the field name, pattern and match type from $condition
      list($fld_name,$pattern) = split("=", $condition, 2);
      $fld_name = trim($fld_name);
      $pattern = trim($pattern);
      $match_type = substr($pattern, 0, 1); //get first char
      //remove first and last chars
      $pattern = substr($pattern, 1, sizeof($pattern)-2); 

      //get the field value
      if ($pos===UNDEFINED)  $pos = $this->pos;
      $fld_value = $this->content[$pos][$fld_name];

      //match with pattern
      if ($match_type=="'")
        {
          $result = ($fld_value==$pattern);
        }
      else
        {
          $result = ereg($pattern, $fld_value);
        }

      return $result;
    }

  /**
   * Returns a new recordset with the recs that match the
   * given condition, $condition is like the WHERE condition
   * of a query, but it matches using regexp-s, e.g.
   * <kbd>(username=/^d.*a/ AND (firstname=/[^a]+/ OR NOT lastname='Hoxha'))</kbd>
   *
   * @todo Currently it supports only one field, no AND, OR, (, ) etc.
   */
  function filter($condition)
    {
      $rs = new $this->type;
      $rs->Open();
      for ($i=0; $i < $this->count; $i++)
        {
          if ($this->match($condition, $i))
            {
              $rec = $this->content[$i]; 
              $rs->addRec($rec);
            }
        }
      $rs->MoveFirst();
      return $rs;
    }

  /**
   * Finds the subset of the recordset that matches the
   * condition and positions the pointer to the first
   * record found. $condition is like the condition in filter().
   * If called without parameter, if finds the next one,
   * and so on up to EOF. Returns false if it couldn't find
   * another record.
   * @see filter()
   */
  function find($condition =UNDEFINED)
    {
      if ($condition<>UNDEFINED)   $this->search($condition);
      return $this->find_next();
    }

  /**
   * Used by find(), finds all the records that match
   * the condition and stores their indeces in
   * the array $this->found_positions.
   * @see find()
   */
  function search($condition)
    {
      $this->found_positions = array();
      $this->current_found = -1;
      for ($i=0; $i < $this->count; $i++)
        {
          if ($this->match($condition, $i))
            {
              $this->found_positions[] = $i;
            }
        }
    }

  /**
   * Used by find(), sets $this->pos to the next
   * found record; returns true if found,
   * or false if there are no more records.
   * @see find()
   */
  function find_next()
    {
      $found_nr = sizeof($this->found_positions);
      if ($this->current_found + 1 < $found_nr )
        {
          $this->current_found += 1;
          $idx = $this->current_found;
          $this->pos = $this->found_positions[$idx];
          return true;
        }
      else
        {
          return false;
        }
    }

  /** Returns an array with the values of the specified field. */
  function getColumn($fld_name)
    {
      $column = array();
      for ($i=0; $i < $this->count; $i++)
        {
          if ( isset($this->content[$i][$fld_name]) )
            {
              $column[] = $this->content[$i][$fld_name];
            }
        }
      return $column;
    }

  /**
   * Returns a new recordset that contains only the specified
   * columns. $fld_names is a comma separated list of field names.
   */
  function getColumns($fld_names)
    {
      $rs = new $this->type;
      $rs->Open();
      $arr_flds = explode(",", $fld_names);
      for ($i=0; $i < $this->count; $i++)
        {
          $rec = array();
          for ($j=0; $j < sizeof($arr_flds); $j++)
            {
              $fld = $arr_flds[$j];
              $rec[$fld] = $this->content[$i][$fld];
            }
          $rs->addRec($rec);
        }
      $rs->MoveFirst();
      return $rs;
    }
        
  /**
   * Returns the recordset as an HTML table
   * (mostly for debuging purposes).
   */
  function toHtmlTable()
    {
      $type = $this->type;
      if ($type=="PagedRS")
        {
          $type .= "(current_page='$this->current_page',"
            ." recs_per_page='$this->recs_per_page',"
            ."nr_of_recs='$this->nr_of_recs')";
        }
      $htmlTable = "
<br>
<table border='0' cellspacing='1' cellpadding='0'>
  <tr><td>
    <table bgcolor='#aaaaaa' border='0' cellspacing='1' cellpadding='2'>
      <tr>
        <td bgcolor='#eeeeee' align='right'>ID:</td>
        <td bgcolor='#f9f9f9'>$this->ID</td>
      </tr>
      <tr>
        <td bgcolor='#eeeeee' align='right'>Type:</td>
        <td bgcolor='#f9f9f9'>$type</td>      
      </tr>
      <tr>
        <td bgcolor='#eeeeee' valign='top' align='right'>Query:</td>
        <td bgcolor='#f9f9f9'>$this->query</td>
      </tr>
    </table
  </td></tr>
  <tr><td>
    <table bgcolor='#aaaaaa' border='0' cellspacing='1' cellpadding='2'>
";
      if ($this->count > 0)
        {
          $this->MoveFirst();
          $rec = $this->Fields();
          $htmlTable .= "\t<tr>\n";
          while ( list($fld_name, $fld_value) = each($rec) )
            {
              $htmlTable .= "\t\t<th bgcolor='#eeeeee'> $fld_name </th>\n";
            }
          $htmlTable .= "\t</tr>\n";
          while (!$this->EOF())
            {
              $rec = $this->Fields();
              $htmlTable .= "\t<tr>\n";
              while ( list($fld_name, $fld_value) = each($rec) )
                {
                  $htmlTable .= "\t\t<td bgcolor='white'> $fld_value </td>\n";
                }
              $htmlTable .= "\t</tr>\n";
              $this->MoveNext();
            }
        }
      $htmlTable .= "
    </table>
  </td></tr>
</table>
";
        
      $this->MoveFirst();
      return $htmlTable;
    }
}
?>