<?php
/*
Copyright 2001,2002,2003 Dashamir Hoxha, dashohoxha@users.sourceforge.net

This file is part of phpWebApp.

phpWebApp is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2 of the License, or
(at your option) any later version.

phpWebApp is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with phpWebApp; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA  
*/


/**
 * Parses the templates and constructs $webPage, which
 * keeps the structure of the page.
 *
 * @package parser 
 * @see WebPage, Render
 */
class Parser
{
  /** A stack of templates that are being parsed. */
  var $tpl_stack;       

  /** The template that is being parsed currently. */
  var $current_tpl;     

  /** When $this->silent is true, $this->add_line() does not add
   * lines to the current template. */
  var $silent;

  /** Parsing mode, can be "file" or "array". */
  var $mode;            

  /** The path of the file that is being parsed. */
  var $path;    

  /** The file pointer of the file that is being parsed. */
  var $fp;              

  /** The array of lines that is being parsed. */
  var $lines;   

  /** The index of line that is being parsed. */ 
  var $idx;             

  /** A stack of the parsing states. */
  var $state_stack;


  function Parser()
    {
      $this->tpl_stack = array();
      $this->current_tpl = UNDEFINED;
      $this->silent = false;
      $this->state_stack = array();
    }

  function push_parsing_state()
    {
      $curr_state["mode"] = $this->mode;
      $curr_state["path"] = $this->path;
      $curr_state["fp"] = $this->fp;
      $curr_state["lines"] = $this->lines;
      $curr_state["idx"] = $this->idx;
      array_push($this->state_stack, $curr_state);
      
      $this->lines = array();
    }

  function pop_parsing_state()
    {
      $prev_state = array_pop($this->state_stack);
      $this->mode = $prev_state["mode"];
      $this->path = $prev_state["path"];
      $this->fp = $prev_state["fp"];
      $this->lines = $prev_state["lines"];
      $this->idx = $prev_state["idx"];
    }

  /** Parse the main (root) template of the page. */
  function parse_main($tpl_file)
    {
      global $webPage;

      $webPage->tpl_file = $tpl_file;
      $webPage->tpl_collection = array();
      $webPage->rs_collection = array();
      $webPage->wb_collection = array();

      $tpl = new MainTpl($tpl_file);
      $tpl = $this->parse_file($tpl_file, $tpl);

      if ($tpl<>UNDEFINED)
        {
          $webPage->rootTpl = &$webPage->tpl_collection[$tpl->id];
        }
    }

  /**
   * Parse the filename into the template given as parameter.
   *
   * @param $filename the file to be parsed
   * @param $tpl      (optional) the template or template reference
   *                  to be filled by parsing this file.
   * @return FileTpl  an object that contains the parsed $filename,
   *                  return UNDEFINED if $filename is not found.
   */
  function parse_file($filename, $tpl =UNDEFINED)
    {   
      if ( !file_exists($filename) )
        {
          $msg = "Parser::parse_file(): file '$filename' not found";
          $err_msg = WebApp::error_msg($msg);
          $this->add_line($err_msg);
          return UNDEFINED;  
        }

      if ($tpl===UNDEFINED)   $tpl = new FileTpl($filename);
      
      //save the current parsing state
      $this->push_parsing_state();

      //set the new parsing state
      $this->mode = "file";
      $this->path = dirname($filename) . "/";
      $this->fp = fopen($filename, "r");  //open file

      global $tplVars;
      $tplVars->pushScope();            
      //add the variable {{./}} which contains the current path
      $tplVars->addVar("./", $this->path);

      //parse the file into $tpl
      $this->push($tpl);
      $this->parse();
      $tpl = $this->pop();

      $tplVars->popScope();

      //close the file
      fclose($this->fp);

      //get the previous parsing state
      $this->pop_parsing_state();

      return $tpl;
    }

  /**
   * Parse the array of lines into the template given as parameter.
   * These lines are usually the content of a web class which have
   * been copied with $this->copy_element(), because they need to
   * be parsed more then once (for each webobject of that webclass).
   *
   * @param $lines array  lines to be parsed
   * @param $path string  path of the file from which these lines 
   *                      have been extracted
   * @param $tpl Template (optional) the template or template reference
   *                      to be filled by parsing this array
   * @return Template     an object that contains the parsed lines
   */
  function parse_array($lines, $path, $tpl =UNDEFINED)
    {
      if (gettype($lines)!="array")
        {
          $msg = "Parser::parse_array():parameter is not array";
          $err_msg = WebApp::error_msg($msg);
          $this->add_line($err_msg);
          return UNDEFINED;
        }

      if ($tpl===UNDEFINED)   $tpl = new Template;

      //save the current parsing state
      $this->push_parsing_state();

      //set the new parsing state
      $this->mode = "array";
      $this->path = $path;
      $this->lines = $lines;
      $this->idx = 0;

      global $tplVars;
      $tplVars->pushScope();            
      //add the variable {{./}} which contains the current path
      $tplVars->addVar("./", $path);

      //parse the file into $tpl
      $this->push($tpl);
      $this->parse();
      $tpl = $this->pop();

      $tplVars->popScope();

      //get the previous parsing state
      $this->pop_parsing_state();

      return $tpl;
    }

  /**
   * Loads the current template from
   * the file that is being parsed currently.
   */
  function parse()
    {
      $closing_tag = false;
      while (!$closing_tag and !$this->end_of_lines())
        {
          $line = $this->get_line();
          if ($line=='')  continue;
          $closing_tag = $this->parse_line($line);
        }
    }

  /**
   * Return true when the line is the closing tag of a template;
   * if the line contains the opening tag of another template, call
   * a function to parse this template, else add the line to the
   * current template.
   */
  function parse_line($line)
    {
      $tag = Parser::get_tag_name($line);
      switch ($tag)
        {
        default:
          $this->add_line($line);
          break;

        case "<!--#":
          $this->parse_Comment($line);
          break;

        case "<Template":
        case "<If":
        case "<Repeat":
          $tag = substr($tag, 1);
          $new_Tag = "new_".$tag;
          $tpl = $this->$new_Tag($line);
          $this->parse_new_tpl($tpl);
          break;

        case "<IfEmpty":
        case "<Header":
        case "<Footer":
        case "<Separator":
          $tag = substr($tag, 1);
          $this->parse_Repeat_subtpl($tag);
          break;

        case "<head":
        case "<body":
        case "<WebBox":
        case "<WebClass":
        case "<WebObject":
          $tag = substr($tag, 1);
          $parse_Tag = "parse_".$tag;
          $this->$parse_Tag($line);
          break;
                                
        case "</Template":
        case "</If":
        case "</Repeat":
        case "</IfEmpty":
        case "</Header":
        case "</Footer":
        case "</Separator":
        case "</head":
        case "</body":
        case "</WebBox":
        case "</WebClass":
        case "</WebObject":
          return true;
          break;

        case "<html":
        case "</html":
          if ($this->current_tpl->type=="MainTpl")
            {
              $this->add_line($line);
            }
          break;
                                
        case "<RepeatBody":
        case "</RepeatBody":
          //these are optional tags inside the <Repeat> tag
          //and are just ignored
          break;                                        

        case "<Include":
        case "<Recordset":
        case "<dbCommand":
        case "<Var":
        case "<Parameter":
          $tag = substr($tag, 1);
          $parse_Tag = "parse_".$tag;
          $this->$parse_Tag($line);
          break;
        }
      return false;
    }

  /**
   * Make the given $tpl current, what will be
   * parsed from now on goes to this template.
   */
  function push(&$tpl)
    {
      //push the current_tpl on tpl_stack
      array_push($this->tpl_stack, $this->current_tpl);

      //make current the new template
      $this->current_tpl = &$tpl;

      //add the current template to the collection
      //of the templates of the $webPage
      global $webPage;
      $webPage->addTemplate($this->current_tpl);
    }

  /**
   * The parsing of the current template has finished;
   * store it in template collection (in $webPage) 
   * and make current the top of the stack.
   */
  function pop()
    {
      //add the current template to the collection
      //of the templates of the $webPage
      global $webPage;
      $webPage->addTemplate($this->current_tpl);
        
      //pop the last template from tpl_stack into current_tpl
      $tpl = $this->current_tpl;
      $last = count($this->tpl_stack) - 1;
      $this->current_tpl = &$this->tpl_stack[$last];
      array_pop($this->tpl_stack);
      return $tpl;
    }

  /**
   * Copies the content of the element $tag_name
   * into $this->current_tpl->content. The difference
   * between copy_element() and parse() is that parse()
   * builds the template structure of the element,
   * however copy_element() just copies all the lines
   * up to the closing tag, without interpreting the template tags.
   * @param $depth number   the depth of recursion
   */
  function copy_element($elem_tag, $depth =0)
    {
      $closing_tag = false;
      while (!$closing_tag and !$this->end_of_lines())
        {
          $line = $this->get_line();
          $tag = Parser::get_tag_name($line);
          switch ($tag)
            {
            default:
              $this->add_line($line);
              break;
            case "<$elem_tag":
              $this->add_line($line);
              $this->copy_element($elem_tag, $depth+1);
              break;
            case "</$elem_tag":
              if ($depth > 0)   $this->add_line($line);
              $closing_tag = true;
              break;
            }
        }
    }
        
  function parse_new_tpl($tpl)
    {
      $tpl->parent = &$this->current_tpl;

      //set a pointer from this template to the new template
      $this->add_line("##\n");
      $this->add_line($tpl->id."\n");

      //parse the new template into $tpl
      $this->push($tpl);
      $this->parse();
      $this->pop();
    }

  /**
   * Creates and returns a new If template.
   * @param $line string contains the starting tag of the If element
   * @return IfTpl
   */
  function new_If($line)
    {
      $condition = Parser::get_attr_value($line, "condition");
      $tpl = new IfTpl($condition);
      return $tpl;
    }

  /**
   * Creates and returns a new Template template.
   * @return Template 
   */
  function new_Template($line)
    {
      $tpl = new Template;
      return $tpl;
    }
        
  /**
   * Creates and returns a new Repeat template.
   * @param $line string contains the starting tag of the Repeat element
   * @return RepeatTpl
   */
  function new_Repeat($line)
    {
      $rs_id = Parser::get_attr_value($line, "rs");
      $tpl = new RepeatTpl($rs_id);
      return $tpl;
    }
        
  /**
   * $tag has the values "IfEmpty", "Header", "Footer" or "Separator"
   * @return Template
   */
  function parse_Repeat_subtpl($tag)
    {
      $tpl = new Template($this->current_tpl->id."_".$tag, $tag);

      $tpl->parent = &$this->current_tpl;

      $this->push($tpl);
      $this->parse();
      $tpl = $this->pop();

      //add the subtemplate to the Repeat template 
      if ($this->current_tpl->type<>"RepeatTpl")
        {
          $msg = "<$tag> element can be used "
            ."only inside a <Repeat> element";
          $err_msg = WebApp::error_msg($msg);
          $this->add_line($err_msg);
        }
      else
        {
          switch ($tag)
            {
            case "IfEmpty":
              $this->current_tpl->if_empty = $tpl;
              break;
            case "Header":
              $this->current_tpl->header = $tpl;
              break;
            case "Footer":
              $this->current_tpl->footer = $tpl;
              break;
            case "Separator":
              $this->current_tpl->separator = $tpl;
              break;
            }
        }
    }

  function parse_head($line)
    {
      if ($this->current_tpl->type<>"MainTpl")
        {
          //give a warning message
          $msg = "Parser: <head> element cannot be used "
            ."in this place";
          $err_msg = WebApp::warning_msg($msg);
          $this->add_line($err_msg);

          //parse it as a usual template
          $tpl = new Template;
          $this->parse_new_tpl($tpl);
          return;
        }

      //create a head template
      $tpl = new Template("HeadTpl", "HeadTpl");
      $tpl->parent = &$this->current_tpl;

      //set a pointer from this template to the new template
      $this->add_line("##\n");
      $this->add_line($tpl->id."\n");      

      //parse
      $this->push($tpl);
      $this->parse();
      $tpl = $this->pop();

      //link it to the MainTpl
      global $webPage;
      $this->current_tpl->head = &$webPage->tpl_collection[$tpl->id];
    }

  function parse_body($line)
    {
      if ($this->current_tpl->type<>"MainTpl")
        {
          //give a warning message
          $msg = "Parser: <body> element cannot be used "
            ."in this place";
          $err_msg = WebApp::warning_msg($msg);
          $this->add_line($err_msg);

          //parse it as a usual template
          $tpl = new Template;
          $this->parse_new_tpl($tpl);
          return;
        }

      //create a body template
      $tpl = new Template("BodyTpl", "BodyTpl");
      $tpl->parent = &$this->current_tpl;

      //set a pointer from this template to the new template
      $this->add_line("##\n");
      $this->add_line($tpl->id."\n");

      //parse
      $this->push($tpl);
      $this->add_line($line);
      $this->parse();
      $tpl = $this->pop();

      //link it to the MainTpl
      global $webPage;
      $this->current_tpl->body = &$webPage->tpl_collection[$tpl->id];
    }

  function parse_WebBox($line)
    {
      global $webPage, $tplVars;

      $box_id = Parser::get_attr_value($line, "ID");

      //create a new WebClassTpl template
      $webClassTpl = new WebClassTpl($box_id, $this->path);
      $webClassTpl->parent = &$this->current_tpl;

      //copy the <WebBox> element into $tpl
      $webClassTpl->before_parse();
      $this->push($webClassTpl);
      $this->copy_element("WebBox");
      $webClassTpl = $this->pop();

      //create a WebObject template for the webbox              
      $tpl = new $box_id;
      $tpl->class = $webClassTpl;
      $tpl->params = array();
      $tpl->parent = &$this->current_tpl;
      $tpl->construct_WebObject($box_id);

      //get state vars and call init(), if this is the first time
      $tpl->getStateVars();

      //set a pointer from the current template to the new template
      $this->add_line("##\n");
      $this->add_line($tpl->id."\n");

      //parse the content of the class into the object
      $tplVars->pushScope();
      $tplVars->addVars($tpl->getObjVars());
      $tpl->before_parse();
      $tpl = $this->parse_array($webClassTpl->content, $webClassTpl->path, $tpl);
      $tpl->after_parse();
      $tplVars->popScope();
      $webPage->addTemplate($tpl);
    }

  function parse_WebClass($line)
    {
      //get the attributes
      $id = Parser::get_attr_value($line, "ID");

      global $webPage;
      if ($webPage->getTemplate($id)<>UNDEFINED)
        {
          $msg = "Redeclaration of <WebClass ID=\"$id\">.";
          print WebApp::warning_msg($msg);
        }

      //create a new template
      $tpl = new WebClassTpl($id, $this->path);
      $tpl->parent = &$this->current_tpl;

      //copy the <WebClass> element into $tpl
      $tpl->before_parse();
      $this->push($tpl);
      $this->parse_Parameter_tags();
      $this->copy_element("WebClass");
      $this->pop();
    }

  function parse_Parameter_tags()
    {
      $line = $this->get_line();
      $tag = Parser::get_tag_name($line);
      while ($tag=="<Parameter")
        {
          $name = Parser::get_attr_value($line, "name");
          $default = Parser::get_attr_value($line, "default");
          $this->current_tpl->params[$name] = $default;
          //get next line
          $line = $this->get_line();
          $tag = Parser::get_tag_name($line);  
        }
      //copy the last line (that does not have 
      //a <Parameter> tag) to the current template
      $this->add_line($line);
    }

  function parse_WebObject($line)
    {
      global $webPage, $tplVars;
                
      //get the other lines that belong to <WebObject ... /> tag
      $tag = "";
      while (!ereg('>[[:space:]]*$', $line))
        {
          $tag .= $line;
          $line = $this->get_line();
        }
      $tag .= $line;

      //get the attributes of the webobject
      $class_name = Parser::get_attr_value($tag, "Class");
      $obj_name = Parser::get_attr_value($tag, "Name");
      $obj_name = WebApp::replaceVars($obj_name);
      $full_obj_name = $class_name . "::" . $obj_name;

      //get the webclass of this webObj
      if (isset($webPage->tpl_collection[$class_name]))
        {
          $webClassTpl = &$webPage->tpl_collection[$class_name];
        }
      else
        {
          $msg = "Template::parse_WebObject: <WebClass> '$class_name' is not defined.";
          $this->add_line(WebApp::error_msg($msg));
          return;
        }

      //create a WebObject template
      $tpl = new $class_name;
      $tpl->class = &$webPage->tpl_collection[$class_name];
      $tpl->parent = &$this->current_tpl->parent;
      $tpl->params = $webClassTpl->params;
      $tpl->construct_WebObject($full_obj_name);

      //get the values of the parameter attributes
      while (list($p_name,$p_default) = each($tpl->params))
        {
          $p_value = Parser::get_attr_value($tag, $p_name);
          if ($p_value != UNDEFINED) 
            {
              $tpl->params[$p_name] = $p_value;
            }
        }

      //get state vars and call init(), if this is the first time
      $tpl->getStateVars();

      //set a pointer from the current template to the new template
      $this->add_line("##\n");
      $this->add_line($tpl->id."\n");

      //parse the content of the class into the object
      $tplVars->pushScope();
      $tplVars->addVars($tpl->getObjVars());
      $tpl->before_parse();
      $tpl = $this->parse_array($webClassTpl->content, $webClassTpl->path, $tpl);
      $tpl->after_parse();
      $tplVars->popScope();
      $webPage->addTemplate($tpl);
    }
        
  function parse_Include($line)
    {
      //get the filename
      $tpl_filename = Parser::get_attr_value($line, "SRC");
      $tpl_filename = WebApp::replaceVars($tpl_filename);

      //create a new file template
      $tpl = new FileTpl($tpl_filename);

      //set the indentation of this FileTpl
      ereg("(^ *)", $line, $regs);
      $tpl->indentation = $regs[1];

      //parse the file into this FileTpl
      $tpl = $this->parse_file($tpl_filename, $tpl);

      if ($tpl<>UNDEFINED)
        {
          //set a pointer from the current template to the new template
          $this->add_line("##\n");
          $this->add_line($tpl->id."\n");
        }
    }
        
  function parse_Recordset($line)
    {
      global $webPage;
                
      //get the attributes 'ID' and 'type'
      $id = Parser::get_attr_value($line, "ID");
      $type = Parser::get_attr_value($line, "type");
      if ($type=="PagedRS")
        {
          $rp = Parser::get_attr_value($line, "recs_per_page");  
        }
                
      //get the query (between <Query> and </Query>)
      $query = $this->parse_Query();

      //consume the closing tag "</Recordset>
      $line = $this->get_line();
                
      //create the Recordset object
      switch ($type)
        {
        default:
          $msg = "Recordset '$id': type '$type' is unknown";
          print WebApp::error_msg($msg);
        case UNDEFINED:
        case "EditableRS":
          $rs = new EditableRS($id, $query);
          break;
        case "StaticRS":
          $rs = new StaticRS($id, $query);
          break;
        case "DynamicRS":
          $rs = new Recordset($id, $query);
          break;
        case "PagedRS":
          $rs = new PagedRS($id, $query, $rp);
          break;
        case "TableRS":
          $rs = new TableRS($id, $query);
          break;
        }

      $webPage->addRecordset($rs);
    }
        
  function parse_dbCommand($line)
    {
      global $webPage, $session;
                
      //get the attributes 'ID' and 'recs_per_page'
      $id = Parser::get_attr_value($line, "ID");
                
      //get the query (between <Query> and </Query>)
      $query = $this->parse_Query();

      //consume the closing tag "</dbCommand>
      $line = $this->get_line();
                
      //create the Recordset object
      $rs = new Recordset($id, $query);
                
      $webPage->addRecordset($rs);
    }   

  /** returns the query (between <Query> and </Query>) */
  function parse_Query()
    {
      $line = $this->get_line();        
      if ( ereg("<Query>(.*)</Query>", $line, $regs) )
        {       
          //all the query is in one line
          $query = trim($regs[1]);
        }
      else      //the query is in several lines
        {
          $line = $this->get_line();    
          while (trim($line)<>"</Query>")
            {
              $query .= $line;
              $line = $this->get_line();        
            }
        }
      return $query;
    }
        
  function parse_Var($line)
    {
      ereg('<Var name="([^"]*)">(.*)', $line, $regs);
      $var_name = $regs[1];
      $line = $regs[2];
      while ( !ereg('(.*)</Var>', $line, $regs) )
        {
          $expression .= $line;
          $line = $this->get_line();    
        }
      $expression .= $regs[1];
      $this->current_tpl->addVar($var_name, $expression);
    }
        
  function parse_Comment($line)
    {
      $end_of_comment = ereg("#-->", $line);
      while (!$end_of_comment)
        {
          $line = $this->get_line();
          $end_of_comment = ereg("#-->", $line);
        }
    }

  /*------------------------------------------------------------*/

  function end_of_lines()
    {
      if ($this->mode=="file")
        {
          $end = feof($this->fp);
        }
      else if ($this->mode=="array")
        {
          $cnt = count($this->lines);
          $end = ($this->idx >= $cnt);
        }
      return $end;
    }

  /** returns a line from the file that is being parsed */
  function get_line()
    {
      if ($this->mode=="file")
        {
          $line = fgets($this->fp,896);
        }
      else if ($this->mode=="array")
        {
          $line = $this->lines[$this->idx];
          $this->idx++;
        }
      return $line;
    }

  /** add a line to the current template */
  function add_line($line)
    {
      if ($this->silent)  return;

      if ($this->current_tpl===UNDEFINED)
        {
          print $line;
        }
      else
        {
          $this->current_tpl->content[] = $line;
        }
    }

  /**
   * If the $line starts with an opening or closing tag
   * ('<tag_name' or '</tag_name') then it returns it,
   * otherwise returns UNDEFINED.
   */
  function get_tag_name($line)
    {
      $line = trim($line);
      if ( ereg("^(</?[^[:space:] >]*)", $line, $regs) )
        {
          return $regs[1];
        }
      else
        {
          return UNDEFINED;
        }
    }
        
  /**
   * Returns the value of the given attribute from the given
   * line (assuming that the line is a start element), or
   * UNDEFINED if not found.
   */
  function get_attr_value($line, $attrib)
    {
      $line = trim($line);
      $pattern = '[[:space:]]+'.$attrib.'[[:space:]]*=[[:space:]]*"([^"]*)"';
      if ( ereg($pattern, $line, $regs) )
        {
          $attr = $regs[1];
          return $attr;
        }
      else
        {
          return UNDEFINED;
        }
    }

  /////////////////////////////////////////////////////////////

  /** Output the state of the parser. */
  function debug()
    {
      WebApp::debug_msg($this->current_tpl->toHtmlTable(),
                        "\$this->current_tpl");
      for ($i=0; $i < sizeof($this->tpl_stack); $i++)
        {
          if (gettype($this->tpl_stack[$i])=="Object")
            {
              WebApp::debug_msg($this->tpl_stack[$i]->toHtmlTable(),
                                "\$this->tpl_stack[\$i]");
            }
          else
            {
              WebApp::debug_msg($this->tpl_stack[$i],
                                "\$this->tpl_stack[\$i]");
            }
        }
    }
}
?>