/*
 * Photo Image Print System
 * Copyright (C) 2001-2004 EPSON KOWA Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2001-2004.
 *
 * This file is part of the `dtrfilter' program.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON KOWA Corporation gives permission
 * to link the code of this program with the `dtr' library and
 * distribute linked combinations including the two.  You must obey
 * the GNU General Public License in all respects for all of the
 * code used other then `dtr'.
 */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <getopt.h>
#include "defs.h"
#include "img.h"
#include "libdtr.h"

#pragma pack(1)
typedef struct
{
	char head[14];
	unsigned long width_byte;
	unsigned long width;
	unsigned long height;
} ATBMP_HEADER, *P_ATBMP_HEADER;
#pragma pack()

static const char head_string[] = "PIPS_Attribute";

static char* progname_ = NULL;
static char usage_[] = "Usage : dtrfilter [-h] [input [output]]";

static int att_loop (FILE*, FILE*);
static void dtr_fatal (char*);

int
main (int argc, char* argv[])
{
	/* Variable declaration part */
	FILE *infp, *outfp;	/* I/O */
	char c;	/* Key word of getopt_long. */
	int index = 0;
	struct option allopt[] =
	{
		{ "help",   0, 0, 'h' }
	}; /* option list of getopt_long. */
	
	progname_ = argv[0];

	while ((c = getopt_long (argc, argv, "h", allopt, &index)) != -1)
	{
		switch (c)
		{
		case 'h':
		default:
			fprintf (stderr, "%s : %s\n", progname_, usage_);
			return 0;
		}
	}

	infp = stdin;
	outfp = stdout;

	return att_loop (infp, outfp);
}

static int
att_loop (FILE* infp, FILE* outfp)
{
	void* p_img;	        /* image handle */
	BMP inbmp, atbmp;       /* bitmap struct */
	int width, height;      /* image size */
	long size;            /* image buffer size */
	ATBMP_HEADER header;

	for (;;)
	{
		p_img = img_init (infp);
		if (!p_img) return 0; /* Normal termination */
		
		width = img_get_width (p_img);
		height = img_get_height (p_img);
		size = width * height;
		
		inbmp.WidthBytes = width * 3;
		atbmp.WidthBytes = width;

		
		inbmp.Bits = calloc (1, size * 3);
		atbmp.Bits = malloc (size);

		if (!inbmp.Bits || !atbmp.Bits)
			dtr_fatal ("Filed memory allocation.");
		memset (atbmp.Bits, 0x01, size);
		
	        img_get_image (p_img, inbmp.WidthBytes, inbmp.Bits);
		if (inbmp.Bits == NULL)
			dtr_fatal ("Cannot read png.");

		if (OutputMemAtBmp (width, height, &inbmp, &atbmp) < 0)
			dtr_fatal ("Failed in making of attribute bitmap.");

		memcpy (header.head, head_string, strlen (head_string));
		header.width_byte = width;
		header.width = width;
		header.height = height;

		if (fwrite (&header, 1, sizeof (header), outfp) != sizeof (header))
			dtr_fatal ("Cannot output attribute bitmap.");
		
		if (fwrite (atbmp.Bits, 1, size, outfp) != size)
			dtr_fatal ("Cannot output attribute bitmap.");

		free (inbmp.Bits);
		free (atbmp.Bits);
		
		img_end (p_img);
	}

	free (inbmp.Bits);
	free (atbmp.Bits);
	
	img_end (p_img);
	return 1;
}

static void
dtr_fatal (char* msg)
{
	fprintf (stderr, "%s : %s\n", progname_, msg);
	exit (1);
}

