/*
 * Photo Image Print System
 * Copyright (C) 2001-2004 EPSON KOWA Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2001-2004.
 *
 * This file is part of the `ekpstm' program.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif
#include <gtk/gtk.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>

#include "interface.h"
#include "support.h"

#include "stm.h"
#include "ekpcom.h"
#include "inkbox.h"
#include "error.xpm"

#if COVER_OPEN || CD_DVD_OPEN
#include "error_i2.xpm"
#endif
#include "error_i3.xpm"
#include "error_i4.xpm"
#include "error_i6.xpm"
#include "error_i8.xpm"
#include "error_ia.xpm"
#include "printing1.xpm"
#include "printing2.xpm"
#include "printing3.xpm"
#include "ready.xpm"
#include "ready_i1.xpm"
#include "warning.xpm"

typedef enum
{
	STS_ERR_FATAL,
	STS_ERR_JAM,
	STS_ERR_NO_INK,
	STS_ERR_NO_PAPER,
	STS_ERR_PRINTER,
	STS_NOT_READY,
	STS_PRINTING,
	STS_IDLE,
	STS_NO_REPLY,
	STS_MOTOR_OFF,		/* add sk     -Fri Mar  9 2001 */
	STS_EKPD_DOWN,		/* add sk     -Mon Mar 12 2001 */
	STS_ERR_PART_CHANGE,
	STS_ERR_ROLL,
	STS_ERR_REPETITION,
	STS_PRINTER_RESET,
	STS_ERR_OTHER_IF,
	STS_ERR_LEVER,
	STS_ERR_TRAY,
	STS_ERR_THICK_PAPER,
	STS_ERR_TRAY_AND_LEVER,
	STS_ERR_CUTTER,
	STS_ERR_CUT_MISS,
	STS_ERR_OTHER_INK,
	STS_ERR_COMBINATION,
	STS_ERR_COPYING,	/* Added for Scanner-Pirinter-Copy  */
	STS_ERR_COVER_OPEN,	/* Added for Scanner-Pirinter-Copy  */
	STS_ERR_CD_DVD_OPEN,	/* Added for CD/DVD Tray Open	    */
	STS_DEFAULT
}PRINTER_STS;

typedef struct INK_NODE {
	unsigned long id;
	int rest;
	struct INK_NODE *next;
} InkNode, *InkList;

typedef struct{
	GtkWidget *window;
	GtkWidget *prt_pix;

	PRINTER_STS cur_status;
	gint num_pict;
	gint cur_pict;
	gint timer;
	GdkPixmap *pixmap[4];
	GdkBitmap *mask[4];
	gchar **inkbox_xpm;
	gint inkbox_use_flag[INK_NUMBER];
	gint inkbox_counter;
	int stm_st;			/* ơ˥ȤΥơե饰 */
	InkList ink_list;
}PRINTSTSSTRUCT;


static PRINTSTSSTRUCT _status_str;

/* 2004.03.08		*/	
/* ink color name 	*/
static const char *ink_name[] = {
  N_("NBlack"),
  N_("Cyan"),
  N_("Magenta"),
  N_("Yellow"),
  N_("Light Cyan"),
  N_("Light Magenta"),
  N_("Dark Yellow"),
  N_("Photo Black"),
  N_("Matte Black"),
  N_("Red"),
  N_("Blue"),
  N_("Gloss Optimizer")
};

static PRINTER_STS ReadStatuslogfile (InkList *);
static void SetAllUserButtonHidden (void);
static void DisplayInformation (GtkText*, char*);
static int GetPrinterStatus (gpointer);
static void auto_down_count_start (void);
static void auto_down_count (int);
static void setInkProg(InkList);
static void ink_list_delete (InkList);

#define OFFSET_SEVENTH_INK_CARTRIDGE 34

static PRINTER_STS
ReadStatuslogfile (InkList *list)
{
	char *lpInk, *lpSts, *lpErr;
	char lpReply[1024], StsCode[3], ErrCode[3];
	char get_status_command[] = { 's', 't', 0x01, 0x00, 0x01 };
	int com_len, rep_len;
	
	/*  */
	StsCode[2] = 0;
	ErrCode[2] = 0;
	*list = NULL;

	com_len = sizeof (get_status_command);
	rep_len = 1024;
	/* եɤ߹ */
	if (sock_write (get_status_command, &com_len) < 0)
	{
		if (sock_reopen () < 0)
		{
			return STS_EKPD_DOWN;
		}
		return STS_NO_REPLY;
	}
  
	sock_read (lpReply, &rep_len);
	if (rep_len == 0)
	{
		return STS_NO_REPLY;
	}

	/* 󥯼 */
	lpInk = strstr (lpReply, "INK:");
	if (lpInk){
		char ic[7] = "0x";
		InkNode *prev = NULL;

		lpInk += 4;	/* skip "INK:" */
		while (*lpInk != ';') {
			InkNode *node;

			node = (InkNode *)calloc(1, sizeof(InkNode));
			if (!node) {
				return STS_NO_REPLY;
			}

			memcpy(ic + 2, lpInk, 4);
			*(ic + 6) = '\0';
			node->id = strtol(ic, (char **)NULL, 16);
			
			if (prev) {
				prev->next = node;
				prev = prev->next;
			} else {
				prev = node;
				*list = prev;
			}
			lpInk += 4;
			if (*lpInk == ',') {
				lpInk++;
			}
			node = node->next;
		}
	}else{
		char snink[3];
		int  nink, i;
		InkNode *prev = NULL;

		/* 2004.03.18		*/
		/* INK:... not reply 	*/
		/* check "AI:CW:"	*/
		lpInk = strstr (lpReply, "AI:CW:");
		if( !lpInk )	
			return STS_NO_REPLY;
		/* Found!		*/	
		lpInk += 6;	/* skip "AI:CW:" */

		/* get no. of ink */
		sprintf( snink, "%.2s", lpInk);
		nink = atoi( snink );		
		lpInk += 2;

		/* setup ink list */
		for( i = 0; i < nink ; i++ ){
			InkNode *node;

			node = (InkNode *)calloc(1, sizeof(InkNode));
			if (!node) {
				return STS_NO_REPLY;
			}

			switch( i ){
			case	0: /* black */
				node->id = 0x1101;break;	
			case	1: /* cyan */
				node->id = 0x3202;break;
			case	2: /* yellow */
				node->id = 0x4304;break;
			case	3: /* magenta */
				node->id = 0x5408;break;
			case	4: /* light cyan */
				node->id = 0x6210;break;
			case	5: /* light magenta */
				node->id = 0x7320;break;
			case	6: /* dark yellow */
				node->id = 0x9440;break;
			case	7: /* light black */
				node->id = 0x1140;break;
			default	:
				node->id = 0x0;break;
			   break;
			}
			
			if (prev) {
				prev->next = node;
				prev = prev->next;
			} else {
				prev = node;
				*list = prev;
			}
			node = node->next;
		}
	}


	/* 󥯻̲ */
	lpInk = strstr(lpReply, "IQ:");
	if(lpInk){
		char ink[3];
		char *stopstring;
		InkNode *node = *list;

		lpInk += 3;
		memset (ink, 0, 3);

#if (USE_INKBOX)
		while (node)
		{
			strncpy(ink, lpInk, 2);
			node->rest = (gint)strtol (ink, &stopstring, 16);
			node = node->next;
			lpInk +=2;
			if (*lpInk == ';') break;
		}
#else
		strncpy(ink, lpInk, 2);
		node->rest = (gint)strtol (ink, &stopstring, 16);
		node = node->next;
		lpInk +=2;
		node->rest = 100;
		while (*lpInk != ';') {
			int inkval;

			strncpy(ink, lpInk, 2);
			inkval = (gint)strtol (ink, &stopstring, 16);
			node->rest = inkval < node->rest ? inkval : node->rest;
			lpInk +=2;
		}
#endif /* USE_INKBOX */
	}

	/* ơ */
	lpSts = strstr(lpReply, "ST:");
	if(lpSts){
		strncpy(StsCode, lpSts + 3, 2);
		if(!strcmp(StsCode, "00")){
			/* 顼ơ */
			lpErr = strstr(lpReply, "ER:");
			
#if DEBUG
			fprintf(stderr, "error code = %s\n", lpErr);
#endif
			if(lpErr){
				strncpy(ErrCode, lpErr + 3, 2);
				if(!strcmp(ErrCode, "04")) return STS_ERR_JAM;
				if(!strcmp(ErrCode, "05")) return STS_ERR_NO_INK;
				if(!strcmp(ErrCode, "06")) return STS_ERR_NO_PAPER;
				if(!strcmp(ErrCode, "10")) return STS_ERR_PART_CHANGE;
				if(!strcmp(ErrCode, "11")) return STS_ERR_ROLL;
				if(!strcmp(ErrCode, "12")) return STS_ERR_REPETITION;
				/* added - Sat Feb 22 2003 */
				if(!strcmp(ErrCode, "00")) return STS_ERR_FATAL;
#if SPC_MODEL
				if(!strcmp(ErrCode, "01")) return STS_ERR_COPYING;
#else
				if(!strcmp(ErrCode, "01")) return STS_ERR_OTHER_IF;
#endif

#if COVER_OPEN				
				/* Added Status 2003.11.25	*/
				if(!strcmp(ErrCode, "02")) return STS_ERR_COVER_OPEN;
#elif CD_DVD_OPEN
				/* Added Status 2004.01.08	*/
				if(!strcmp(ErrCode, "02")) return STS_ERR_CD_DVD_OPEN;
#endif	
				
				if(!strcmp(ErrCode, "03")) return STS_ERR_LEVER;
				if(!strcmp(ErrCode, "0C")) return STS_ERR_TRAY;
				if(!strcmp(ErrCode, "0D")) return STS_ERR_THICK_PAPER;
				if(!strcmp(ErrCode, "15")) return STS_ERR_TRAY_AND_LEVER;
				if(!strcmp(ErrCode, "1C")) return STS_ERR_CUTTER;
				if(!strcmp(ErrCode, "1D")) return STS_ERR_CUT_MISS;
				if(!strcmp(ErrCode, "1E")) return STS_ERR_OTHER_INK;
				if(!strcmp(ErrCode, "23")) return STS_ERR_COMBINATION;

			}
			return STS_ERR_FATAL;
		}
#if DEBUG
		/* Debug */
		fprintf(stderr, "status code = %s\n", StsCode);
#endif

		if(!strcmp(StsCode, "01") || !strcmp(StsCode, "07")) return STS_NOT_READY;
		if(!strcmp(StsCode, "03") || !strcmp(StsCode, "02")) return STS_PRINTING;
		if(!strcmp(StsCode, "04")) return STS_IDLE;
		if(!strcmp(StsCode, "09")) return STS_MOTOR_OFF; /* add sk     -Fri Mar  9 2001 */
		if(!strcmp(StsCode, "CN")) return STS_PRINTER_RESET;
	}

	/* Debug */
	// fprintf(stderr, "...No Reply\n", StsCode);
	
	return STS_NO_REPLY;
}

/* ֤ˤäɽ֤Ѳܥ򤹤٤ɽˤ */
static void
SetAllUserButtonHidden ()
{
	GtkWidget *widget;
  
	widget = lookup_widget (_status_str.window, "eject_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "continue_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "reset_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "ink_button");
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "cancel_button");
	gtk_widget_set_sensitive (widget, TRUE); /* add sk  - Wed Mar 14 2001 */
	gtk_widget_hide (widget);

	widget = lookup_widget (_status_str.window, "close_button");
	gtk_widget_hide (widget);

	return;
}

/* BOX˾ܺپɽ */
static void
DisplayInformation (GtkText *text, char *info)
{
	/* ֤Ƭˤ */
	gtk_text_set_point (text, 0);
  
	/* ξɽ򥯥ꥢ */
	/*  gtk_text_forward_delete(text, -1); */
	gtk_text_forward_delete (text, gtk_text_get_length (text));
  
	/* ܺپɽ */
	gtk_text_insert (text, NULL, NULL, NULL, info, -1);
  
	/* ưƬ᤹ᡢƬ֤˥ߡʸƺ */
	gtk_text_set_point (text, 0);
	gtk_text_insert (text, NULL, NULL, NULL, " ", -1);
	gtk_text_backward_delete (text, 1);
	return;
}

static int
GetPrinterStatus (gpointer data)
{
	PRINTER_STS status;
	gint i;
	GtkWidget *widget, *txt, *label;
	InkList ink_list;

	/* 󥯻̤ι */
	status = ReadStatuslogfile(&ink_list);
	setInkProg (ink_list);
	ink_list_delete(ink_list);

	/* Auto down */
	if (status == STS_IDLE)
		auto_down_count (0);	/* count */
	else
		auto_down_count (1);	/* reset */
  
  
	if (status == STS_EKPD_DOWN)
	{
		widget = create_msg_no_ekpd ();
		gtk_widget_show (widget);
		return 0;
	}
  
	if (status == STS_MOTOR_OFF)
		status = _status_str.cur_status;
  
	if (status == _status_str.cur_status)
	{
		/* add sk     - Fri Mar  9 2001 */
		if ((status == STS_ERR_FATAL || status == STS_ERR_JAM)
		    &&  _status_str.stm_st == 2)
		{
			SetAllUserButtonHidden();

			widget = lookup_widget (_status_str.window, "reset_button");
			gtk_widget_show(widget);
			_status_str.stm_st = 0;
		}
      
		/* ˥᡼ι */
		if(_status_str.num_pict == 1)
			return 1;

		_status_str.cur_pict++;

		if(_status_str.cur_pict > _status_str.num_pict - 1)
			_status_str.cur_pict = 0;

		gtk_pixmap_set (GTK_PIXMAP(_status_str.prt_pix),
				_status_str.pixmap[_status_str.cur_pict],
				_status_str.mask[_status_str.cur_pict]);
		return 1;
	}
  
	if (_status_str.cur_status != STS_DEFAULT)
	{
		/* ܤˤPIXMAPλȤ򳰤 */
		for (i = 0; i < _status_str.num_pict; i++)
		{
			gdk_pixmap_unref (_status_str.pixmap[i]);
			gdk_bitmap_unref (_status_str.mask[i]);
		}
	}
  
	/* ܤˤPIXMAPκꡢɽӥܥξ֤ѹ */
	_status_str.cur_pict = 0;
	_status_str.stm_st = 0;

	label = lookup_widget (_status_str.window, "st_label");
	txt = lookup_widget (_status_str.window, "st_txt");

	switch(status)
	{
	case STS_ERR_FATAL:
		_status_str.stm_st = 1;
      
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("General error"));
		DisplayInformation(GTK_TEXT(txt), _("Delete all print jobs and turn the printer off. "
						    "Remove any foreign objects from inside the printer. "
						    "After a few minutes, turn the printer back on. "
						    "If the problem persists, contact your dealer."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_JAM:
		_status_str.stm_st = 1;
      
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i8);
		gtk_label_set_text(GTK_LABEL(label), _("Paper jam"));
		DisplayInformation(GTK_TEXT(txt), _("Press the Maintenance button on the printer or click the Eject button when it appears on the screen. Remove any remaining jammed paper by hand."));

		SetAllUserButtonHidden();

#ifndef NO_ACTION
		widget = lookup_widget (_status_str.window, "eject_button");
		gtk_widget_show(widget);
#endif
		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);

		break;

	case STS_ERR_NO_INK:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i4);

		gtk_label_set_text(GTK_LABEL(label), _("Ink out"));
		DisplayInformation(GTK_TEXT(txt), _("Ink is out.\n"
						    "Replace the ink cartridge."));

		SetAllUserButtonHidden();

#ifdef INK_CHENGE_OK
		widget = lookup_widget (_status_str.window, "ink_button");
		gtk_widget_show(widget);
#endif

		if (_status_str.cur_status == STS_IDLE || _status_str.cur_status == STS_DEFAULT)
			widget = lookup_widget (_status_str.window, "close_button");
		else
			widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);

		break;

	case STS_ERR_NO_PAPER:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Paper not loaded correctly"));
		DisplayInformation(GTK_TEXT(txt), _("Reload the paper and press the Maintenance button on the printer or click the Continue button when it appears on the screen. "
						    "To cancel print jobs, click the Cancel button."));
		SetAllUserButtonHidden();

#ifndef NO_ACTION
		widget = lookup_widget (_status_str.window, "continue_button");
		gtk_widget_show(widget);
#endif
		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);

		break;
		/*
		  case STS_ERR_PRINTER:
		  _status_str.num_pict = 2;
		  _status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
		  &_status_str.mask[0], NULL,
		  (gchar**)error);
		  _status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
		  &_status_str.mask[1], NULL,
		  (gchar**)error_ia);
		  gtk_label_set_text(GTK_LABEL(label), _(""));
		  DisplayInformation(GTK_TEXT(txt), _(""));
		  SetAllUserButtonHidden();

		  widget = lookup_widget (_status_str.window, "close_button");
		  gtk_widget_show(widget);
		  break;
		*/

	case STS_ERR_PART_CHANGE:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("Maintenance call"));
		DisplayInformation(GTK_TEXT(txt), _("Parts inside your printer require maintenance or replacement. "
						    "Contact your dealer."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
      
	case STS_ERR_ROLL:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("The paper has fed incorrectly"));
		DisplayInformation(GTK_TEXT(txt), _("Press the roll paper button to return the paper to the correct position."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
      
	case STS_ERR_REPETITION:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Multi-page feed error"));
		DisplayInformation(GTK_TEXT(txt), _("Several pages have been fed into the printer at once. "
						    "Remove and reload the paper, then press the maintenance button."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_OTHER_IF:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)ready);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)ready_i1);
		gtk_label_set_text(GTK_LABEL(label), _("Printing data from another port"));
		DisplayInformation(GTK_TEXT(txt), _("Please wait."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_LEVER:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);

		gtk_label_set_text(GTK_LABEL(label), _("Incorrect release lever position"));
		DisplayInformation(GTK_TEXT(txt), _("Move the release lever to the correct position."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_TRAY:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Paper out or incorrect output tray position"));
		DisplayInformation(GTK_TEXT(txt), _("Moving the output tray to the lower position starts printing automatically. "
						    "To cancel print jobs, click the Cancel button."));
		SetAllUserButtonHidden();

		/* 2004.03.02 close button -> cancel button */
		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);
		
		break;

	case STS_ERR_THICK_PAPER:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Paper thickness error"));
		/* PM-970C */
		DisplayInformation(GTK_TEXT(txt), _("Move the release lever to the lower position and remove the loaded paper "
						    "from the manual feed slot. "
						    "Load the correct thickness of paper for this printer, "
						    "then press the paper button."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_TRAY_AND_LEVER:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i6);
		gtk_label_set_text(GTK_LABEL(label), _("Incorrect output tray or adjust lever position"));
		/* PM-970C */
		DisplayInformation(GTK_TEXT(txt), _("Cancel all print jobs and turn the printer off.\n"
						    "Move the output tray or adjust lever to the correct position as follows.\n"
						    "Defined paper or post card:\n"
						    "Output tray - lower position, Adjust lever - left side\n"
						    "Envelope:\n"
						    "Output tray - lower position, Adjust lever - right side\n"
						    "CD/DVD/Thick paper:\n"
						    "Output tray - upper position, Adjust lever - left"));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_CUTTER:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("Cutter error"));
		DisplayInformation(GTK_TEXT(txt), _("Turn the printer off, and remove any jammed paper. "
						    "Wait a few minutes, then turn the printer back on."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_CUT_MISS:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_ia);
		gtk_label_set_text(GTK_LABEL(label), _("Unable to cut"));
		DisplayInformation(GTK_TEXT(txt), _("Turn the printer off, and remove any jammed paper. "
						    "Wait a few minutes, then turn the printer back on."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_OTHER_INK:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i4);
		gtk_label_set_text(GTK_LABEL(label), _("Different ink cartridge installed"));
		DisplayInformation(GTK_TEXT(txt), _("Replacing the current ink cartridge with the previously "
						    "installed color ink cartridge starts printing automatically. "
						    "If you do not want to replace the ink cartridge, click the Cancel button."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);
		break;

	case STS_ERR_COMBINATION:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i4);
		gtk_label_set_text(GTK_LABEL(label), _("Incompatible combination of ink cartridges installed"));
		/* PM-970C */
		DisplayInformation(GTK_TEXT(txt), _("Replace the current ink cartridge witn the correct ink combination. "
						    "The following combinations are valid:\n"
						    "7 colors: Dark Yellow/Light Magenta/Light Cyan/Black/Cyan/Magenta/Yellow\n"
						    "4 colors: Yellow/Magenta/Cyan/Black/Cyan/Magenta/Yellow"));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
      
	case STS_NOT_READY:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)ready);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)ready_i1);
		gtk_label_set_text(GTK_LABEL(label), _("Printer busy"));
		DisplayInformation(GTK_TEXT(txt), _("The printer is busy cleaning the print head or charging ink. "
						    "Please wait."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_PRINTING:
		auto_down_count_start ();

		_status_str.num_pict = 3;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)printing1);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)printing2);
		_status_str.pixmap[2] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[2], NULL,
								     (gchar**)printing3);
		gtk_label_set_text(GTK_LABEL(label), _("Printing"));
		DisplayInformation(GTK_TEXT(txt), _("Please wait."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);

		break;

	case STS_IDLE:
		_status_str.num_pict = 1;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)ready);
		gtk_label_set_text(GTK_LABEL(label), _("Ready"));
		DisplayInformation(GTK_TEXT(txt), _("Ready to print."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

	case STS_PRINTER_RESET:
		_status_str.stm_st = 1;
      
		_status_str.num_pict = 1;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)warning);
		gtk_label_set_text(GTK_LABEL(label), _("Cancel"));
		DisplayInformation(GTK_TEXT(txt), _("Turn the printer off, and remove any jammed paper. "
						    "After a few minutes, turn the printer back on."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;

#if SPC_MODEL
	/* Added new error message */
	case STS_ERR_COPYING:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)ready);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)ready_i1);
		gtk_label_set_text(GTK_LABEL(label), _("Copying"));
		DisplayInformation(GTK_TEXT(txt), _("Please wait."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
#endif
#if COVER_OPEN
	/*  Added new error message */
	case STS_ERR_COVER_OPEN:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i2);
		gtk_label_set_text(GTK_LABEL(label), _("Printer cover open"));
		DisplayInformation(GTK_TEXT(txt), _("Close the printer cover."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);
		break;
#endif
#if CD_DVD_OPEN
	/*  Added new error message */
	case STS_ERR_CD_DVD_OPEN:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i2);
		gtk_label_set_text(GTK_LABEL(label), _("CD/DVD guide open"));
		DisplayInformation(GTK_TEXT(txt), _("Remove the CD/DVD tray, then close the CD/DVD guide."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "cancel_button");
		gtk_widget_show(widget);
		break;
#endif
	

	default:
		_status_str.num_pict = 2;
		_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[0], NULL,
								     (gchar**)error);
		_status_str.pixmap[1] = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
								     &_status_str.mask[1], NULL,
								     (gchar**)error_i3);
		gtk_label_set_text(GTK_LABEL(label), _("Communication error"));
		DisplayInformation(GTK_TEXT(txt), _("Check all connections and make sure all devices are on. "
						    "If the power was turned off during printing, cancel the print job. "
						    "If the error does not clear, see your manual."));
		SetAllUserButtonHidden();

		widget = lookup_widget (_status_str.window, "close_button");
		gtk_widget_show(widget);
		break;
	}

	gtk_pixmap_set(GTK_PIXMAP(_status_str.prt_pix),
		       _status_str.pixmap[0],
		       _status_str.mask[0]);
	_status_str.cur_status = status;
	return 1;
}



static int _down_count_start;
static int _down_limit;
static int _down_counter;

void
auto_down_set (int sec)
{
	_down_count_start = 0;
	_down_limit = sec;
	_down_counter = 0;
	return;
}

static void
auto_down_count_start (void)
{
	_down_count_start = 1;
	return;
}

static void
auto_down_count (int flag)
{
	if (_down_limit <= 0 || _down_count_start == 0)
		return;

	if (flag)			/* Refresh */
		_down_counter = 0;
	else
		_down_counter ++;

	if (_down_counter >= _down_limit)
		gtk_main_quit ();

	return;
}

void
init_stm (GtkWidget* window)
{
	/* ɽwidget */
	_status_str.prt_pix = lookup_widget (window, "prt_pix");
  
	_status_str.num_pict = 1;
	_status_str.pixmap[0] = gdk_pixmap_create_from_xpm_d(window->window,
							     &_status_str.mask[0], NULL,
							     (gchar**)warning);
	gtk_pixmap_set(GTK_PIXMAP(_status_str.prt_pix),
		       _status_str.pixmap[0],
		       _status_str.mask[0]);
	gtk_widget_show (_status_str.prt_pix);

	_status_str.window = window;
	_status_str.cur_status = STS_DEFAULT;
	_status_str.cur_pict = 0;
	_status_str.num_pict = 1;
	_status_str.stm_st = 0;
	_status_str.timer = gtk_timeout_add(1000, GetPrinterStatus, NULL);
	_status_str.inkbox_xpm = inkbox_xpm_new ();
	_status_str.inkbox_counter = 0;
	return;
}


void
end_stm (void)
{
	int i;
  
	/* PIXMAPλȤ򳰤 */
	for (i = 0; i < _status_str.num_pict; i++)
	{
		gdk_pixmap_unref (_status_str.pixmap[i]);
		gdk_bitmap_unref (_status_str.mask[i]);
	}

	/* ޡκ */
	gtk_timeout_remove(_status_str.timer);
  
	/* socket */
	sock_close ();
  
	if (_status_str.inkbox_xpm)
		inkbox_xpm_ref (_status_str.inkbox_xpm);

	gtk_main_quit();
}

void
set_cancel_status (void)
{
	GtkWidget* widget;

	/* "顼"̵
	   if (_status_str.stm_st == 1)
	   _status_str.stm_st = 2;
	*/
	widget = lookup_widget (_status_str.window, "cancel_button");
	gtk_widget_set_sensitive (widget, FALSE); /* add sk  - Wed Mar 14 2001 */
	return;
}

#if (USE_INKBOX)
static void
setInkProg(InkList list)
{
	InkNode *node = list;
	GtkWidget *pix;
	GdkPixmap *inkbox_pixmap;
	GdkBitmap *inkbox_mask = NULL;
	int i;
	char pix_name[] = "pixmap?";

	/* 2004.03.08 for tooltips */
	GtkTooltipsData *tips;
	GtkWidget *event;
	char event_name[] = "eventbox?";

	while (node) {
		if (node->rest < 0 || node->rest > 100) {
			_status_str.inkbox_use_flag[0] = 0;
		}
		node = node->next;
	}
	
	node = list;
	if (_status_str.inkbox_use_flag[0] != 0 && node != NULL)
	{
		if (_status_str.inkbox_use_flag[_status_str.inkbox_counter] == 0
		    || _status_str.inkbox_counter >= INK_NUMBER)
			_status_str.inkbox_counter = 0;
		
		for (i = 0; i < _status_str.inkbox_counter; i++) {
			if (!node) {
				_status_str.inkbox_use_flag[0] = 0;
				return;
			}
			node = node->next;
		}
		
		pix_name[6] = '1' + _status_str.inkbox_counter;
		pix = lookup_widget (_status_str.window, pix_name);


		inkbox_chenge_volume (_status_str.inkbox_xpm, node->id, node->rest);
		inkbox_pixmap = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
							     &inkbox_mask, NULL,
							     _status_str.inkbox_xpm);
		gtk_pixmap_set(GTK_PIXMAP(pix),
			       inkbox_pixmap,
			       inkbox_mask);
		gtk_widget_show (pix);
		gdk_pixmap_unref (inkbox_pixmap);
		gdk_bitmap_unref (inkbox_mask);
		
		/* 2004.03.08 for tooltips */
		event_name[8] = '1' + _status_str.inkbox_counter;
		event = lookup_widget (_status_str.window, event_name);
		gtk_widget_show ( event );
		
		_status_str.inkbox_use_flag[_status_str.inkbox_counter] = 1;
		_status_str.inkbox_counter++;
	}
	else
	{
		for (i = 0; i < INK_NUMBER; i++)
		{
			pix_name[6] = '1' + i;
			pix = lookup_widget (_status_str.window, pix_name);

			/* 2004.03.08 for tooltips */
			event_name[8] = '1' + i;
			event = lookup_widget (_status_str.window, event_name);
			
			if (node != NULL && (node->rest >= 0 && node->rest <= 100))
			{

				inkbox_chenge_volume (_status_str.inkbox_xpm, node->id, node->rest);
				inkbox_pixmap = gdk_pixmap_create_from_xpm_d(_status_str.window->window,
									     &inkbox_mask, NULL,
									     _status_str.inkbox_xpm);
				gtk_pixmap_set(GTK_PIXMAP(pix),
					       inkbox_pixmap,
					       inkbox_mask);
				_status_str.inkbox_use_flag[i] = 1;
				gtk_widget_show (pix);
				gdk_pixmap_unref (inkbox_pixmap);
				gdk_bitmap_unref (inkbox_mask);

				/* 2004.03.08		    */
				/* set tooltips label again */
				tips = gtk_tooltips_data_get( event );
				if( tips ){
					gtk_tooltips_set_tip( tips->tooltips, tips->widget,
						       (char*)_(ink_name[ inkbox_get_inkid(node->id) ]), NULL );	
				}
				gtk_widget_show ( event );
				
				node = node->next;
			}
			else
			{
				_status_str.inkbox_use_flag[i] = 0;
				gtk_widget_hide ( event );
				gtk_widget_hide (pix);
			}
		}
	}
	return;
};

#else

static void
setInkProg(InkList list)
{
	InkNode *node = list;
	GtkWidget *widget;
	float color, black;

	if (node == 0) {
		color = black = 0.0;
	} else {
		color = (float)node->rest;
		if (color <= 0.0 || color > 100.0) {
			color = 0.0;
		} else {
			color = color / 100;
		}
		black = (float)node->next->rest;
		if (black <= 0.0 || black > 100.0) {
			black = 0.0;
		} else {
			black = black / 100;
		}
	}

	widget = lookup_widget (_status_str.window, "black_prog");
	gtk_progress_bar_update(GTK_PROGRESS_BAR(widget), (gfloat)color);
	widget = lookup_widget (_status_str.window, "color_prog");
	gtk_progress_bar_update(GTK_PROGRESS_BAR(widget), (gfloat)black);

	return;
}
#endif /* USE_INKBOX */

static void
ink_list_delete (InkList list)
{
	InkNode *node;

	while (list) {
		node = list->next;
		free(list);
		list = node;
	}
	return;
}
