/*
 * Photo Image Print System
 * Copyright (C) 2000-2004 EPSON KOWA Corporation.
 * Copyright (C) SEIKO EPSON CORPORATION 2000-2004.
 *
 *  This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 * As a special exception, EPSON KOWA Corporation gives permission to
 * link the code of this program with libraries which are covered by
 * the EPSON KOWA PUBLIC LICENCE and distribute their linked
 * combinations.  You must obey the GNU General Public License in all
 * respects for all of the code used other than the libraries which
 * are covered by EPSON KOWA PUBLIC LICENCE.
 */
#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h>
#include <sys/mman.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <ctype.h>
#include "gsconfig.h"


/* Prototype */
static BOOL
getRc (LP_GS_CONF);
static BOOL
make_gsc (LP_GS_CONF, int, int, char*);
static int
getRcError (FILE*);
static int
scanRc (FILE*);
static char*
existsp (char*);
static int
fileChecker (char*);


/* functions */
static BOOL
getRc (LP_GS_CONF lp_gsc)
{
  FILE *rcfp;
  char lin[BUFMAX];

  int resol = 0;
  int ink = 0;
  char msize[10];

  if ((rcfp = fopen (RSC_PATH, "rb")) == NULL)
    {
      fprintf (stderr, "%s : No such file\n", RSC_PATH);
      return 1;
    }
  
  switch (scanRc (rcfp))
    {
    case -1:
      return getRcError (rcfp);
    case 0:
      return 1;
    }
  
  while (fgets (lin, BUFMAX - 1, rcfp) != NULL)
    {
      int size, i;
      char str[BUFMAX];
      existsp (lin);

      switch (*lin)
	{
	case '\0':
	  break;

	case '[':
	  goto END_LOOP;

	default:
	  if(sscanf (lin, "%[^=]s", str) != 1)
	    return getRcError (rcfp);

	  size = strlen (str);

	  for (i = 0; i < Param_Num; i++)
	    {
	      if (!strcmp (str, Params[i]))
		{
		  size++;
		  if (sscanf (lin + size, "%s", str) != 1)
		    return getRcError (rcfp);
		  
		  switch (i)
		    {
		    case ID_INK:
		      if (strcmp (str, "MONO"))
			ink = 1; /* Color */
		      else
			ink = 2; /* Mono */
		      break;
		    case ID_RESOL: resol = atoi (str); break;
		    case ID_MSIZE: strcpy (msize, str); break;
		    }
		}
	    }
	}
    }

END_LOOP:
  fclose (rcfp);
  rcfp = NULL;

  if (make_gsc (lp_gsc, ink, resol, msize))
    return getRcError(rcfp);
  
  return 0;
}

static BOOL
make_gsc (LP_GS_CONF lp_gsc, int ink, int resol,
	  char* msize)
{
  int i;
  
  if (lp_gsc == NULL || ink == 0 || resol == 0 || msize == "")
    return 1;

  for (i = 0; Media_Size[i].name != ""; i++)
    {
      if (!strcmp (msize, Media_Size[i].name))
	break;
    }

  if (Media_Size[i].name == NULL)
    return 1;

  lp_gsc->format = ink;
  lp_gsc->resolution = resol;
  lp_gsc->media_size_x = (Media_Size[i].x * resol) / 360 ;
  lp_gsc->media_size_y = (Media_Size[i].y * resol) / 360 ;

  return 0;
}

static BOOL
getRcError (FILE *rcfp)
{
  if (rcfp != NULL) fclose (rcfp);
  fprintf (stderr, "pipsrc not read\n");
  return 1;
}


static int
scanRc (FILE *fp)
{
  int linNo, flag;
  char lin[BUFMAX], ep_name[BUFMAX];
  char printer[] = PRINTER_MODEL;
  flag = 0;
  for (linNo = 1; fgets (lin, BUFMAX - 1, fp) != NULL; linNo++)
    {
      existsp (lin);
      switch (*lin)
	{
	case '\0': break;

	case '[':
	  flag = 1;
	  if (sscanf (lin, "[%[^]]s", ep_name) == EOF)
	    return -1;

	  if (!strncmp (existsp (printer), ep_name, strlen (ep_name)))
	    return linNo;
	  break;

	default:
	  if (flag == 0)
	    return -1;
	}
    }
  return 0;
}


static char*
existsp (char *lin)
{
  char tmpLin[BUFMAX];
  int i, j;
  
  j = 0;
  
  for (i = 0; lin[i] != '\0'; i++)
    if (!isspace (lin[i]))
      tmpLin[j++] = lin[i];

  if (tmpLin[j - 1] != '\0')
    {
      tmpLin[j] = '\0';
    }
  strcpy (lin, tmpLin);
  return lin;
}


static int
fileChecker (char *buffer)
{
  if (!strncmp (buffer, "%!", 2))
    return 0;
  if (buffer[0] == 0x04 && !strncmp (buffer + 1, "%!", 2))
    return 0;

  return 1;
}


int
main (int argc, char **argv)
{
  GS_CONF gsc;
  char *buffer;
  char gs_path[] = GS_PATH;

  char gscom[BUFMAX];

  buffer = mmap (0, LBUFMAX, PROT_READ, MAP_PRIVATE, 0, 0);
  if (buffer == MAP_FAILED)
    return 1;

  if (fileChecker (buffer))
    {
      int nbuf;
      char tmpbuf[LBUFMAX];

      buffer = tmpbuf;
      munmap (buffer, LBUFMAX);
      while ((nbuf = fread (buffer, sizeof (char), LBUFMAX - 1, stdin)))
	{
	  fwrite (buffer, sizeof (char), nbuf, stdout);
	}
      return 0;
    }

  munmap (buffer,LBUFMAX);

  if (getRc (&gsc))
    {
      gsc.format = Def_Conf.format;
      gsc.resolution = Def_Conf.resolution;
      gsc.media_size_x = Def_Conf.media_size_x;
      gsc.media_size_y = Def_Conf.media_size_y;
    }

  if (gsc.format == 1)		/* Color */
    sprintf (gscom, "%s -r%d -g%ldx%ld -q -dNOPROMPT -dNOPAUSE -dSAFER -sDEVICE=png16m -sOutputFile=- - -c quit",
	     gs_path, gsc.resolution, gsc.media_size_x, gsc.media_size_y);

  else if (gsc.format == 2)		/* Mono */
    sprintf (gscom, "%s -r%d -g%ldx%ld -q -dNOPROMPT -dNOPAUSE -dSAFER -sDEVICE=pnggray -sOutputFile=- - -c quit",
	     gs_path, gsc.resolution, gsc.media_size_x, gsc.media_size_y);

  /* fprintf (stderr, "%s\n", gscom); */
  system (gscom);

  return 0;
}
